# -*- coding: utf-8 -*-
from collections import defaultdict


class EasyMapper(object):
    """
    Позволяет описать правила перевода слов и осуществлять перевод в обе стороны

    >>> em = EasyMapper([('a', 1), ('b', 2)])
    >>> em.map('a')
    1
    >>> em.map(1)
    1
    >>> em.unmap(1)
    'a'
    >>> list(em.map_iterable(['a', 'd', 'b']))
    [1, 'd', 2]
    """

    def __init__(self, aliases_mapping):
        """
        :param aliases_mapping: dict or list of tuples. Маппинг имен.
        """
        if isinstance(aliases_mapping, dict):
            self.aliases_mapping = aliases_mapping.items()
        elif isinstance(aliases_mapping, list):
            self.aliases_mapping = aliases_mapping
        else:
            raise ValueError('Bad aliases_mapping. Expect: dict or list of tuples. Got: %s' % type(aliases_mapping))

        self.a_b_map = defaultdict(list)
        self.b_a_map = defaultdict(list)

        for a_alias, b_alias in self.aliases_mapping:
            self.a_b_map[a_alias].append(b_alias)
            self.b_a_map[b_alias].append(a_alias)

    @staticmethod
    def _map(mapping, word):
        return mapping.get(word, [word])[0]

    @classmethod
    def _map_iterable(cls, mapping, iterable):
        for word in iterable:
            yield cls._map(mapping, word)

    @classmethod
    def _map_dict_keys(cls, mapping, dictionary):
        result = {}
        for key, value in dictionary.iteritems():
            new_key = cls._map(mapping, key)
            if new_key in result:
                raise KeyError('Attemp to override existing key')
            result[new_key] = value
        return result

    def map(self, word):
        """Перевод слова ->"""
        return self._map(self.a_b_map, word)

    def unmap(self, word):
        """Перевод слова <-"""
        return self._map(self.b_a_map, word)

    def map_iterable(self, iterable):
        """Перевод итерируемого объекта ->"""
        return self._map_iterable(self.a_b_map, iterable)

    def unmap_iterable(self, iterable):
        """Перевод итерируемого объекта <-"""
        return self._map_iterable(self.b_a_map, iterable)

    def map_dict_keys(self, dictionary):
        """Переименовывание ключей словаря ->"""
        return self._map_dict_keys(self.a_b_map, dictionary)

    def unmap_dict_keys(self, dictionary):
        """Переименовывание ключей словаря <-"""
        return self._map_dict_keys(self.b_a_map, dictionary)


class ListMapper(object):
    """
    Мапает переданный список объектов на исходный список ключей

    Если ключи, переданные в конструктор, повторяются, то на их место ставится один и тот же объект

    >>> lm = ListMapper([1, 2, 3, 4, 5])
    >>> print lm.map([3, 5, 2, 55])
    [None, 2, 3, None, 5]
    >>> print lm.map([2])
    [None, 2, None, None, None]

    >>> lm = ListMapper([1, 4])
    >>> print lm.map([{'key': 4, 'cool': 'very'}], key_getter=lambda item: item['key'])
    [None, {'key': 4, 'cool': 'very'}]
    """
    @staticmethod
    def default_key_getter(item):
        return item

    def __init__(self, keys):
        self._keys_len = len(keys)
        self._key_map = defaultdict(list)
        for i, key in enumerate(keys):
            self._key_map[key].append(i)

    def check_key_getter(self, key_getter):
        if key_getter is None:
            key_getter = self.default_key_getter
        if not hasattr(key_getter, '__call__'):
            raise TypeError("key_getter should be callable.")
        return key_getter

    def map(self, iterable, key_getter=None):
        """
        Отобразить список объектов на исходный список ключей

        Возвращает список такого же размера, как исходный список ключей
        Если не найдено подходящего элемента - вернет None на соответсующей позиции
        Если ключи повторяются, то будет выбран последний объект

        key_getter - функция получения ключа из объекта
        """
        result = [None] * self._keys_len
        key_getter = self.check_key_getter(key_getter)
        for item in iterable:
            key = key_getter(item)
            if key in self._key_map:
                for pos in self._key_map[key]:
                    result[pos] = item
        return result
