# -*- coding: utf-8 -*-
import re
import urllib
import urlparse

from mpfs.core.services.clck_service import clck


URL_RE = re.compile(r'^https?://')


def any_url_to_private_hash(url_or_hash, split_url=False):
    """
    Принимает публичный хэш или публичную ссылку, возвращает публичный хэш и путь

    :param url_or_hash: публичный ключ или любая публичная ссылка, короткая или длинная, с путём до ресурса или без
    :param split_url: отделять ли путь от хеша в результате
    :return: если split_url=True, то кортеж из публичного хеша и пути;
             путь может быть None, если отсутствует в url_or_hash
             если split_url=False, то строка вида "HASH" или "HASH:/path"

    В url_or_hash можно передать:

    хэш: RZof0sWwbPYDnrz1x+rJYT2pJP03SByhS1QkOZU3tI=
    хэш с путём: "RZof0sWwbPYDnrz1x+rJYT2pJP03SByhS1QkOZU3tI=:/path/to resource"

    длинная ссылка с урлэнкоженным хэшем:
        https://disk.yandex.ru/public/?hash=RZof0sWwbPYDnrz1x%2BrJYT2pJP03SByhS1QkOZU3tI%3D
        https://yadi.sk/public/?hash=RZof0sWwbPYDnrz1x%2BrJYT2pJP03SByhS1QkOZU3tI%3D
    длинная ссылка с урлэнкоженными хэшем и путём:
        https://disk.yandex.ru/public/?hash=RZof0sWwbPYDnrz1x%2BrJYT2pJP03SByhS1QkOZU3tI%3D%3A/path/to%20resource

    короткая ссылка:
        https://yadi.sk/i/1s7-C6gc3Tqx3d
    короткая ссылка с путём, где каждый элемент урлэкожен:
        https://yadi.sk/i/1s7-C6gc3Tqx3d/path/to%20resource
    """

    # сюда приходит значение параметра url-декодированное и декодированное из utf-8

    if URL_RE.match(url_or_hash):
        # пробуем распарсить длинную ссылку
        private_hash_with_path = clck.full_url_to_public_hash(url_or_hash)
        if private_hash_with_path:
            private_hash, _, path = private_hash_with_path.partition(':')
        else:
            # Это короткая ссылка. Убираем query-string и пытаемся выделить путь до ресурса внутри папки
            url = urlparse.urlsplit(url_or_hash)._replace(query='', fragment='')
            url_path_parts = url.path.split('/')
            if len(url_path_parts) > 3:
                url_path = '/'.join(url_path_parts[:3])
                path = '/' + '/'.join(url_path_parts[3:])
                path = urllib.unquote(path)
                url = url._replace(path=url_path)
            else:
                path = None
            url = url.geturl()
            private_hash = clck.short_url_to_public_hash(url)
    else:
        # это не ссылка, а просто хеш
        private_hash, _,  path = url_or_hash.partition(':')

    if split_url:
        return private_hash, path or None
    else:
        if path:
            private_hash += ':' + path
        return private_hash
