# -*- coding: utf-8 -*-
from time import time


class InMemoryRPSLimiterError(Exception):
    pass


class LimitExceed(InMemoryRPSLimiterError):
    pass


class Timeout(InMemoryRPSLimiterError):
    pass


class InMemoryRPSLimiter(object):
    """Лимитирование RPS в памяти"""

    def __init__(self, limit):
        """Создание нового limiter-а

        :param number limit: граница RPS-а
        """
        if limit < 1:
            raise ValueError('Limit cant be less then `1`')
        self._limit = limit
        self._current_limit = limit
        self._last_ts = time()

    def check(self, requests_num=1):
        """Исчерпан лимит или нет

        :param int requests_num: кол-во единовременных запросов
        :raises LimitExceed: лимит исчерпан
        :rtype NoneType:
        """
        now_ts = time()
        delta = now_ts - self._last_ts
        restored_limit = delta * self._limit
        new_limit = min(self._limit, self._current_limit + restored_limit)

        if new_limit < requests_num:
            raise LimitExceed()
        else:
            self._last_ts = now_ts
            self._current_limit = new_limit - requests_num

    def block_until_allowed(self, requests_num=1):
        """Дождаться момента, когда можно сделать запрос

        :param int requests_num: кол-во единовременных запросов
        :raises Timeout: таймаут
        :rtype NoneType:
        """
        while True:
            try:
                self.check(requests_num=requests_num)
            except LimitExceed:
                if time() - self._last_ts > 1:
                    raise Timeout()
            else:
                break
