# -*- coding: utf-8 -*-
import re
from enum import Enum
from collections import namedtuple

# helper-class as workaround for custom attributes with python-enums
DataUnit = namedtuple('DataUnit', ['suffix', 'size'])


class DataSize(object):
    PATTERN = re.compile(r'^([+-]?\d+)([a-zA-Z]{0,2})$')

    BYTES_PER_KB = 1024
    BYTES_PER_MB = BYTES_PER_KB * BYTES_PER_KB
    BYTES_PER_GB = BYTES_PER_MB * BYTES_PER_KB
    BYTES_PER_TB = BYTES_PER_GB * BYTES_PER_KB

    def __init__(self, size):
        self.size = size

    @staticmethod
    def parse(value):
        unit = DataUnits.BYTES
        try:
            match = DataSize.PATTERN.match(str(value))

            matched_suffix = match.group(2)
            if matched_suffix:
                unit = DataUnits.from_suffix(matched_suffix)

            matched_value = match.group(1)
            amount = long(matched_value)
        except Exception:
            raise ValueError(str(value) + " is not valid data size")

        return DataSize.of(amount, unit)

    @staticmethod
    def of(amount, unit):
        return DataSize(amount * unit.size())

    @staticmethod
    def of_kilobytes(kilobytes):
        return DataSize(kilobytes * DataSize.BYTES_PER_KB)

    @staticmethod
    def of_megabytes(megabytes):
        return DataSize(megabytes * DataSize.BYTES_PER_MB)

    @staticmethod
    def of_gigabytes(gigabytes):
        return DataSize(gigabytes * DataSize.BYTES_PER_GB)

    @staticmethod
    def of_terabytes(terabytes):
        return DataSize(terabytes * DataSize.BYTES_PER_TB)

    def to_bytes(self):
        return self.size

    def __lt__(self, another):
        return self.to_bytes() < another.to_bytes()


class DataUnits(Enum):
    BYTES = DataUnit("B", 1)
    KILOBYTES = DataUnit("KB", DataSize.of_kilobytes(1).to_bytes())
    MEGABYTES = DataUnit("MB", DataSize.of_megabytes(1).to_bytes())
    GIGABYTES = DataUnit("GB", DataSize.of_gigabytes(1).to_bytes())
    TERABYTES = DataUnit("TB", DataSize.of_terabytes(1).to_bytes())

    @staticmethod
    def from_suffix(suffix):
        for candidate in DataUnits:
            if candidate.suffix() == suffix:
                return candidate
        raise ValueError("Unknown data unit suffix: " % suffix)

    def size(self):
        return self.value.size

    def suffix(self):
        return self.value.suffix
