# -*- coding: utf-8 -*-
import types
import collections


def normalize_offset(offset):
    """Нормализация сдвига"""
    if offset is None:
        return 0
    return int(offset)


def normalize_limit(limit):
    """Нормализация размера выборки"""
    if limit is None:
        return None
    return int(limit)


def validate_slice(slice_obj, allow_negative_indices=True, allow_step=True):
    if not isinstance(slice_obj, slice):
        raise TypeError("indices must be slice, not %s" % type(slice_obj))
    for index in (slice_obj.start, slice_obj.stop, slice_obj.step):
        if not isinstance(index, (int, long, types.NoneType)):
            raise TypeError("slice indices must be integers or None")
        if (not allow_negative_indices and
                isinstance(index, (int, long)) and
                index < 0):
            raise ValueError("Negative indices not allowed")
    if not allow_step and slice_obj.step not in (None, 1):
        raise ValueError("Steps not allowed")


def slice_to_offset_limit(slice_obj):
    """Приведение slice к offset и limit"""
    validate_slice(slice_obj, allow_negative_indices=False, allow_step=False)

    offset = normalize_offset(slice_obj.start)
    if slice_obj.stop is None:
        return offset, None
    elif offset > slice_obj.stop:
        return offset, 0
    else:
        limit = normalize_limit(slice_obj.stop - offset)
        return offset, limit


def offset_limit_to_slice(offset, limit):
    """Приведение offset и limit к slice"""
    offset = normalize_offset(offset)
    limit = normalize_limit(limit)
    if offset < 0 or limit is not None and limit < 0:
        raise ValueError("Offset: %s Limit: %s" % (offset, limit))
    limit = limit + offset if limit is not None else None
    return slice(offset, limit)


class SequenceChain(collections.Sequence):
    """Позволяет представить несколько рядов как один и прозрачно получать элементы по индексу/срезу

    В отличиe от `(list_1 + ... + list_N)[slice_obj]`
    не создает единый список со всеми элементами, а достает нужные данные из отдельных рядов

    В отличие от `itertools.chain` + `itertools.islice`
    не требует итерации по первым элементам

    Особенно это актуально, когда sequences - это данные не в памяти, а в БД.
    В случае использования SequenceChain мы запрашиваем только нужные данные.

    >>> chained_sequences = SequenceChain(range(3), range(3))
    >>> chained_sequences[:]
    [0, 1, 2, 0, 1, 2]
    >>> chained_sequences[1:]
    [1, 2, 0, 1, 2]
    >>> chained_sequences[2:4]
    [2, 0]
    """
    def __init__(self, *sequences):
        for seq in sequences:
            if not isinstance(seq, collections.Sequence):
                raise TypeError("%r is not sequence" % seq)
        self._sequences = sequences

    def __len__(self):
        return sum(len(s) for s in self._sequences)

    def __getitem__(self, key):
        if isinstance(key, (int, long)):
            key = slice(key, key + 1)
            return list(self._generate_items(key))[0]
        return list(self._generate_items(key))

    def __repr__(self):
        return "%s%r" % (self.__class__.__name__, tuple(self._sequences))

    def _generate_items(self, slice_obj):
        validate_slice(slice_obj, allow_negative_indices=False, allow_step=False)

        start = slice_obj.start or 0
        stop = slice_obj.stop
        if stop is not None and stop - start <= 0:
            raise StopIteration()

        for sequence in self._sequences:
            if start > 0:
                sequence_len = len(sequence)
                if sequence_len <= start:
                    # текущий ряд не попадает в выборку
                    # уменьшаем сдвиг на длину ряда и переходим к следующему ряду
                    start -= sequence_len
                    if stop is not None:
                        stop -= sequence_len
                    continue

            for item in sequence[slice(start, stop)]:
                yield item
                if stop is not None:
                    stop -= start
                    stop -= 1
                    if stop <= 0:
                        raise StopIteration()
                # т.к. мы начали отдавать элементы, то для всех следующих
                # последовательностей сдвиг будет равен 0
                start = 0


class AdhocSequence(collections.Sequence):
    """Класс для конструирования объекта "похожего" на ряд(sequence)"""

    def __init__(self, getitem_func, len_func):
        """Конструктор

        :param getitem_func: функция, принимающая два kwarg-а(offset, limit)
                             и отдающая элементы по переданному срезу
        :param len_func: функция без параметров для получения длины ряда
        """
        self.getitem_func = getitem_func
        self.len_func = len_func

    def __getitem__(self, key):
        offset, limit = slice_to_offset_limit(key)
        # Требуется, чтобы метод принимал offset и limit как kwarg-и.
        # При желании можно порефакторить на более гибкий подход
        return self.getitem_func(offset=offset, limit=limit)

    def __len__(self):
        return self.len_func()
