# -*- coding: utf-8 -*-
"""

MPFS
UTIL

Процессор шаблонов

"""
import os

from datetime import datetime
from urllib import quote_plus, quote
from jinja2 import Template, FileSystemLoader, Environment
from jinja2.exceptions import TemplateNotFound

import mpfs.engine.process
from itertools import imap
from mpfs.config import settings

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def urlencode(value, plus=False):
    '''
    Кастомный фильтр
    https://jira.yandex-team.ru/browse/CHEMODAN-5562
    '''
    return quote_plus(value, safe="") if plus else quote(value, safe="")


def formatsize(value, string='[ bytes | KB | MB | GB | TB | PB]', locale='en', withUnit=True):
    for u in imap(lambda x: x.strip(), string.strip('[]').split('|')):
        if value < 1024.0:
            size_pattern = ("%g"+ '\xC2\xA0' +"%s").decode('utf-8')
            result = size_pattern % (round(value, 1), u)
            if locale in ['ru','uk','tr']:
                result = result.replace('.',',')
            return result
        value /= 1024.0
    suf = map(lambda x: x.strip(), string.strip('[]').split('|'))[-1]
    value *= 1024.0
    result = "%g&nbsp;%s" % (value, suf) if withUnit else "%g" % value
    return result


def strftime(value):
    result = {}
    x = datetime.fromtimestamp(value)
    for item in ('month', 'day', 'hour', 'minute', 'second'):
        result[item] = '%02d' % getattr(x, item)
    result['year'] = x.year
    return result
    

class TemplateProcessor(object):
    def __init__(self, template, args):
        self.env = Environment()
        self.env.filters['urlencode'] = urlencode
        self.env.filters['formatsize'] = formatsize
        self.env.filters['strftime'] = strftime
        self.env.loader = FileSystemLoader(settings.system['fs_paths']['template_dir'])
        self.template = template
        self.args = args
        
    def process(self, part):
        filepath = self.template + os.path.sep + part
        self.locale = self.args.get('locale', '') or ''
        
        try:
            obj = self.env.get_template(filepath + '.' + self.locale)
        except TemplateNotFound:
            self.locale = settings.user['default_locale']
            obj = self.env.get_template(filepath + '.' + self.locale)
            self.args['locale'] = self.locale
            
        return obj.render(**self.args).encode('utf-8')
