# -*- coding: utf-8 -*-
import re

from datetime import datetime


# From django.utils.translation.trans_real.parse_accept_lang_header
accept_language_re = re.compile(r'''
    ([A-Za-z]{1,8}(?:-[A-Za-z]{1,8})*|\*)       # "en", "en-au", "x-y-z", "*"
    (?:\s*;\s*q=(0(?:\.\d{,3})?|1(?:.0{,3})?))? # Optional "q=1.00", "q=0.8"
    (?:\s*,\s*|$)                               # Multiple accepts per header.
    ''', re.VERBOSE)


localized_months = {
    'ru': {
        1: u'января',
        2: u'февраля',
        3: u'марта',
        4: u'апреля',
        5: u'мая',
        6: u'июня',
        7: u'июля',
        8: u'августа',
        9: u'сентября',
        10: u'октября',
        11: u'ноября',
        12: u'декабря',
    },
    'en': {
        1: u'January',
        2: u'February',
        3: u'March',
        4: u'April',
        5: u'May',
        6: u'June',
        7: u'July',
        8: u'August',
        9: u'September',
        10: u'October',
        11: u'November',
        12: u'December',
    },
    'tr': {
        1: u'Ocak',
        2: u'Şubat',
        3: u'Mart',
        4: u'Nisan',
        5: u'Mayıs',
        6: u'Haziran',
        7: u'Temmuz',
        8: u'Ağustos',
        9: u'Eylül',
        10: u'Ekim',
        11: u'Kasım',
        12: u'Aralık',
    },
    'ua': {
        1: u'січня',
        2: u'лютого',
        3: u'березня',
        4: u'квітня',
        5: u'травня',
        6: u'червня',
        7: u'липня',
        8: u'серпня',
        9: u'вересня',
        10: u'жовтня',
        11: u'листопада',
        12: u'грудня',
    },
    'uk': {
        1: u'січня',
        2: u'лютого',
        3: u'березня',
        4: u'квітня',
        5: u'травня',
        6: u'червня',
        7: u'липня',
        8: u'серпня',
        9: u'вересня',
        10: u'жовтня',
        11: u'листопада',
        12: u'грудня',
    },
}


def parse_accept_lang_header(lang_string):
    """
    Parses the lang_string, which is the body of an HTTP Accept-Language
    header, and returns a list of (lang, q-value), ordered by 'q' values.

    Any format errors in lang_string results in an empty list being returned.
    """
    result = []
    pieces = accept_language_re.split(lang_string)
    if pieces[-1]:
        return []
    for i in range(0, len(pieces) - 1, 3):
        first, lang, priority = pieces[i : i + 3]
        if first:
            return []
        priority = priority and float(priority) or 1.0
        result.append((lang, priority))
    result.sort(key=lambda k: k[1], reverse=True)
    return result


lang_territory_splitter_re = re.compile(r'[-_]')


def locale_to_lang(locale):
    """
    Возвращает код языка из локали. Другими словами, превращает ru-RU в ru, en_US в en и т.п.
    :param locale:
    """
    result = lang_territory_splitter_re.split(locale)
    if len(result) >= 1:
        return result[0].lower()
    return None


def unixtime_to_localized_date_string(unixtime, locale):
    """
    Для поддерживаемых в Диске локалей возвращается "<число> <локализованный месяц> <год>".
    Для остальных возвращается "<число двумя цифрами>.<месяц двумя цифрами>.<год>".
    
    https://st.yandex-team.ru/CHEMODAN-48640
    """
    d = datetime.fromtimestamp(unixtime)
    if locale in localized_months:
        return u'%d %s %d' % (d.day, localized_months[locale][d.month], d.year)
    return u'%.2d.%.2d.%d' % (d.day, d.month, d.year)
