# -*- coding: utf-8 -*-
import re

from mpfs.common.util import from_json
from mpfs.common.util.mobile_client import MobileClientVersion
from mpfs.config import settings


USER_AGENT_YANDEX_SEARCH_MOBILE_REGEX = settings.user_agent['yandex_search_mobile_regex']
USER_AGENT_YANDEX_MAIL_MOBILE_REGEX = settings.user_agent['yandex_mail_mobile_regex']
USER_AGENT_YANDEX_NOTES_REGEX = settings.user_agent['yandex_notes_regex']
USER_AGENT_LENTA_WORKER_REGEX = settings.user_agent['lenta_worker_regex']


class UserAgentParser(object):
    _yandex_search_mobile_user_agent_regex = re.compile(USER_AGENT_YANDEX_SEARCH_MOBILE_REGEX)
    _yandex_mail_mobile_user_agent_regex = re.compile(USER_AGENT_YANDEX_MAIL_MOBILE_REGEX)
    _yandex_notes_user_agent_regex = re.compile(USER_AGENT_YANDEX_NOTES_REGEX)
    _lenta_worker_regex = re.compile(USER_AGENT_LENTA_WORKER_REGEX)

    @staticmethod
    def parse(raw_user_agent, user_agent_prefix='Yandex.Disk'):
        if not raw_user_agent:
            return UserAgent()

        if not raw_user_agent.startswith(user_agent_prefix):
            return UserAgent()

        try:
            raw_json = raw_user_agent[len(user_agent_prefix):]
            parsed = from_json(raw_json)
        except Exception:
            parsed = {}
        return UserAgent(parsed.get('os'), parsed.get('id'), parsed.get('vsn'), parsed.get('device'),
                         parsed.get('session_id'), parsed.get('assessor', False))

    @staticmethod
    def get_os(user_agent):
        user_agent_data = UserAgentParser.parse(user_agent)
        return user_agent_data.os

    @staticmethod
    def get_id(user_agent):
        user_agent_data = UserAgentParser.parse(user_agent)
        return user_agent_data.id

    @classmethod
    def get_vsn(cls, user_agent):
        user_agent_data = cls.parse(user_agent)
        return user_agent_data.vsn

    @staticmethod
    def is_yandex_disk_mobile(user_agent):
        """Возвращает True для мобильных клиентов Яндекс.Диска"""
        # User-Agent: Yandex.Disk {"os":"iOS","src":"disk.mobile","vsn":"2.14.7215","id":"E9C69BDA-0837-4867-A9B0-3AFCAAC3342A","device":"tablet"}
        # User-Agent: Yandex.Disk {"os":"android 6.0.1","device":"phone","src":"disk.mobile","vsn":"3.04-20002","id":"f412ce07323db938d57a19aba1dbe1d1"}
        return UserAgentParser.parse(user_agent).is_mobile_client()

    @staticmethod
    def is_yandex_search_mobile(user_agent):
        if not user_agent:
            return False
        return bool(UserAgentParser._yandex_search_mobile_user_agent_regex.match(user_agent))

    @staticmethod
    def is_yandex_mail_mobile(user_agent):
        if not user_agent:
            return False
        return bool(UserAgentParser._yandex_mail_mobile_user_agent_regex.match(user_agent))

    @staticmethod
    def is_yandex_notes(user_agent):
        if not user_agent:
            return False
        return bool(UserAgentParser._yandex_notes_user_agent_regex.match(user_agent))

    @staticmethod
    def is_lenta_worker(user_agent):
        if not user_agent:
            return False
        return bool(UserAgentParser._lenta_worker_regex.match(user_agent))

    @staticmethod
    def get_unique_id(user_agent):
        """Возвращает какой-нибудь уникальный идентификатор из User-Agent.

        Умеет доставать device id для запросов мобильных клиентов.
        """
        return UserAgentParser.parse(user_agent).id

    @staticmethod
    def get_session_id(user_agent):
        return UserAgentParser.parse(user_agent).session_id


class UserAgent(object):
    IOS = 'iOS'
    ANDROID = 'android'

    MAC = 'mac'
    WIN = 'windows'
    DESKTOP = {MAC, WIN}

    os = None
    id = None
    vsn = None

    def __init__(self, os=None, id=None, vsn=None, device_type=None, session_id=None, is_assessor=False):
        self.os = os
        self.id = id
        self.vsn = vsn
        self.device_type = device_type
        self.session_id = session_id
        self.is_assessor = is_assessor

    def is_desktop_client(self):
        return bool(self.os and self.os in self.DESKTOP)

    def is_mac(self):
        return bool(self.os and self.os == self.MAC)

    def is_win(self):
        return bool(self.os and self.os == self.WIN)

    def is_mobile_client(self):
        return bool(self.os and (self.os.startswith(self.IOS) or self.os.startswith(self.ANDROID)))

    def is_ios(self):
        return bool(self.os and (self.os.startswith(self.IOS)))

    def is_android(self):
        return bool(self.os and (self.os.startswith(self.ANDROID)))

    def get_version(self):
        if not self.vsn:
            return None
        if self.is_mobile_client():
            try:
                return MobileClientVersion.build_from_version(self.vsn)
            except Exception:
                return None
        return None

    def get_os_family(self):
        if self.os is None:
            raise ValueError('No OS specified')
        if self.os.startswith(self.IOS):
            return self.IOS
        elif self.os.startswith(self.ANDROID):
            return self.ANDROID
        elif self.is_win():
            return self.WIN
        elif self.is_mac():
            return self.MAC
        else:
            raise ValueError('Unknown OS')
