# -*- coding: utf-8 -*-
"""
Содержит логику построения конфигов
"""
import os
import sys

from mpfs.common.util import unstraighten_dictionary
from mpfs.config.constants import (GLOBAL_SETTINGS_PATH, ADMINS_OVERRIDES_PATH,
                                   ENV_OVERRIDES_PATH_TEMPLATE, ENV_PKG_OVERRIDES_PATH_TEMPLATE,
                                   ALLOWED_PACKAGES, SECRET_KEYS_SETTINGS_PATH, ACCESS_OVERRIDES_PATH,
                                   MPFS_ENVIRONMENT_PREFIX)
from mpfs.config.base import Config
from mpfs.config.environment import LowerStringEnvironmentVariable
from mpfs.config.errors import (NoSourceConfigError, ConfigError,
                                ValidationConfigError, BadFormatConfigError)
from mpfs.config.provider import MPFSWorkingEnvironmentConfigProvider, YAMLConfigProvider

# Соответствие пути до опции в файле с ключами /etc/yandex/disk-secret-keys.yaml путю в настройках global_settings.yaml
# Например, если в файле с ключами secret key лежит так:
#   zaberun:
#     url: asdf
# А в настройках так:
#   services:
#     zaberun:
#       secret_key: 123
# То надо прописать в SECRET_KEYS_MAPPER строку вида: 'zaberun url': 'services zaberun secret_key'
SECRET_KEYS_MAPPER = {
    'zaberun url': 'services zaberun secret_key',
    'zaberun stid_aes': 'services zaberun stid_secret',
    'tvm key': 'services tvm secret_key',
    'tvm_2_0 key': 'services tvm_2_0 secret_key',
    'credential_crypter key': 'credential_crypter secret',
    'docviewer mds_key': 'platform docviewer mds_key_secret',
    'docviewer sign': 'platform docviewer sign_secret',
    'mrstat yt_token': 'mrstat yt_token',
    'mrstat stat_password': 'mrstat publish_user StatRobotPassword',
    'api_admin zookeeper password': 'zookeeper password'
}

ENVIRONMENT_KEYS_MAPPER = {
    'QLOUD_DATACENTER': LowerStringEnvironmentVariable('queue2 host_datacenter'),
}


def _patch_overrides_by_path(settings_patch, settings_key_path, value):
    settings_key_path_parts = settings_key_path.split(' ')
    patch = settings_patch
    for part in settings_key_path_parts[:-1]:
        if part not in patch:
            patch[part] = dict()
        patch = patch[part]

    value_name = settings_key_path_parts[-1]
    patch[value_name] = value
    return settings_patch


def _get_secret_keys_overrides_patch(secret_keys):
    settings_patch = {}

    for secret_key_path, settings_key_path in SECRET_KEYS_MAPPER.iteritems():
        secret_key_path_parts = secret_key_path.split(' ')

        value = secret_keys
        is_key_found = True
        for part in secret_key_path_parts:
            if part in value:
                value = value[part]
            else:
                is_key_found = False
                break

        if is_key_found:
            settings_patch = _patch_overrides_by_path(settings_patch, settings_key_path, value)

    return settings_patch


def _get_environment_keys_overrides_patch():
    settings_patch = {}

    for environment_key, environment_obj in ENVIRONMENT_KEYS_MAPPER.iteritems():
        if environment_key in os.environ:
            settings_patch = _patch_overrides_by_path(
                settings_patch,
                environment_obj.settings,
                environment_obj.cast(os.environ[environment_key])
            )

    return settings_patch


def mpfs_config_builder():
    """
    Дефолтная логика сборки конфига

    https://wiki.yandex-team.ru/users/waser/newmpfsconfigs#porjadokiosobennostiobrabotki
    """
    config = Config()
    environment = MPFSWorkingEnvironmentConfigProvider().load_as_dict()

    # установка глобального конфига
    global_config = YAMLConfigProvider(GLOBAL_SETTINGS_PATH).load_as_dict()
    config.update(global_config)

    # установка переопределений
    for overrides_path_template in (ENV_OVERRIDES_PATH_TEMPLATE, ENV_PKG_OVERRIDES_PATH_TEMPLATE):
        overrides_file = overrides_path_template % environment
        try:
            overrides_patch = YAMLConfigProvider(overrides_file).load_as_dict()
        except NoSourceConfigError as e:
            # переопределяющего конфига может не быть
            print >> sys.stderr, e
        else:
            try:
                config.patch(overrides_patch)
            except ValidationConfigError as e:
                e.config_path = overrides_file
                raise

    # оверрайдим ключи - обязательно со allow_new=False, чтобы не запускаться, если в global_settings местоположение
    # ключей поменялось, а в файле из salt нет
    patch_config(config, ACCESS_OVERRIDES_PATH, False)

    # установка админских переопределений
    patch_config(config, ADMINS_OVERRIDES_PATH, True)

    # переопределение секретных ключей для заберуна, если файл с секретными ключами найден
    patch_config(config, SECRET_KEYS_SETTINGS_PATH, False, _get_secret_keys_overrides_patch)

    # установка переопределений из environment
    patch_config_from_env(config, False, _get_environment_keys_overrides_patch)

    # переопределение через переменные окружения, начинающиеся с mpfs.config.constants.MPFS_ENVIRONMENT_PREFIX
    patch_from_straighten_environment(config, True)

    return config


def patch_config(config, file_path, allow_new, format_func=None):
    try:
        patch = YAMLConfigProvider(file_path).load_as_dict()
    except (NoSourceConfigError, BadFormatConfigError) as e:
        print >> sys.stderr, e
    else:
        try:
            if format_func:
                patch = format_func(patch)
            config.patch(patch, allow_new=allow_new)
        except ValidationConfigError as e:
            e.config_path = file_path
            raise


def patch_config_from_env(config, allow_new, format_func):
    try:
        patch = format_func()
        config.patch(patch, allow_new=allow_new)
    except ValidationConfigError as e:
        e.config_path = 'environment'
        raise


def patch_from_straighten_environment(config, allow_new):
    straighten_config = config.get_straighten_paths(prefix=MPFS_ENVIRONMENT_PREFIX)
    env_vars = {k.lower(): v for k, v in os.environ.iteritems() if k.startswith(MPFS_ENVIRONMENT_PREFIX)}
    for env_name, env_value in env_vars.iteritems():
        if env_name not in straighten_config:
            print >> sys.stderr, "Skip env `%s`: not in config" % env_name
            continue

        settings_path, value_type = straighten_config[env_name].rsplit(' ', 1)
        type_cast = {'str': str, 'int': int, 'float': float}
        try:
            env_value = type_cast[value_type](env_value)
        except (ValueError, KeyError) as e:
            print >> sys.stderr, e
            continue

        env_patch = unstraighten_dictionary({settings_path: env_value}, delimiter=' ')
        try:
            config.patch(env_patch, allow_new=allow_new)
        except ValidationConfigError as e:
            e.config_path = "straighten_environment"
            raise
