# -*- coding: utf-8 -*-
"""
Поставщики настроек

Если нужно получать откуда либо настройки, то создаем новый класс,
который умеет получать их и приводить к dict-у.
"""
import os
import sys
import yaml

from mpfs.config.constants import (MPFS_ENVIRONMENT_VAR, MPFS_PACKAGE_VAR,
                                   ENVIRONMENT_FILE_PATH, ALLOWED_PACKAGES,
                                   ALLOWED_ENVIRONMENTS, PACKAGE_FILE_PATH)
from mpfs.config.errors import (ProviderConfigError, NoSourceConfigError,
                                BadFormatConfigError)
from mpfs.common.util import unstraighten_dictionary


class ConfigProvider(object):
    """Базовый класс поставщиков конфигов"""
    def load(self):
        return self

    def as_dict(self):
        if hasattr(self, '_data'):
            return self._data
        raise NotImplementedError()

    def load_as_dict(self):
        return self.load().as_dict()


class YAMLConfigProvider(ConfigProvider):
    """Загрузка конфига из YAML файла"""
    def __init__(self, config_path):
        self.config_path = config_path
        self._data = {}

    def load(self):
        try:
            with open(self.config_path, 'r') as fh:
                self._data = yaml.load(fh)
            if not isinstance(self._data, dict):
                raise BadFormatConfigError('Possibly conf file empty or bad format. Got "NoneType". Expected: "dict". Check "%s".' % self.config_path)
        except IOError as e:
            raise NoSourceConfigError, e, sys.exc_info()[2]
        except (yaml.parser.ParserError, yaml.scanner.ScannerError, TypeError) as e:
            raise ProviderConfigError, e, sys.exc_info()[2]
        return super(YAMLConfigProvider, self).load()


class EnvironmentConfigProvider(ConfigProvider):
    """Загрузка конфига из переменных окружения"""
    def __init__(self, prefix_filter=None, delimiter=None):
        self.prefix_filter = prefix_filter
        self.delimiter = delimiter
        self._data = {}

    def load(self):
        env_vars = os.environ.copy()
        if self.prefix_filter is not None:
            env_vars = {k[len(self.prefix_filter):].lower(): v for k, v in env_vars.iteritems() if k.startswith(self.prefix_filter)}

        if self.delimiter is not None:
            env_vars = unstraighten_dictionary(env_vars, delimiter=self.delimiter)
        self._data = env_vars
        return super(EnvironmentConfigProvider, self).load()


class MPFSWorkingEnvironmentConfigProvider(ConfigProvider):
    """
    Загрузка MPFS окружения

    Формируется словарь вида:
    {
        'ENVIRONMENT': 'development',
        'PACKAGE': 'disk'
    }
    """
    QA_FILE = '/etc/yandex/environment.qa'
    """https://st.yandex-team.ru/CHEMODAN-12254"""
    load_params = (
        ('PACKAGE', MPFS_PACKAGE_VAR, PACKAGE_FILE_PATH, ALLOWED_PACKAGES),
        ('ENVIRONMENT', MPFS_ENVIRONMENT_VAR, ENVIRONMENT_FILE_PATH, ALLOWED_ENVIRONMENTS),
    )

    def __init__(self):
        self._data = {}

    @staticmethod
    def load_from_env(env_var_name, allowed_values):
        env_var_value = os.environ.get(env_var_name)
        if env_var_value not in allowed_values:
            raise ProviderConfigError('"%s" required. Got: "%s". Allowed values: %s' % (env_var_name, env_var_value, allowed_values))
        return env_var_value

    @staticmethod
    def load_from_file(file_path, allowed_values):
        try:
            with open(file_path, 'r') as fh:
                file_text = fh.read()
        except IOError as e:
            raise ProviderConfigError, e, sys.exc_info()[2]
        file_text = file_text.strip("\n")
        if file_text not in allowed_values:
            raise ProviderConfigError('"%s" required. Got: "%s". Allowed values: %s' % (file_path, file_text, allowed_values))
        return file_text

    def load(self):
        for name, env_var, file_path, allowed_values in self.load_params:
            try:
                self._data[name] = self.load_from_env(env_var, allowed_values)
            except ProviderConfigError:
                try:
                    self._data[name] = self.load_from_file(file_path, allowed_values)
                except ProviderConfigError:
                    # Может это QA
                    if name == 'ENVIRONMENT' and os.path.isfile(self.QA_FILE):
                        self._data['ENVIRONMENT'] = 'prestable'
                    else:
                        raise
        return super(MPFSWorkingEnvironmentConfigProvider, self).load()
