# -*- coding: utf-8 -*-
"""

MPFS
CORE

Класс адреса ресурса

"""
import base64
import datetime
import hashlib
import re
import os
import urllib

from os.path import sep
from copy import deepcopy

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common import errors
from mpfs.common.util import natsort, generete_name_with_suffix, public_urls

error_log = mpfs.engine.process.get_error_log()


'''
Правила именования путей:

Здесь и далее адрес ресурса обозначает сочетание UID владельца и пути к ресурсу.
Путь или id к [каталогу|файлу|ресурсу] обозначает полный путь  в системе, ВМЕСТЕ с именем.

Правила формирования адреса ресурса:
- адрес имеет формат типа UID:PATH
- PATH всегда начинается с /
- PATH каталога не всегда завершается /
- PATH не должен знать о существовании отдельных сервисов, этим занимается mpfs.core.factory

Правила именования симлинка:
- симлинк имеет формат типа UID:INT
- INT это integer

'''

CLN = ':'


class ResourceId(object):
    SEP = CLN
    FILE_ID_LEN = 64

    def __init__(self, uid, file_id):
        if not (uid and file_id):
            raise errors.AddressError('`uid` and `file_id` must be specified. Got uid: "%s", file_id: "%s"' % (uid, file_id))
        if len(file_id) != self.FILE_ID_LEN:
            raise ValueError('"file_id" part of resource_id must be %s characters long' % self.FILE_ID_LEN)
        self.uid = uid
        self.file_id = file_id

    @classmethod
    def parse(cls, raw_resource_id):
        """
        Создать объект ResourceId из строкового представления

        :param str raw_resource_id: Строка вида <uid>:<file_id>.
        """
        uid, file_id = raw_resource_id.split(cls.SEP, 1)
        return cls(uid, file_id)

    def serialize(self):
        """Вернуть строковое представление :class:`~ResourceId`.

        :rtype: str | unicode
        """
        return '%s%s%s' % (self.uid, self.SEP, self.file_id)

    def __str__(self):
        return self.serialize()

    def __repr__(self):
        return '%s(uid=%r, file_id=%r)' % (self.__class__.__name__, self.uid, self.file_id)

    def __ne__(self, other):
        return not self == other

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return self.uid == other.uid and self.file_id == other.file_id

    def copy(self):
        return self.__class__(self.uid, self.file_id)


class SharingURLAddress(object):
    """Класс для работы с адресацией офисных файлов по хэшу.

    Адресация офисных файлов имеет вид:
    https://disk.yandex.com/edit/d/<editor_doc_id>

    editor_doc_id состоит из:
      * зашифрованного uid'а владельца
      * office_doc_short_id (ключ, по которому у владельца можно найти файл в базе)

    Для работы с внутренним содержимым такой адресации (например, uid'ом) используется класс:
    :class:`mpfs.core.office.util.SharingURLAddressHelper`
    """
    SEP = ':'

    def __init__(self, encrypted_uid, office_doc_short_id):
        if not (encrypted_uid and office_doc_short_id):
            raise errors.AddressError('`encrypted_uid` and `office_doc_short_id` must be specified. '
                                      'Got encrypted_uid: "%s", office_doc_short_id: "%s"' % (encrypted_uid,
                                                                                              office_doc_short_id))
        self.encrypted_uid = str(urllib.unquote(encrypted_uid))
        self.office_doc_short_id = str(urllib.unquote(office_doc_short_id))

    @classmethod
    def parse(cls, office_online_sharing_url):
        """
        Создать объект SharingURLAddress из строкового представления

        :param str office_online_sharing_url
        """

        try:
            # добавляем = знаки, если их не хватает
            unquoted = urllib.unquote(str(office_online_sharing_url))
            sharing_url = unquoted + "=" * (-len(unquoted) % 4)
            decoded = base64.urlsafe_b64decode(sharing_url)
            encrypted_uid, office_doc_short_id = decoded.rsplit(cls.SEP, 1)
        except Exception as exc:
            raise errors.AddressError('Failed to parse office_online_sharing_url: %s' % exc.message)

        return cls(encrypted_uid=encrypted_uid,
                   office_doc_short_id=office_doc_short_id)

    def serialize(self):
        return base64.urlsafe_b64encode('%s%s%s' % (self.encrypted_uid, self.SEP, self.office_doc_short_id)).rstrip('=')

    def __str__(self):
        return self.serialize()

    def __repr__(self):
        return '%s(encrypted_uid=%r, office_doc_short_id=%r)' % (self.__class__.__name__,
                                                                 self.encrypted_uid,
                                                                 self.office_doc_short_id)

    def __ne__(self, other):
        return not self == other

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return self.encrypted_uid == other.encrypted_uid and self.office_doc_short_id == other.office_doc_short_id

    def copy(self):
        return self.__class__(self.encrypted_uid, self.office_doc_short_id)


class LentaResourceId(ResourceId):
    """
    Костыль для ленты, позволяющий использовать смешанную адресацию: uid + path или uid + file_id

    На момент написания не у всех MPFS ресурсов был file_id, в частности не было у корневых папок.
    Делалось для нужд ленты.
    """
    def __init__(self, uid, file_id_or_path):
        try:
            super(LentaResourceId, self).__init__(uid, file_id_or_path)
            self._common_id_fabric = ResourceId
        except ValueError:
            self.uid = uid
            self.file_id = file_id_or_path
            self._common_id_fabric = Address.Make

    def make_common_id(self):
        """
        В зависимости от вида возвращает объект ResourceId или Address
        """
        return self._common_id_fabric(self.uid, self.file_id)


class Address(object):
    """
    uid             - UID владельца
    id              - полный адрес
    path            - PATH ресурса
    storage_name    - место, где ресурс хранится (disk, narod, video ...)
    storage_id      - адрес storage
    name            - строка имени ресурса
    parent_id       - адрес родительского каталога
    parent_name     - имя родительского каталога
    is_folder       - каталог, если это уже известно, иначе нечто
    is_root         - path == "/"
    is_storage      - path in (/disk/, /narod/, ..., /N-1, /N)
    """
    FULL_PATH_RE = re.compile(r'[^/]*:/')

    def __init__(self, address, is_folder=False, is_file=False, type='', strict=False, uid=None):
        if not address:
            raise errors.AddressError('address is empty')

        if not self.FULL_PATH_RE.match(address):
            if uid is None:
                raise errors.AddressError('bad address format: %s' % address)
            else:
                address = CLN.join([str(uid), address])

        try:
            self.uid, rawpath = address.split(CLN, 1)
        except Exception:
            raise errors.AddressError('address format is UID:PATH, not %s' % address)

        if not self.uid or not rawpath:
            raise errors.AddressError('address format is UID:PATH, not %s' % address)

        if rawpath[:1] != sep:
            raise errors.AddressError('address path should starts with / not %s' % address)

        if sep*2 in rawpath:
            raise errors.AddressError('address contains double slash: %s' % address)

        chunks = filter(None, rawpath.split(sep))
        self.is_folder = rawpath[-1:] == sep or len(chunks) == 1 or is_folder
        self.is_file = is_file

        if self.is_file:
            self.is_folder = False
        if self.is_folder:
            self.is_file = False
        if type:
            self.is_file   = type == 'file'
            self.is_folder = type == 'dir'

        if not chunks and rawpath == sep:
            '''
            Case root path
            '''
            self.id = self.storage_id = self.parent_id = address
            self.path = self.name = self.rid = self.storage_name = \
                self.storage_path = self.parent_name = self.parent_path = rawpath
        else:
            self.id = address
            self.path = sep + sep.join(chunks)
            self.name = chunks[-1:][0]
            self.storage_name = chunks[:1][0]
            self.storage_path = sep + self.storage_name
            self.storage_id   = self.uid + CLN + self.storage_path
            '''
            rid: for some services name and resource id are different
            '''
            self.rid = self.name

            if self.storage_name in ('mail', '/mail', '/mail/') and not self.is_folder:
                if ':' not in chunks[-1]:
                    self.name = sep.join(chunks[-2:])

            if self.storage_name in ('mulca', '/mulca', '/mulca/'):
                self.name = 'unknown'
            else:
                if self.storage_name in ('yavideo', '/yavideo', '/yavideo/'):
                    if len(chunks) == 2:
                        self.is_file  = False
                        self.is_folder= True

            if len(chunks) < 2:
                self.parent_name = sep
                self.parent_path = sep
            else:
                self.parent_name = chunks[-2]
                self.parent_path = sep + sep.join(chunks[:-1])

            self.parent_id = self.uid + CLN + self.parent_path

        if strict:
            if ((self.name in settings.address['bad_names']) or
                    len(self.name) > settings.address['max_name_length'] or
                    len(self.path) > settings.address['max_path_length']):
                raise errors.AddressError('bad address format: %s' % address)

    def __repr__(self):
        return "%r.%r" % (self.__class__, self.id)

    def get_path_depth_lvl(self):
        if self.path == sep:
            return 0
        else:
            return self.path.count(sep)

    @property
    def is_system(self):
        return self.is_storage or self.is_root

    @property
    def is_storage(self):
        return len(filter(None, self.path.split(sep))) == 1

    @property
    def is_root(self):
        return self.path == sep

    def clone(self, raw):
        new_address = Address(raw)
        new_address.is_file   = self.is_file
        new_address.is_folder = self.is_folder
        return new_address

    def add_suffix(self, suffix):
        '''
        Добавит к имени файла суффикс
        Сделает это мудро, не затронув расширения
        '''
        new = generete_name_with_suffix(self.name, suffix)
        return self.rename(new)

    def rename(self, new_name):
        return self.get_parent().get_child_resource(new_name)

    def clone_to_parent(self, new_parent):
        '''
        new_parent - full id of new parent
        '''
        if self.uid != new_parent.uid:
            errors.AddressError('for clone must be equal uids')

        new_address = Address(self.id, self.is_folder, self.is_file)
        chunks = filter(None, new_parent.path.split(sep))
        new_address.parent_path = sep + sep.join(chunks)
        new_address.parent_id = self.uid + CLN + new_address.parent_path
        new_address.parent_name = new_parent.name
        new_address.path = new_parent.path + sep + self.name
        new_address.id = self.uid + CLN + new_address.path
        new_address.storage_name = chunks[:1][0]
        new_address.storage_path = sep + new_address.storage_name
        new_address.storage_id = self.uid + CLN + new_address.storage_path
        return new_address

    def get_child_resource(self, rid):
        '''
        rid - child's name or id
        '''
        name = filter(None, rid.split(sep)).pop()
        new_address = Address(self.id, self.is_folder, self.is_file)
        new_address.storage_id   = self.storage_id
        new_address.storage_path = self.storage_path
        new_address.storage_name = self.storage_name
        new_address.parent_id   = self.id
        new_address.parent_name = self.name
        new_address.parent_path = self.path
        new_address.path = self.path + sep + name
        new_address.path = sep + sep.join(filter(None, new_address.path.split(sep)))
        new_address.id   = self.uid + CLN + new_address.path
        new_address.name = name
        return new_address

    def get_child(self, rid):
        return self.get_child_resource(rid)

    def get_child_file(self, rid):
        '''
        rid - child's name or id
        '''
        new_address = self.get_child_resource(rid)
        new_address.is_folder = False
        new_address.is_file = True
        return new_address

    def get_child_folder(self, new_address):
        '''
        rid - child's name or id
        '''
        if not isinstance(new_address, Address):
            new_address = self.get_child_resource(new_address)
        new_address.is_file = False
        new_address.is_folder = True
        if new_address.is_storage:
            new_address.storage_name = new_address.name
            new_address.storage_id = new_address.id
            new_address.storage_path = new_address.path
        return new_address

    def get_parent(self):
        '''
        Returns parent path
        '''
        parent_address = Address(self.parent_id)
        parent_address.is_folder = True
        return parent_address

    def clean_id(self):
        self.id = self.get_clean_id()

    def get_clean_id(self):
        return self.uid + CLN + self.get_clean_path()

    def clean_path(self):
        self.path = self.get_clean_path()

    def get_clean_path(self):
        return sep + sep.join(filter(None, self.path.split(sep)))

    def change_storage(self, new_storage_name):
        self.storage_name = new_storage_name
        self.storage_path = sep + new_storage_name
        self.storage_id = self.uid + CLN + self.storage_path
        self.path = sep + new_storage_name + sep + sep.join(filter(None, self.path.split(sep))[1:])
        self.id = self.uid + CLN + self.path
        self.parent_path = sep + new_storage_name + sep + sep.join(filter(None, self.parent_path.split(sep))[1:])
        self.parent_id = self.uid + CLN + self.parent_path

    def change_uid(self, new_uid):
        self.storage_id = new_uid + CLN + self.storage_path
        self.id = new_uid + CLN + self.path
        self.parent_id = new_uid + CLN + self.parent_path
        self.uid = new_uid

    def add_trash_suffix(self):
        delta = datetime.datetime.now() - datetime.datetime.utcfromtimestamp(0)
        suffix_microseconds = str((delta.days * 24 * 3600 + delta.seconds) * 10 ** 6 + delta.microseconds)

        suffix = str(suffix_microseconds) + self.path
        suffix = suffix.encode('utf-8') if isinstance(suffix, unicode) else suffix
        suffix = hashlib.sha1(suffix).hexdigest()

        self.id = '_'.join([self.id, suffix])
        self.path = '_'.join([self.path, suffix])

    @property
    def ext(self):
        return os.path.splitext(self.path)[1][1:]

    def change_parent(self, new_parent, old_parent=None):
        if not isinstance(new_parent, Address):
            try:
                new_parent = Address(new_parent)
            except errors.AddressError:
                new_parent = Address('%s:%s' % (self.uid, new_parent))
        if old_parent:
            if not isinstance(old_parent, Address):
                try:
                    old_parent = Address(old_parent)
                except errors.AddressError:
                    old_parent = Address('%s:%s' % (self.uid, old_parent))
            old_parent_parts = old_parent.path.split(sep)
            self_parts       = self.path.split(sep)[len(old_parent_parts):]
            new_parent_parts = new_parent.path.split(sep)
            new_parent_parts.extend(self_parts)
            self.path = sep + sep.join(filter(None, new_parent_parts))
            self.parent_path = sep.join(new_parent_parts[:-1])
        else:
            self.path = sep + sep.join(filter(None, new_parent.path.split(sep))) + sep + self.name
            self.parent_path  = new_parent.path
            self.parent_id    = new_parent.id

        self.uid = new_parent.uid
        self.id  = self.uid + CLN + self.path
        self.parent_id = self.uid + CLN + self.parent_path
        self.storage_id   = new_parent.storage_id
        self.storage_path = new_parent.storage_path
        self.storage_name = new_parent.storage_name

    def drop_path_to_root(self):
        path_end = self.path.split(sep)[-1]
        self.path = sep + self.storage_name + sep + path_end
        self.id   = self.uid + CLN + self.path
        self.parent_path = self.storage_path
        self.parent_id   = self.storage_id

    def is_subfolder(self, address):
        return self.path.startswith(address.path + sep)

    def get_locked_top(self):
        splitted = filter(None, self.path.split(sep))
        top_parent_path = sep + sep.join(splitted[0:2])
        return Address('%s:%s' % (self.uid, top_parent_path))

    @classmethod
    def Make(classname, uid, path, type=None):
        return classname('%s%s%s' % (uid, CLN, path), type)

    @classmethod
    def MakeRelative(classname, uid, root, path):
        result = path.replace(root, '')
        if not result:
            result = sep
        return classname('%s%s%s' % (uid, CLN, result))

    @staticmethod
    def path_contains_path(what, where, equals=False):
        """Функция проверки вхождения одного пути в другой
        в том числе полного совпадения.
        """
        what_parts = filter(lambda x: x, what.split(sep))
        where_parts = filter(lambda x: x, where.split(sep))

        if equals:
            return what_parts == where_parts

        for i in xrange(len(what_parts)):
            try:
                if what_parts[i] != where_parts[i]:
                    return False
            except IndexError:
                return False

        return True

    def get_parents(self):
        """Возвращает адреса родительских папок (предков) до корня (не включая)."""
        parents_addresses = []
        current_address = self
        while not current_address.is_root:
            parent_address = current_address.get_parent()
            parents_addresses.append(parent_address)
            current_address = parent_address
        return parents_addresses

    def get_path_without_area(self):
        return sep.join(filter(None, self.path.split(sep))[1:])


class SymlinkAddress(object):
    """
    uid  - UID владельца
    id   - полный адрес
    path - path линка
    """
    def __init__(self, address):
        if not address:
            raise errors.AddressError('symlink is empty')
        try:
            self.uid, rawpath = address.split(CLN, 1)
        except Exception:
            raise errors.AddressError('symlink format is UID:UUID4, not %s' % address)

        if not self.uid or not rawpath:
            raise errors.AddressError('symlink format is UID:UUID4, not %s' % address)

        if not rawpath.isalnum():
            raise errors.AddressError('symlink path must be uuid4, not %s' % address)

        self.id = address
        self.path = rawpath

    @classmethod
    def Make(classname, uid, path):
        return classname('%s%s%s' % (uid, CLN, path))


class GroupAddress(Address):
    def __init__(self, address, real_address, is_folder=False, is_file=False, type=''):
        super(GroupAddress, self).__init__(real_address, is_folder=is_folder, is_file=is_file, type=type)
        self.committer, self.committer_path = address.split(CLN, 1)
        self.visible_id = address


class PublicAddress(object):
    """
    hash - приватный хеш главного ресурса или публичная ссылка, короткая или полная
    path - относительный адрес
    """
    def __init__(self, address):
        if not address:
            raise errors.AddressError('hash is empty')
        self.hash, self.path = public_urls.any_url_to_private_hash(address, split_url=True)
        if self.path:
            self.relative = Address.Make('0', self.path)
        else:
            self.relative = None
        if not self.hash:
            raise errors.AddressError('hash format is HASH:PATH or HASH, not %s' % address)

        self.id = address

    @classmethod
    def Make(classname, hash, path):
        return classname('%s%s%s' % (hash, CLN, path))

    @classmethod
    def MakeRelative(classname, hash, root, path):
        result = path.replace(root, '')
        if not result:
            result = sep
        return classname('%s%s%s' % (hash, CLN, result))

    def has_relative(self):
        return self.relative is not None


def change_parent(child, new_parent, old_parent):
    child.change_parent(new_parent, old_parent)
    return child


def group_by_parent(resources):
    resources.sort(natsort.natcasecmp)
