# -*- coding: utf-8 -*-
import random
import time
from collections import defaultdict, OrderedDict
from bson import ObjectId
from itertools import izip

from mpfs.core import factory
from mpfs.core.address import Address, ResourceId
from mpfs.core.albums.dao.album_items import AlbumItemDAO
from mpfs.core.albums.dao.albums import AlbumDAO
from mpfs.core.albums.models import Album, AlbumItem
from mpfs.core.controllers import BaseModelController
from mpfs.core.filesystem.base import Filesystem
from mpfs.core.global_gallery.logic.controller import GlobalGalleryController
from mpfs.core.user.constants import PHOTOUNLIM_AREA_PATH, DISK_AREA_PATH, ATTACH_AREA_PATH
from mpfs.metastorage.mongo.collections.albums import AlbumsCollection, AlbumItemsCollection
from mpfs.metastorage.mongo.collections.filesystem import (
    PhotounlimDataCollection, UserDataCollection, AttachDataCollection,
)
from mpfs.metastorage.mongo.util import propper_key
from mpfs.metastorage.postgres.schema import ItemsSortingMethod


class AlbumsController(BaseModelController):
    model = Album
    collection = AlbumsCollection()
    dao = AlbumDAO()

    @staticmethod
    def select_covers(albums):
        cover_items = [album.cover for album in albums if album.cover]
        # вернутся только те элементы объекты для которых удалось достать
        cover_items = AlbumItemsController().bulk_load_objects_for_items(cover_items)
        # устанавливаем новые обложки альбомов, которые не удалось достать
        for album in albums:
            if album.cover not in cover_items:
                album.select_new_cover()

    def bulk_load(self, count_items=False, load_source_ids=False):
        """
        Загружает все попадающие в фильтр альбомы и их обложки наименьшим количеством запросов.
        """
        albums = list(self)

        if count_items:
            albums_map_by_owner = defaultdict(dict)
            for album in albums:
                album.items_count = 0
                albums_map_by_owner[album.uid][album.id] = album

            for uid, albums_map in albums_map_by_owner.iteritems():
                # вытаскиваем элементы альбомов и подгружаем для них ресурсы
                items = AlbumItem.controller.filter(album_id={'$in': albums_map.keys()}, uid=uid)
                items = AlbumItemsController().bulk_load_objects_for_items(items)

                item_map = defaultdict(dict)
                for item in items:
                    item_map[item.album_id][item.id] = item

                for album_id, items in item_map.iteritems():
                    album = albums_map[album_id]
                    album.items_count = len(items)
                    album.items = items.values()
                    if album.cover.id in items:
                        album.cover = items[album.cover.id]
                    elif items:
                        album.cover = items.itervalues().next()
        else:
            self.select_covers(albums)
        if load_source_ids:
            GlobalGalleryController.load_and_set_source_ids_for_resources([x.cover.object for x in albums])
        return albums

    def find_with_sorting_by_items_count(self, uid, album_type, amount=None):
        albums = []
        for item, items_count in self.dao.find_with_sorting_by_items_count(uid, album_type, amount, self.offset):
            album = self.model.from_db(db_object=item)
            album.items_count = items_count
            albums.append(album)
        self.select_covers(albums)
        return albums

    def _fetch_one(self, spec):
        album = self.dao.find_with_cover(spec)
        if album and len(album) > 0:
            return album[0]
        return None

    def _fetch_all(self):
        self._result_cache = [self.model.from_db(db_object=i) for i in self.dao.find_with_cover(**self._as_pymongo_kwargs())]
        return self._result_cache


class AlbumItemsController(BaseModelController):
    model = AlbumItem
    collection = AlbumItemsCollection()
    dao = AlbumItemDAO()

    def bulk_load(self):
        """Полностью загружает список элементов наименьшим количеством запросов."""
        # загружаем сами элементы
        albums_items = list(self)
        return self.bulk_load_objects_for_items(albums_items)

    @staticmethod
    def is_shared_resource(resource):
        return getattr(resource, 'is_shared', False)

    def bulk_append_items_from_paths(self, album_id, uid, paths, commit=True, if_not_exists=False):
        """
        Добавление элементов(из путей) в конец альбома.

        Значения параметров смотри у bulk_create_items_from_paths
        """
        last_index = AlbumsController().get(uid=uid, id=album_id).get_last_item_index()
        order_indexes = range(last_index, last_index + len(paths))
        return self.bulk_create_items_from_paths(self, album_id, uid, paths, order_indexes, commit=commit, if_not_exists=if_not_exists)

    def bulk_create_items_from_paths(self, album_id, uid, paths, order_indexes, commit=True, if_not_exists=False, album_items_sorting=None):
        """
        Создаёт элементы альбома из путей ресурсов.

        :param album_id: Идентификатор альбома.
        :param uid: Идентификатор владельца альбома.
        :param paths: Список путей ресурсов из которых следует создать элементы альбома.
        :param order_indexes: Список номеров элементов для каждого пути из `paths`.
        :param commit: Сохранять созданные элементы в БД или нет.
        :param if_not_exists: Добавлять или нет одинаковые элементы в альбом.
        """
        if len(order_indexes) != len(paths):
            raise ValueError('Lengths of "order_indexes" and "paths" must be equal.')
        ordered_paths = [{'order': o, 'path': p} for o, p in izip(order_indexes, paths)]

        album_obj_ids = set()
        if if_not_exists:
            # отфильтровываем заведомо одинаковые файлы по path
            ordered_paths = OrderedDict((o['path'], o) for o in ordered_paths).values()
            # формируем фильтр по уже имеющимся в альбоме файлам
            album = AlbumsController().get(uid=uid, id=album_id)
            if album:
                album_obj_ids = {item.obj_id for item in album.items if item.obj_type == AlbumItem.RESOURCE}

        # преобразуем пути в Address'а на случай путей с/без uid'а в начале
        addresses = [Address(p['path'], uid=uid) for p in ordered_paths]

        resources = {}
        for resource in factory.get_resources(uid, addresses, available_service_ids=['/disk', PHOTOUNLIM_AREA_PATH, '/attach',]):
            if (if_not_exists and
                    'file_id' in resource.meta and
                    resource.meta['file_id'] in album_obj_ids):
                # фильтруем одинаковые файлы
                continue
            if getattr(resource, 'is_shared', False):
                resources[resource.visible_address.path] = resource
            else:
                resources[resource.path] = resource

        objs = []
        need_unzip_file_id_resources = []
        for ordered_path in ordered_paths:
            order_index, path = ordered_path['order'], ordered_path['path']
            resource = resources.get(propper_key(path))
            if resource:
                date_created = time.time()
                # достаём file_id где бы он ни был: в data или в zdata
                if resource.is_file_id_zipped():
                    need_unzip_file_id_resources.append(resource)

                if album_items_sorting is not None:
                    if album_items_sorting == ItemsSortingMethod.BY_DATE_CREATED:
                        # прибавляем случайное число от 0.000 до 0.999 для того что бы не прыгал порядок у фото
                        # сделаных в одну и ту же секунду (в базе время у файлов может храниться с точностью до секунды)
                        order_index = (resource.meta.get('etime') or resource.ctime or resource.mtime) + (random.randint(0, 999) / 1000.0)
                    elif album_items_sorting == ItemsSortingMethod.BY_PHOTOSLICE_TIME:
                        order_index = date_created

                # создаём элемент
                item = AlbumItem(uid=uid, album_id=album_id, order_index=order_index, obj_id=resource.meta['file_id'], id=str(ObjectId()), date_created=date_created)
                if self.is_shared_resource(resource):
                    item.obj_type = AlbumItem.SHARED_RESOURCE
                    item.group_id = resource.group.gid
                else:
                    item.obj_type = AlbumItem.RESOURCE

                # грязным хаком подсовываем во вновь созданный элемент альбома ресурс, на который он ссылается,
                # чтоб лишний раз не ходить в БД за ним
                item._object = resource
                objs.append(item)

        if objs:
            # раззиповываем file_id у ресурсов, у которых он не раззипован
            for resource in need_unzip_file_id_resources:
                resource.save()
            # сохраняем элементы в БД и возвращаем их
            if commit:
                return self.bulk_create(objs)
            else:
                return objs
        else:
            return []

    def bulk_append_items_from_albums_ids(self, album_id, uid, albums_ids, commit=True):
        """
        Добавление элементов(из идентификаторов альбомов) в конец альбома.

        Значения параметров смотри у bulk_create_items_from_albums_ids
        """
        last_index = AlbumsController().get(uid=uid, id=album_id).get_last_item_index()
        order_indexes = range(last_index, last_index + len(albums_ids))
        return self.bulk_create_items_from_albums_ids(self, album_id, uid, albums_ids, order_indexes, commit=commit)

    def bulk_create_items_from_albums_ids(self, album_id, uid, albums_ids, order_indexes, commit=True):
        """
        Создаёт элементы альбома из идентификаторов альбомов.

        :param album_id: Идентификатор альбома.
        :param uid: Идентификатор владельца альбома.
        :param album_ids: Список идентификаторов альбомов из которых следует создать элементы альбома.
        :param order_indexes: Список номеров элементов для каждого идентификатора из `albums_ids`.
        :param commit: Сохранять созданные элементы в БД или нет.
        """
        if len(order_indexes) != len(albums_ids):
            raise ValueError('Lengths of "order_indexes" and "albums_ids" must be equal.')
        ordered_album_ids = [{'order': o, 'album_id': a} for o, a in izip(order_indexes, albums_ids)]
        ordered_album_ids.sort(key=lambda i: i['order'])

        objs = []
        albums = list(AlbumsController().filter(uid=uid, id={'$in': [a['album_id'] for a in ordered_album_ids]}))
        for ordered_album_id in ordered_album_ids:
            # создаём элемент
            order, cur_album_id = ordered_album_id['order'], ordered_album_id['album_id']
            for album in albums:
                if album.id == cur_album_id:
                    break
            else:
                # не нашли нужный альбом
                continue

            item = AlbumItem(uid=uid, album_id=album_id, order_index=order, obj_type=AlbumItem.ALBUM, obj_id=album.id, date_created=int(time.time()))
            item._object = album
            objs.append(item)

        if objs:
            # сохраняем элементы в БД и возвращаем их
            if commit:
                return self.bulk_create(objs)
            else:
                return objs
        else:
            return []

    def bulk_load_objects_for_items(self, albums_items):
        # разбираем элементы по типам объектов на которые они ссылаются
        uid_albums_ids_map = defaultdict(list)
        uid_items_map = defaultdict(list)
        for item in albums_items:
            if hasattr(item, '_object'):
                continue
            if item.obj_type == AlbumItem.ALBUM:
                uid_albums_ids_map[item.uid].append(item.obj_id)
            else:
                if item.obj_type == AlbumItem.SHARED_RESOURCE:
                    owner_uid = AlbumItem.get_shared_resource_owner_uid(item.uid, item.group_id)
                else:
                    owner_uid = item.uid
                uid_items_map[owner_uid].append(item)
        # Мы не можем загрузить ресурсы для uid-а None
        uid_items_map.pop(None, None)

        objects = {}
        # загружаем пачкой все альбомы
        for owner_uid, albums_ids in uid_albums_ids_map.iteritems():
            albums = AlbumsController().filter(uid=owner_uid, id={'$in': albums_ids}).bulk_load()
            for album in albums:
                objects[(AlbumItem.ALBUM, album.id)] = album

        # загружаем пачками все ресурсы
        for owner_uid, items in uid_items_map.iteritems():
            files_ids = [item.obj_id for item in items]
            resources = [r for r in self.bulk_load_resources(owner_uid, files_ids) if r]
            resources = factory.reload_resources_for_user(items[0].uid, resources)
            resources = {r.meta['file_id']: r for r in resources if r is not None}
            for item in items:
                res = resources.get(item.obj_id, None)
                if res:
                    objects[(item.obj_type, res.meta['file_id'])] = res

        # раскладываем объекты в элементы альбомов
        ret = []
        for item in albums_items:
            if hasattr(item, '_object'):
                ret.append(item)
            else:
                obj = objects.get((item.obj_type, item.obj_id), None)
                if obj:
                    item.object = obj
                    ret.append(item)
        return ret

    @staticmethod
    def bulk_load_resources(uid, file_ids):
        """
        Загружает ресурсы пользователя.

        Если ресурс невозможно получить, то возвращает None вместо ресурса
        """
        file_ids_set = set(file_ids)
        user_data = UserDataCollection()
        attach_data = AttachDataCollection()
        photounlim_data = PhotounlimDataCollection()

        filter_ = {'uid': uid, 'data.file_id': {'$in': list(file_ids_set)}}
        raw_resources = user_data.get_all(**filter_)
        loaded_file_ids = {i['data']['file_id'] for i in raw_resources}
        # оптимизация
        for collection in (photounlim_data, attach_data):
            if len(loaded_file_ids) < len(file_ids):
                left_file_ids = list(file_ids_set - loaded_file_ids)
                raw_resources += collection.get_all(**{'uid': uid, 'data.file_id': {'$in': left_file_ids}})
                loaded_file_ids = {i['data']['file_id'] for i in raw_resources}

        ret = [None] * len(file_ids)
        if raw_resources:
            resource_ids = [ResourceId(raw_resoure['uid'], raw_resoure['data']['file_id'])
                            for raw_resoure in raw_resources]
            resources = factory.get_resources_by_resource_ids(
                uid,
                resource_ids,
                enable_service_ids=[DISK_AREA_PATH, PHOTOUNLIM_AREA_PATH, ATTACH_AREA_PATH],
                enable_optimization=True
            )
            for resource in resources:
                ret_index = file_ids.index(resource.meta['file_id'])
                # ставим ресурс на свое место в ответе
                ret[ret_index] = resource
        return ret

    def find_items_by_file_ids(self, uid, album_id, file_ids):
        return [self.model.from_db(db_object=item.get_mongo_representation(skip_missing_fields=True))
                for item in self.dao.find_items_by_file_ids(uid, album_id, file_ids)]

    @staticmethod
    def filter_blocked(items):
        """
        Отфильтровывает заблокированные ресурсы.

        :param list|AlbumItemsController items: Список элементов для фильтрации.
                                                Если объекты для элементов не загружены,
                                                то будет ходить за каждым элементом в базу,
                                                поэтому лучше вызывать после `bulk_load_resources`.

        :return: Список элементов с незаблокированными ресурсами.
        :rtype: list
        """
        FILE_ITEM_TYPES = set([AlbumItem.SHARED_RESOURCE, AlbumItem.RESOURCE])
        resources = [item.object for item in items if item.obj_type in FILE_ITEM_TYPES]
        resources = set(Filesystem().filter_by_blocked_hids(resources))
        result = []
        for item in items:
            if item.obj_type in FILE_ITEM_TYPES and item.object not in resources:
                continue
            else:
                result.append(item)
        return result
