# -*- coding: utf-8 -*-

from bson import ObjectId

from mpfs.core.filesystem.dao.file import FileDAOItem
from mpfs.dao.base import BaseDAO, BaseDAOItem, PostgresBaseDAOImplementation
from mpfs.dao.fields import AlbumItemTypeField, Md5Field, NullableStringField, ObjectIdField, RealField, StringField, \
    UidField, JsonField
from mpfs.dao.session import Session
from mpfs.metastorage.postgres.queries import SQL_INTERSECT_RESOURCE_ID_WITH_ALBUM_ITEMS
from mpfs.dao.fields import AlbumItemTypeField, Md5Field, NullableStringField, ObjectIdField, RealField, StringField, \
    UidField, DateTimeField
from mpfs.metastorage.postgres.schema import album_items
from mpfs.platform.exceptions import FieldValidationError


class AlbumItemDAOItem(BaseDAOItem):
    mongo_collection_name = 'album_items'
    postgres_table_obj = album_items
    is_sharded = True

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = ObjectIdField(mongo_path='_id', pg_path=album_items.c.id)
    uid = UidField(mongo_path='uid', pg_path=album_items.c.uid)
    album_id = ObjectIdField(mongo_path='album_id', pg_path=album_items.c.album_id)
    description = NullableStringField(mongo_path='description', pg_path=album_items.c.description, default_value=None)
    group_id = Md5Field(mongo_path='group_id', pg_path=album_items.c.group_id, default_value=None)
    order_index = RealField(mongo_path='order_index', pg_path=album_items.c.order_index)
    obj_id = StringField(mongo_path='obj_id', pg_path=album_items.c.obj_id)
    obj_type = AlbumItemTypeField(mongo_path='obj_type', pg_path=album_items.c.obj_type)
    date_created = DateTimeField(mongo_path='ctime', pg_path=album_items.c.date_created, default_value=None)
    face_info = JsonField(mongo_path='face_info', pg_path=album_items.c.face_info, default_value=None)
    exclude_keys_after_conversion_to_mongo = {
        'group_id': None,
        'description': None,
    }


class AlbumItemDAO(BaseDAO):
    dao_item_cls = AlbumItemDAOItem

    def __init__(self):
        super(AlbumItemDAO, self).__init__()
        self._pg_impl = PostgresAlbumItemDAOImplementation(self.dao_item_cls)

    def find_items_by_file_ids(self, uid, album_id, file_ids):
        impl = self._get_impl(uid)
        return impl.find_items_by_file_ids(uid, album_id, file_ids)


class PostgresAlbumItemDAOImplementation(PostgresBaseDAOImplementation):
    def find_items_by_file_ids(self, uid, album_id, file_ids):
        session = Session.create_from_uid(uid)
        result = session.execute(SQL_INTERSECT_RESOURCE_ID_WITH_ALBUM_ITEMS,
                                 {'uid': uid,
                                  'album_id': self.dao_item_cls.get_field_pg_representation('album_id', album_id),
                                  'file_ids': file_ids})
        return [self.doc_to_item(item)
                for item in result]

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if not isinstance(doc_or_docs, list):
            if '_id' not in doc_or_docs:
                doc_or_docs['_id'] = ObjectId()
        else:
            for doc in doc_or_docs:
                if '_id' not in doc:
                    doc['_id'] = ObjectId()

        return super(PostgresAlbumItemDAOImplementation, self).insert(doc_or_docs, manipulate, continue_on_error, **kwargs)
