# -*- coding: utf-8 -*-

from bson import ObjectId
from pymongo import ReadPreference as MongoReadPreference

from mpfs.core.albums.dao.album_items import AlbumItemDAO
from mpfs.dao.base import BaseDAO, BaseDAOItem, MongoBaseDAOImplementation, PostgresBaseDAOImplementation
from mpfs.dao.fields import AlbumLayoutField, BoolField, DateTimeField, NullableStringField, ObjectIdField, RealField, \
    StringArrayField, StringField, UidField, IntegerField
from mpfs.dao.session import Session
from mpfs.metastorage.postgres.queries import SQL_SELECT_ALBUMS_WITH_ITEMS_COUNT_SORT
from mpfs.metastorage.postgres.schema import albums


class AlbumDAOItem(BaseDAOItem):
    mongo_collection_name = 'albums'
    postgres_table_obj = albums
    is_sharded = True

    @classmethod
    def get_postgres_primary_key(cls):
        return 'id'

    id = ObjectIdField(mongo_path='_id', pg_path=albums.c.id)
    uid = UidField(mongo_path='uid', pg_path=albums.c.uid)
    title = StringField(mongo_path='title', pg_path=albums.c.title)

    cover_id = ObjectIdField(mongo_path='cover._id', pg_path=albums.c.cover_id, default_value=None)
    cover_offset_y = RealField(mongo_path='cover_offset_y', pg_path=albums.c.cover_offset_y, default_value=None)

    description = NullableStringField(mongo_path='description', pg_path=albums.c.description, default_value=None)
    public_key = NullableStringField(mongo_path='public_key', pg_path=albums.c.public_key, default_value=None)
    public_url = NullableStringField(mongo_path='public_url', pg_path=albums.c.public_url, default_value=None)
    short_url = NullableStringField(mongo_path='short_url', pg_path=albums.c.short_url, default_value=None)
    is_public = BoolField(mongo_path='is_public', pg_path=albums.c.is_public, default_value=False)
    is_blocked = BoolField(mongo_path='is_blocked', pg_path=albums.c.is_blocked, default_value=False)
    block_reason = NullableStringField(mongo_path='block_reason', pg_path=albums.c.block_reason, default_value=None)
    flags = StringArrayField(mongo_path='flags', pg_path=albums.c.flags, default_value=None)
    layout = AlbumLayoutField(mongo_path='layout', pg_path=albums.c.layout, default_value=None)
    date_created = DateTimeField(mongo_path='ctime', pg_path=albums.c.date_created, default_value=None)
    date_modified = DateTimeField(mongo_path='mtime', pg_path=albums.c.date_modified, default_value=None)
    social_cover_stid = NullableStringField(mongo_path='social_cover_stid', pg_path=albums.c.social_cover_stid, default_value=None)
    fotki_album_id = IntegerField(mongo_path='fotki_album_id', pg_path=albums.c.fotki_album_id,
                                  default_value=None)
    album_type = NullableStringField(mongo_path='album_type', pg_path=albums.c.album_type, default_value=None)
    album_items_sorting = NullableStringField(mongo_path='album_items_sorting', pg_path=albums.c.album_items_sorting, default_value=None)
    is_desc_sorting = BoolField(mongo_path='is_desc_sorting', pg_path=albums.c.is_desc_sorting, default_value=False)
    validation_ignored_mongo_dict_fields = ('cover',)
    hidden = BoolField(mongo_path='hidden', pg_path=albums.c.hidden, default_value=None)

    exclude_keys_after_conversion_to_mongo = {
        'cover_offset_y': None,
        'description': None,
        'public_key': None,
        'public_url': None,
        'short_url': None,
        'block_reason': None,
        'flags': None,
        'layout': None,
        'ctime': None,
        'mtime': None,
        'social_cover_stid': None,
    }


class AlbumDAO(BaseDAO):
    dao_item_cls = AlbumDAOItem

    def __init__(self):
        super(AlbumDAO, self).__init__()
        self._mongo_impl = MongoAlbumDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresAlbumDAOImplementation(self.dao_item_cls)

    def find_with_cover(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        impl = self._get_impl(spec)
        return impl.find_with_cover(spec, fields, skip, limit, sort, **kwargs)

    def find_with_sorting_by_items_count(self, uid, album_type, limit=None, offset=0):
        impl = self._get_impl(uid)
        return impl.find_with_sorting_by_items_count(uid, album_type, limit, offset)


class MongoAlbumDAOImplementation(MongoBaseDAOImplementation):
    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        return super(MongoAlbumDAOImplementation, self).insert(doc_or_docs, manipulate, continue_on_error, **kwargs)

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        return super(MongoAlbumDAOImplementation, self).update(spec, document, upsert, multi, **kwargs)

    def find(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        # чтобы сделать логику как в постгре
        result = list(super(MongoAlbumDAOImplementation, self).find(spec, fields, skip, limit, sort, **kwargs))
        for entry in result:
            entry['is_public'] = entry.get('is_public', False)
            entry['is_blocked'] = entry.get('is_blocked', False)
        return result

    def find_with_cover(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        return self.find(spec, fields, skip, limit, sort, **kwargs)


class PostgresAlbumDAOImplementation(PostgresBaseDAOImplementation):
    item_dao = AlbumItemDAO()

    @staticmethod
    def _replace_cover_with_id(doc):
        """
        При сохранении существующего альбома, в конвертер update запросов в качестве запроса
        передается "$set: {полное монго представление}". Конвертер, если в запросе есть $set,
        просто ищет в dao_item поле по совпадению монго пути и ключа, не находит и разваливается.
        Поэтому существует этот костыль. Нужно только для апдейта.
        """
        if 'cover' in doc:
            cover = doc['cover']
            if cover is not None and '_id' in cover:
                doc['cover._id'] = cover['_id']
            else:
                doc['cover._id'] = None
            del doc['cover']

    @staticmethod
    def _remove_empty_cover(doc):
        if 'cover' in doc and doc['cover'] is None:
            del doc['cover']

    def _get_cover_object(self, album):
        cover = album.get('cover')
        if cover is not None and '_id' in cover:
            cover_id = cover['_id']
            cover = self.item_dao.find_one({'_id': cover_id, 'uid': album['uid']})
            album['cover'] = cover

    def insert(self, doc_or_docs, manipulate=True, continue_on_error=False, **kwargs):
        if not isinstance(doc_or_docs, list):
            if '_id' not in doc_or_docs:
                doc_or_docs['_id'] = ObjectId()
            self._remove_empty_cover(doc_or_docs)
        else:
            for doc in doc_or_docs:
                if '_id' not in doc:
                    doc['_id'] = ObjectId()
                self._remove_empty_cover(doc)

        return super(PostgresAlbumDAOImplementation, self).insert(doc_or_docs, manipulate, continue_on_error, **kwargs)

    def update(self, spec, document, upsert=False, multi=False, **kwargs):
        if '$set' in document:
            self._replace_cover_with_id(document['$set'])
        else:
            self._remove_empty_cover(document)
        return super(PostgresAlbumDAOImplementation, self).update(spec, document, upsert, multi, **kwargs)

    def find_with_cover(self, spec=None, fields=None, skip=0, limit=0, sort=None, **kwargs):
        albums = list(self.find(spec, fields, skip, limit, sort, **kwargs))
        for album in albums:
            self._get_cover_object(album)
        return albums

    def find_with_sorting_by_items_count(self, uid, album_type, limit=None, offset=0):
        session = Session.create_from_uid(uid)
        query_params = {
            'uid': uid,
            'album_type': album_type,
            'limit': limit,
            'offset': offset
        }
        sql_result = session.execute(SQL_SELECT_ALBUMS_WITH_ITEMS_COUNT_SORT, query_params)
        for doc in sql_result:
            album = self.doc_to_item(doc).get_mongo_representation(skip_missing_fields=True)
            self._get_cover_object(album)
            items_count = doc["items_count"]
            yield album, items_count
