# -*- coding: utf-8 -*-
import mpfs.core.albums.events

from mpfs.core.event_dispatcher.events import subscribe
from mpfs.common.static.tags.push import ALBUMS


def album_title_change(album):
    return {
        'root': {
            'tag': 'album',
            'parameters': {
                'id': album.id,
                'public_key': album.public_key,
                'type': 'title_change',
                'title': album.title,
            }
        }
    }


def album_cover_change(album):
    return {
        'root': {
            'tag': 'album',
            'parameters': {
                'id': album.id,
                'public_key': album.public_key,
                'type': 'cover_change',
            }
        },
        'values': [
            {
                'tag': 'cover',
                'parameters': {
                    'id': getattr(album.cover, 'id', None),
                },
            }
        ]
    }


def album_cover_offset_change(album):
    return {
        'root': {
            'tag': 'album',
            'parameters': {
                'id': album.id,
                'public_key': album.public_key,
                'type': 'cover_offset_change',
            }
        },
        'values': [
            {
                'tag': 'cover',
                'parameters': {
                    'cover_offset_y': getattr(album, 'cover_offset_y', 0),
                },
            }
        ]
    }


def album_publish(album):
    return {
        'root': {
            'tag': 'album',
            'parameters': {
                'id': album.id,
                'type': 'publish' if album.is_public else 'unpublish',
                'public_key': album.public_key,
                'is_public': album.is_public,
            }
        }
    }


def album_items_change(album, items, op_type):
    formated_items = [{'tag': 'item', 'parameters': {'id': i.id}} for i in items]
    return {
        'root': {
            'tag': 'album',
            'parameters': {
                'id': album.id,
                'type': op_type,
                'public_key': album.public_key,
            }
        },
        'values': formated_items,
    }


def album_create(album):
    return {
        'root': {
            'tag': 'album',
            'parameters': {
                'id': album.id,
                'type': 'album_create',
                'public_key': album.public_key,
            }
        }
    }


def album_remove(album):
    return {
        'root': {
            'tag': 'album',
            'parameters': {
                'id': album.id,
                'type': 'album_remove',
                'public_key': album.public_key,
            }
        }
    }


@subscribe(mpfs.core.albums.events.AlbumChangeEvent)
def send_notify(event):
    event_data = event.data
    request = event_data['request']
    album = event_data['album']
    connection_id = getattr(request, 'connection_id', '')
    push_data = None

    if isinstance(event, mpfs.core.albums.events.AlbumChangeTitleEvent):
        push_data = album_title_change(album)
    elif isinstance(event, mpfs.core.albums.events.AlbumChangeCoverEvent):
        push_data = album_cover_change(album)
    elif isinstance(event, mpfs.core.albums.events.AlbumChangeCoverOffsetEvent):
        push_data = album_cover_offset_change(album)
    elif isinstance(event, mpfs.core.albums.events.AlbumChangePublishEvent):
        push_data = album_publish(album)
    elif isinstance(event, mpfs.core.albums.events.AlbumChangeItemsAppendEvent):
        push_data = album_items_change(album, event._data['items'], 'append_items')
    elif isinstance(event, mpfs.core.albums.events.AlbumChangeItemsRemoveEvent):
        push_data = album_items_change(album, event._data['items'], 'items_remove')
    elif isinstance(event, mpfs.core.albums.events.AlbumCreateEvent):
        push_data = album_create(album)
    elif isinstance(event, mpfs.core.albums.events.AlbumRemoveEvent):
        push_data = album_remove(album)

    if push_data:
        data = {
            'class': ALBUMS,
            'connection_id': connection_id,
            'uid': album.uid,
            'xiva_data': push_data,
        }
        from mpfs.core.pushnotifier.queue import PushQueue
        push_queue = PushQueue()
        push_queue.put(data)
