# -*- coding: utf-8 -*-
import os
import time

import mpfs.core.albums.events as albums_events
from mpfs.common import errors
from mpfs.config import settings

from mpfs.common.errors import ResourceNotFound, BadRequestError, AlbumItemCanNotBeMoved
from mpfs.common.util.translation import unixtime_to_localized_date_string
from mpfs.core.albums.controllers import AlbumsController, AlbumItemsController
from mpfs.core.filesystem.resources.disk import DiskFolder, MPFSFile
from mpfs.core.filesystem.resources.attach import AttachFolder
from mpfs.common.util import from_json, AutoSuffixator
from mpfs.core.albums import errors as albums_errors
from mpfs.core.albums.models import Album, AlbumItem
from mpfs.core.address import Address, ResourceId, CLN
from mpfs.core.bus import Bus
from mpfs.core.metastorage.decorators import user_exists_and_not_blocked
from mpfs.core.metastorage.decorators import user_is_writeable
from mpfs.core.operations import manager
from mpfs.core.queue import mpfs_queue
from mpfs.core.user.base import User
from mpfs.core.user.constants import PUBLIC_UID, FAVORITES_ALBUM_NAMES
from mpfs.core.services.video_service import video, VideoStreaming
from mpfs.core.services.zaberun_service import Zaberun
from mpfs.core import factory
from mpfs.core.services.passport_service import passport
from mpfs.core.services.socialproxy_service import SocialProxy
from mpfs.core.albums.logic.common import (
    get_public_album, check_user_permissions_for_album, get_album_by_public_key_or_url)
from mpfs.metastorage.postgres.schema import AlbumType

ALBUM_GENERATED_NAME_FILE_INFO_USAGE_LIMIT = settings.album['generated_name_file_info_usage_limit']
MAX_ALBUMS_COUNT_PER_PAGE = settings.album['max_albums_count_per_page']
FEATURE_TOGGLES_CREATE_GEO_ALBUMS_FOR_WEB_USERS = settings.feature_toggles['create_geo_albums_for_web_users']
ALBUM_DEFAULT_NAME = 'album'


def _allow_public_preview(item, req_uid):
    if item.album.is_public:
        return True
    elif not req_uid:
        return False
    elif req_uid == item.uid and item.is_shared:
        return True
    else:
        return False


def _prepare_formatter_for_item(item, request):
    if item.obj_type in (AlbumItem.RESOURCE, AlbumItem.SHARED_RESOURCE) and item.object:
        if _allow_public_preview(item, getattr(request, 'uid', None)):
            item.object.setup_previews(PUBLIC_UID)
        item.object.set_request(request)
    elif item.obj_type == AlbumItem.ALBUM and item.object:
        _prepare_formatter_for_album(item.object, request)


def _prepare_formatter_for_album(album, request):
    if album.cover:
        _prepare_formatter_for_item(album.cover, request)


def _prepare_formatter(objects, request):
    # Форматерная магия позволяющая фильтровать мету. Концы искать там: mpfs.frontend.formatter.disk.json.JSON
    if not isinstance(objects, (list, tuple)):
        objects = [objects]
    for obj in objects:
        if isinstance(obj, Album):
            _prepare_formatter_for_album(obj, request)
        elif isinstance(obj, AlbumItem):
            _prepare_formatter_for_item(obj, request)


def _generate_album_title(uid, album_items, album_type=AlbumType.PERSONAL.value):
    """
    https://st.yandex-team.ru/CHEMODAN-48640
    """
    locale = User(uid).get_supported_locale()

    if album_type == AlbumType.FAVORITES.value:
        return FAVORITES_ALBUM_NAMES[locale] if locale in FAVORITES_ALBUM_NAMES else FAVORITES_ALBUM_NAMES['ru']

    resources = [
        item.object for item in album_items if item.obj_type in {AlbumItem.RESOURCE, AlbumItem.SHARED_RESOURCE}]

    if not resources or len(resources) > ALBUM_GENERATED_NAME_FILE_INFO_USAGE_LIMIT:
        return unixtime_to_localized_date_string(time.time(), locale)

    has_resource_wihtout_etime = any(x for x in resources if x.meta.get('etime') is None)
    if has_resource_wihtout_etime:
        min_time = min(x.mtime for x in resources)
        max_time = max(x.mtime for x in resources)
    else:
        min_time = min(x.meta['etime'] for x in resources)
        max_time = max(x.meta['etime'] for x in resources)

    left = unixtime_to_localized_date_string(min_time, locale)
    right = unixtime_to_localized_date_string(max_time, locale)
    if left == right:
        return left
    return u'%s – %s' % (left, right)


def _bulk_create_items(album, items, if_not_exists=False):
    """
    Создаёт элементы альбома.

    Разбивает элементы по типам и прокидывает в соответсвующие методы
    """
    if not album:
        raise ResourceNotFound()
    # выясняем номер первого добавляемого элемента
    last_index = album.get_last_item_index()
    if last_index < 0:
        # если результат отрицательный, значит в альбоме ещё нет элементов
        last_index = 0

    resources_indexes = []
    resources_paths = []
    albums_indexes = []
    albums_ids = []
    last_index += 1
    for index, item in enumerate(items):
        if item.get('type') == AlbumItem.ALBUM:
            albums_ids.append(item['album_id'])
            albums_indexes.append(index + last_index)
        elif item.get('type') == AlbumItem.RESOURCE:
            resources_paths.append(item['path'])
            resources_indexes.append(index + last_index)

    items = []
    album_id, uid = album.id, album.uid
    if resources_paths:
        resources_items = AlbumItem.controller.bulk_create_items_from_paths(
            album_id, uid, resources_paths, resources_indexes,
            commit=False, if_not_exists=if_not_exists, album_items_sorting=album.album_items_sorting
        )
        items.extend(resources_items)
    if albums_ids:
        albums_items = AlbumItem.controller.bulk_create_items_from_albums_ids(
            album_id, uid, albums_ids, albums_indexes, commit=False
        )
        items.extend(albums_items)
    return sorted(items, key=lambda i: i.order_index, reverse=bool(album.is_desc_sorting))


def _albums_create(req, raw_album):
    cover_index = raw_album.pop('cover', None)
    if_not_exists = bool(getattr(req, 'if_not_exists', 0))

    # достаём элементы альбома
    raw_items = raw_album.get('items', [])
    cover_path = None
    if cover_index is not None:
        if cover_index < 0 or not raw_items or cover_index > len(raw_items) - 1:
            raise albums_errors.AlbumsCoverIndexOutOfRangeError()
        cover_path = raw_items[cover_index].get('path')

    album_type = raw_album.get('album_type') or getattr(req, 'album_type', None)
    if album_type not in (AlbumType.FAVORITES.value, AlbumType.PERSONAL.value):  # geo альбомы создаются не тут
        raise BadRequestError(extra_msg='Unknown album type')

    is_public = album_type != AlbumType.FAVORITES.value and raw_album.get('is_public', True)
    # создаём альбом
    raw_album['album_type'] = album_type
    album = Album(**raw_album)
    album.is_public = is_public
    album.uid = req.uid
    # создаём элементы альбома
    items = _bulk_create_items(album, raw_items, if_not_exists=if_not_exists)

    if not raw_album.get('title') or album_type == AlbumType.FAVORITES.value:
        album.title = _generate_album_title(req.uid, items, album_type)

    # в PostgreSQL есть FK album.cover_id -> album_item.id; надо либо всё записать в одной транзакции, либо по очереди
    album.save()

    # проставляем album_id всем элементам альбома перед сохранением
    for item in items:
        item.album_id = album.id
    # сохраняем элементы в БД
    if items:
        items = AlbumItem.controller.bulk_create(items)

    # выбираем обложку альбома
    cover = None
    if cover_path is None:
        # ищем подходящий на обложку элемент альбома
        for item in items:
            if item.obj_type in (AlbumItem.RESOURCE, AlbumItem.SHARED_RESOURCE) and item.object.type == 'file':
                cover = item
                break
    else:
        for item in items:
            if item.object.id == cover_path:
                cover = item
                break
    album.cover = cover

    # При создании альбома обновить обложку надо синхронно
    album.force_sync_update_social_cover = True
    album.save()

    if is_public:
        album.publish()

    _prepare_formatter(album, req)
    _prepare_formatter(items, req)
    album.items = items
    albums_events.AlbumCreateEvent(album=album, request=req).send()
    return album


ALBUMS_CREATE_ACCEPTABLE_KEYS = ['title', 'cover', 'layout', 'flags', 'description', 'cover_offset_y', 'fotki_album_id']


def _prepare_raw_album(req):
    raw_album = {k: getattr(req, k, None) for k in ALBUMS_CREATE_ACCEPTABLE_KEYS if getattr(req, k, None)}
    if raw_album.get('cover'):
        cover_path = raw_album.pop('cover')
        raw_album['items'] = [{'type': AlbumItem.RESOURCE, 'path': cover_path}]
        raw_album['cover'] = 0
    if raw_album.get('flags') is not None:
        raw_album['flags'] = filter(None, [f.strip() for f in raw_album['flags'].split(',')])
    return raw_album


def albums_create(req):
    """
    Создаёт пустой альбом по query string параметрам.

    Query string аргументы:
      * uid [обязательный]
      * title - название альбома. Default: "Новый альбом". Названия локализуются в зависимости от локали пользователя.
      * cover - путь к файлу-обложке альбома, будет автоматически добавлен как первый элемент альбома.
      * layout - сетка альбома. Допустимые значения: "squares", "rows", "waterfall", "fit_width".
      * flags - список флагов альбома через запятую.
      * description - описание альбома.
      * album_type - тип альбома.
    """
    return _albums_create(req, _prepare_raw_album(req))


def albums_create_from_folder(req):
    """
    Создаёт альбом из папки.

    Query string аргументы:
      * uid [обязательный]
      * path [обязательный] - путь до папки, из которой надо сделать альбом.
      * media_type - список типов файлов, которые будут добавлены в альбом(EX: media_type=image,video).
      * title - название альбома. Default: "Новый альбом". Названия локализуются в зависимости от локали пользователя.
      * cover - путь к файлу-обложке альбома, будет автоматически добавлен как первый элемент альбома.
      * layout - сетка альбома. Допустимые значения: "squares", "rows", "waterfall", "fit_width".
      * flags - список флагов альбома через запятую.
      * description - описание альбома.
      * album_type - тип альбома.
    """
    address = Address.Make(req.uid, req.path)
    folder = factory.get_resource(req.uid, address)
    if not isinstance(folder, (DiskFolder, AttachFolder)):
        raise ResourceNotFound()

    raw_album = _prepare_raw_album(req)
    items = folder.list()['list']
    if req.media_type is None:
        accepted_media_types = None
    else:
        accepted_media_types = {m.strip(' ') for m in req.media_type.split(',')}

    cover_path = None
    if 'items' not in raw_album:
        raw_album['items'] = []
    else:
        cover_path = raw_album['items'][0]['path']
    for item in items:
        # добавляем только файлы
        if item['type'] != 'file':
            continue
        # фильтруем по медиа типу
        if accepted_media_types is not None and item['media_type'] not in accepted_media_types:
            continue
        # откидываем обложку - она уже в items(актуально, если передана обложка-путь)
        if item['id'] == cover_path:
            continue
        raw_album['items'].append({'type': 'resource', 'path': item['id']})
    return _albums_create(req, raw_album)


def albums_create_with_items(req):
    """
    Создаёт альбом с элементами.

    Query string аргументы:
      * uid [обязательный]
      * no_items - не получать список файлов в ответе
      * if_not_exists
      * preview_size
      * preview_crop
      * preview_quality
      * connection_id
      * preview_quality
      * preview_allow_big_size
      * album_type

    Тело запроса:
    {
        "title": "Сан-Франциско",
        "cover": 0,
        "layout": "waterfall",
        "flags": ["show_dates"],
        "fotki_album_id": <идентификатор соответствующего альбома в Я.Фотки (если есть)>
        "items": [
            {"type": "resource", "path": "/disk/sf.jpg"},
            {"type": "resource", "path": "/disk/shared/sf.jpg"},
            {"type": "album", "album_id": "2fd4e1c67a2d28fced849ee1bb76e7391b93eb12"}
        ]
    }
    """
    if not req.http_req.data:
        raise albums_errors.AlbumsJsonBodyExpectedError()

    raw_album = from_json(req.http_req.data)
    return _albums_create(req, raw_album)


def albums_list(req):
    """
    Возвращает список альбомов пользователя.

    Query string аргументы:
      * uid [обязательный]
      * count_items - Отдавать количество доступных фотографий в альбоме. 0 или 1. По умолчанию 0
      * amount - количество возвращаемых альбомов
      * offset - сколько альбомомв пропустить прежде чем начать формировать ответ
      * album_type - тип возвращаемых альбомов. personal|geo|favorites|faces
    """
    allowed_album_types = settings.album['allowed_album_types']
    if req.album_type not in allowed_album_types:
        raise errors.BadRequestError('Wrong value of parameter album_type')

    load_source_ids = req.meta and 'source_ids' in req.meta
    if req.amount:
        if req.amount > MAX_ALBUMS_COUNT_PER_PAGE:
            raise errors.BadRequestError('Too big amount')
        elif req.amount < 0:
            raise errors.BadRequestError('Amount can\'t be less then zero')
        albums_controller = AlbumsController(limit=req.amount, offset=req.offset)
        amount = req.amount
    else:
        albums_controller = Album.controller
        amount = None

    if FEATURE_TOGGLES_CREATE_GEO_ALBUMS_FOR_WEB_USERS and req.album_type == AlbumType.GEO.value:
        if req.user and not req.user.has_geo_albums():
            mpfs_queue.put({'uid': req.uid}, 'create_geo_albums_operation')

    if req.album_type == AlbumType.FACES.value:
        albums = albums_controller.find_with_sorting_by_items_count(req.uid, req.album_type, amount)
    else:
        albums = albums_controller.filter(uid=req.uid).order_by('-mtime')
        if req.album_type == AlbumType.PERSONAL.value:
            albums.spec.update({'$and': [
                {'$or': [
                    {'album_type': {'$exists': False}},
                    {'album_type': 'personal'}
                ]},
                {'$or': [
                    {'hidden': False},
                    {'hidden': {'$exists': False}}
                ]}
            ]})
        else:
            albums.spec.update({'album_type': req.album_type,
                                '$or': [
                                    {'hidden': False},
                                    {'hidden': {'$exists': False}}
                                ]
                                })
        albums = albums.bulk_load(count_items=bool(req.count_items), load_source_ids=load_source_ids)
    _prepare_formatter(albums, req)
    return albums


def album_get(req):
    """
    Возвращает альбом пользователя с элементами.

    Query string аргументы:
      * uid [обязательный]
      * album_id [обязательный]
      * amount - количество элементов альбома на странице
      * last_item_id - идентификатор последнего элемента альбома полученного на прошлой странице
    """
    album = Album.controller.get(uid=req.uid, id=req.album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()

    last_item_order_index = None
    if req.last_item_id:
        last_item = album.items.get(id=req.last_item_id)
        if last_item:
            last_item_order_index = last_item.order_index

    # фильтруем элементы
    album.ensure_cover()
    album.items = album.bulk_load_items(last_item_order_index=last_item_order_index, amount=req.amount, only_public_items=False)

    _prepare_formatter(album, req)
    _prepare_formatter(album.items, req)

    return album


def _get_album_item(**kwargs):
    """
    Получение элемента альбома

    Если передан album_id, то проверяется принадлежность элемента к альбому
    """
    item = AlbumItem.controller.get(**{k: v for k, v in kwargs.iteritems() if v is not None})
    if not item:
        raise ResourceNotFound()
    return item


def album_item_info(req):
    """
    Получить информацию об элементе альбома.

    Query string аргументы:
      * item_id [обязательный] - идентификатор элемента
      * uid [обязательный] - идентификатор владельца
      * album_id - идентификатор альбома. Если передан, то проверяется принадлежность элемента альбому
    """
    item = _get_album_item(uid=req.uid, id=req.item_id, album_id=req.album_id)
    _prepare_formatter(item, req)
    return item


def album_item_check(req):
    """
    Проверяет наличие элемента в одном из альбомов пользователя

    Query string аргументы:
      * uid [обязательный]
      * path [обязательный]
      * album_id - идентификатор альбома. Если передан, то проверяется принадлежность элемента альбому
    """
    resource = factory.get_resource(req.uid, req.path)
    obj_id = resource.meta['file_id']

    item = _get_album_item(uid=req.uid, obj_id=obj_id, album_id=req.album_id)
    if not item.object:
        raise ResourceNotFound()
    return item


def album_set_attr(req):
    """
    Устанавливает значения атрибутов альбома.

    Query string аргументы:
      * uid [обязательный]
      * album_id [обязательный]
      * title - название альбома.
      * cover - порядковый номер элемента в альбоме, или id элемента.
      * cover_offset_y - смещение обложки альбома по вертикали. Тип элемента - float.
      * layout - сетка альбома. Допустимые значения: "squares", "rows", "waterfall", "fit_width".
      * flags - список флагов альбома через запятую.
      * description - описание альбома.
      * no_items - не получать список файлов в ответе
    """
    album = Album.controller.get(uid=req.uid, id=req.album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()

    # оставляем в данных только те поля которые разрешено редактировать
    raw_album = dict([(k, getattr(req, k, None)) for k in ALBUMS_CREATE_ACCEPTABLE_KEYS if getattr(req, k, None)])

    # Альбом favorites нельзя переименовать
    if album.album_type == AlbumType.FAVORITES.value:
        raw_album.pop('title', None)
    # если было указано смещение, даже нулевое, то добавляем его в raw_album
    cover_offset_y = getattr(req, 'cover_offset_y', None)
    if cover_offset_y is not None:
        raw_album.update({'cover_offset_y': cover_offset_y})

    # разбираемся с обложкой
    if 'cover' in raw_album:
        raw_cover = raw_album.get('cover')
        cover = None
        try:
            raw_cover = float(raw_cover)
            # значит указали номер элемента
            covers = AlbumItem.controller.filter(uid=req.uid, album_id=req.album_id)
            covers = list(covers.order_by('order_index')[raw_cover:raw_cover+1])
            cover = covers[0] if covers else None
        except ValueError:
            # значит указали идентификатор элемента
            cover = AlbumItem.controller.get(uid=req.uid, album_id=req.album_id, id=raw_cover)
        if cover is None:
            raw_album.pop('cover', None)
        else:
            raw_album['cover'] = cover

    if raw_album.get('flags') is not None:
        raw_album['flags'] = filter(None, [f.strip() for f in raw_album['flags'].split(',')])

    prev_title = album.title

    for k, v in raw_album.iteritems():
        setattr(album, k, v)

    album.save()
    album.ensure_cover()

    if 'title' in raw_album:
        albums_events.AlbumChangeTitleEvent(album=album, request=req, prev_title=prev_title).send()
    if 'cover' in raw_album:
        albums_events.AlbumChangeCoverEvent(album=album, request=req).send()
    if 'cover_offset_y' in raw_album:
        albums_events.AlbumChangeCoverOffsetEvent(album=album, request=req).send()

    _prepare_formatter(album, req)

    return album


def _album_append_items(uid, album_id, raw_items, req):
    """
    Добавить элементы в альбом.

    :param raw_items: list вида [{"type": "resource", "path": "/disk/sf.jpg"}, ...]
    """
    album = Album.controller.get(uid=uid, id=album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()

    if_not_exists = bool(getattr(req, 'if_not_exists', 0))
    items = _bulk_create_items(album, raw_items, if_not_exists=if_not_exists)

    # сохраняем элементы в БД
    if items:
        items = AlbumItem.controller.bulk_create(items)

    _prepare_formatter(album, req)
    _prepare_formatter(items, req)

    album.ensure_cover(itemset_changed=True)
    albums_events.AlbumChangeItemsAppendEvent(album=album, request=req, items=items).send()
    album.items = items
    album.save()
    return album


def album_append_items(req):
    """
    Добавить элементы в альбом.

    Query string аргументы:
      * uid [обязательный]
      * album_id [обязательный]

    Тело запроса:
    {
        "items": [
            {"type": "resource", "path": "/disk/sf.jpg"},
            {"type": "resource", "path": "/disk/shared/sf.jpg"},
            {"type": "album", "album_id": "2fd4e1c67a2d28fced849ee1bb76e7391b93eb12"}
        ]
    }
    """
    if not req.http_req.data:
        raise albums_errors.AlbumsJsonBodyExpectedError()
    raw_items = from_json(req.http_req.data)['items']
    return _album_append_items(req.uid, req.album_id, raw_items, req)


def album_append_item(req):
    """
    Добавляет в альбом один элемент.

    Query string аргументы:
      * uid [обязательный]
      * album_id [обязательный]
      * type [обязательный]
      * path
      * src_album_id
    """
    raw_item = {'type': req.type}
    if req.type == AlbumItem.ALBUM:
        raw_item['album_id'] = req.src_album_id
    elif req.type == AlbumItem.RESOURCE:
        raw_item['path'] = req.path
    album = _album_append_items(req.uid, req.album_id, [raw_item], req)
    if album.items:
        return album.items[0]
    raise albums_errors.AlbumsUnableToAppendItem()


def album_copy(req):
    """
    Копировать альбом.

    Query string аргументы:
      * uid [обязательный]
      * album_id [обязательный]
    """
    album = Album.controller.get(uid=req.uid, id=req.album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()

    new_album = album.copy()
    new_album.ensure_cover(itemset_changed=True)

    _prepare_formatter(new_album, req)

    return new_album


def album_remove(req):
    """
    Удалить альбом.

    Query string аргументы:
      * uid [обязательный]
      * album_id [обязательный]
    """
    album = Album.controller.get(uid=req.uid, id=req.album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()
    if album.album_type == AlbumType.FAVORITES.value:
        raise albums_errors.AlbumsUnableToDelete()
    albums_events.AlbumRemoveEvent(album=album, request=req).send()
    album.delete()


def album_item_move(req):
    """
    Переместить элемент в альбоме на новое место.

    Удалён. Сейчас пользователь не может произвольно упорядочивать элементы в альбоме
    """
    raise AlbumItemCanNotBeMoved()
    """
    item = _get_album_item(uid=req.uid, id=req.item_id, album_id=req.album_id)
    start = req.to_index if req.to_index == 0 else abs(req.to_index) - 1
    end = req.to_index + 1
    new_neighbours = list(AlbumItem.controller.filter(uid=item.uid, album_id=item.album_id).order_by('order_index')[start:end])
    if len(new_neighbours) == 2 and (new_neighbours[0].id == item.id or new_neighbours[1].id == item.id):
        # пытаемся переместить элемент на позицию по соседству с сами собой, поэтому ничего не делаем
        pass
    elif len(new_neighbours) == 2:
        # втыкаем элемент между двумя соседями
        prev_index = new_neighbours[0].order_index
        next_index = new_neighbours[1].order_index
        new_order_index = (prev_index + next_index) / 2.0
        item.order_index = new_order_index
        item.save()
    elif len(new_neighbours) == 1 and new_neighbours[0].id == item.id:
        # вставка элемента после себя или перед собой в отсутствии соседей, ничего не делаем
        pass
    elif len(new_neighbours) == 1 and start == 0:
        # вставка элемента в начало списка
        prev_index = 0
        next_index = new_neighbours[0].order_index
        new_order_index = (prev_index + next_index) / 2.0
        item.order_index = new_order_index
        item.save()
    elif len(new_neighbours) == 1:
        # вставка в конец списка
        item.order_index = new_neighbours[0].order_index + 1
        item.save()
    elif len(new_neighbours) == 0:
        # вставка за конец списка
        album = Album.controller.get(uid=item.uid, id=item.album_id)
        last_item_index = album.get_last_item_index()
        if last_item_index != item.order_index:
            item.order_index = last_item_index + 1
            item.save()
    album = Album.controller.get(uid=req.uid, id=item.album_id)
    album.save()
    _prepare_formatter(item, req)

    return item
    """

def album_item_remove(req):
    """
    Удалить элемент из альбома.

    Query string аргументы:
      * uid [обязательный]
      * item_id [обязательный]
      * album_id - идентификатор альбома. Если передан, то проверяется принадлежность элемента альбому
    """
    item = _get_album_item(uid=req.uid, id=req.item_id, album_id=req.album_id)
    item.delete()
    albums_events.AlbumChangeItemsRemoveEvent(album=item.album, request=req, items=[item]).send()


def album_item_set_attr(req):
    """
    Изменить комментарий к элементу альбома.

    Query string аргументы:
      * uid [обязательный]
      * item_id [обязательный]
      * description - комментарий к элементу
      * album_id - идентификатор альбома. Если передан, то проверяется принадлежность элемента альбому
    """
    item = _get_album_item(uid=req.uid, id=req.item_id, album_id=req.album_id)
    if req.description is not None:
        item.description = req.description
        item.save()

    _prepare_formatter(item, req)

    return item


def album_publish(req):
    """
    Опубликовать альбом.

    Query string аргументы:
      * uid [обязательный]
      * album_id [обязательный]
      * no_items - не получать список файлов в ответе
    """
    album = Album.controller.get(uid=req.uid, id=req.album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()
    album.publish()
    album.ensure_cover()
    albums_events.AlbumChangePublishEvent(album=album, request=req).send()

    _prepare_formatter(album, req)

    return album


def album_unpublish(req):
    """
    Приватизировать альбом.

    Query string аргументы:
      * uid [обязательный]
      * album_id [обязательный]
      * no_items - не получать список файлов
    """
    album = Album.controller.get(uid=req.uid, id=req.album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()
    album.unpublish()
    album.ensure_cover()
    albums_events.AlbumChangePublishEvent(album=album, request=req).send()

    _prepare_formatter(album, req)

    return album


def album_video_streams(req):
    """

    Query string аргументы:
      * uid [обязательный]
      * item_id [обязательный]
      * album_id [обязательный]
      * use_http
      * user_ip
    """
    item = _get_album_item(uid=req.uid, id=req.item_id, album_id=req.album_id)

    resource = item.get_resource()
    if not isinstance(resource, MPFSFile):
        raise errors.FileNotFound()
    return VideoStreaming().get_video_info(
        resource.uid, resource.file_mid(),
        use_http=bool(req.use_http),
        user_ip=req.user_ip,
        client_id=req.client_id
    )


@user_exists_and_not_blocked
@user_is_writeable
def albums_exclude_from_generated(req):
    return Bus(request=req).add_generated_albums_exclusion(
        req.uid,
        req.path,
        req.album_type,
    )


def album_find_in_favorites(req):
    """
    Проверяет находятся ли ресурсы в альбомы Избранное.

    Request:
    GET /json/album_find_in_favorites?uid=<uid>
    {
        "resource_ids": [
            <resource-id_1>,
            <resource-id_2>,
            ...
            <resource-id_N>
        ]
    }

    Query parameters:
      * uid [обязательный]

    Request body:
    resource_ids: список resource_id (максимум 100 шт.);
                  uid'ы игнорируются (для быстрого поиска смотрим только на file_id)

    Response:
    Status codes:
      * 200: успешно проверили переданные resource_id
      * 400: в теле передали больше 100 resource_id или передали пустое тело

    Response body:
    Содержит список переданных resource_id, которые есть в альбоме Избранное и их ID в альбоме.
    {
        "album_id": <favorite-album-id>,
        "items": [
            {"resource_id": <resource-id_1>, "item_id": <item-id_1>},
            {"resource_id": <resource-id_2>, "item_id": <item-id_2>},
            ...
            {"resource_id": <resource_id_M>, "item_id": <item-id_M>}
        ]
    }
    """
    if not req.http_req.data:
        raise albums_errors.AlbumsGETBodyExpectedError()
    try:
        resource_ids = from_json(req.http_req.data)['resource_ids']
    except Exception:
        raise albums_errors.AlbumsInvalidJSONBodyError()
    if len(resource_ids) > 100:
        raise albums_errors.AlbumsTooManyResourcesError()
    # Подготавливаем мапу для генерации результата
    # (у нас будут найденные file_id, нам нужно сопоставить их с resource_id)
    file_id_to_resource_id = {}
    for raw_resource_id in resource_ids:
        try:
            resource_id = ResourceId.parse(raw_resource_id)
        except Exception:
            raise albums_errors.AlbumsInvalidJSONBodyError()
        file_id_to_resource_id[resource_id.file_id] = raw_resource_id

    album = Album.controller.get(uid=req.uid, album_type=AlbumType.FAVORITES.value)
    if album is None:
        return {'items': []}

    in_album_items = AlbumItem.controller.find_items_by_file_ids(req.uid, album.id, file_id_to_resource_id.keys())
    # filter out items that we can't load
    existing_items = AlbumItemsController().bulk_load_objects_for_items(in_album_items)

    result = {'album_id': album.id,
              'items': [{'resource_id': file_id_to_resource_id[item.obj_id],
                         'item_id': str(item.id)}
                        for item in existing_items]}

    return result


def public_album_block(req):
    """
    Заблокировать альбом.

    Query string аргументы:
      * public_key [обязательный] - url или public_key альбома
      * reason [обязательный] - причина блокировки
    """
    album = get_album_by_public_key_or_url(req.public_key)
    if album.is_blocked is not True:
        album.is_blocked = True
        album.block_reason = req.reason
        album.save()
    return album


def public_album_unblock(req):
    """
    Разблокировать альбом.

    Query string аргументы:
      * public_key [обязательный] - url или public_key альбома
    """
    album = get_album_by_public_key_or_url(req.public_key)
    if album.is_blocked is True:
        album.is_blocked = False
        album.block_reason = ''
        album.save()
    return album


def public_album_get(req):
    """
    Получить публичный альбом с элементами.

    Query string аргументы:
      * public_key [обязательный]
      * uid - идентификатор пользователя запрашивающего публичный альбом
      * last_item_id - идентификатор последнего элемента альбома полученного на прошлой странице
      * amount - количество элементов на странице
      * preview_size - размер кастомного превью
      * preview_crop - разрешить обрезку альбома (0 или 1)
      * preview_quality - качество превью [0..100]
      * preview_allow_big_size - позволяет отдавать превью в размере > 1280
    """
    album = get_public_album(req.public_key, req.uid)
    album.ensure_cover()

    last_item = None
    last_item_order_index = None
    if req.last_item_id:
        last_item = album.items.get(id=req.last_item_id)
        if last_item:
            last_item_order_index = last_item.order_index

    items = []
    if req.uid == album.uid:
        # если пришёл владелец альбома, то загружаем все запрошенные элементы
        if album.cover and not album.cover.is_available:
            album.cover = None
        if req.amount != 0:
            items = album.bulk_load_items(last_item_order_index=last_item_order_index, amount=req.amount, only_public_items=False)
    else:
        # Если пришел не владелец и альбом публичный,
        # то загружаем только те элементы которые можно показывать посторонним.
        if req.amount != 0:
            items = album.bulk_load_items(last_item_order_index=last_item_order_index, amount=req.amount, only_public_items=True)

        if album.cover and not album.cover.is_public:
            # скрываем ковёр альбома если у владельца недостаточно прав
            album.cover = None

    _prepare_formatter(album, req)
    _prepare_formatter(items, req)

    album.items = items
    return album


def public_album_items_list(req):
    """
    Получить элементы альбома.

    Query string аргументы:
      * public_key [обязательный]
      * uid - идентификатор пользователя запрашивающего публичный альбом
      * last_item_id - идентификатор последнего элемента альбома полученного на прошлой странице
      * amount - количество элементов на странице
      * preview_size - размер кастомного превью
      * preview_crop - разрешить обрезку альбома (0 или 1)
      * preview_quality - качество превью [0..100]
      * preview_allow_big_size - позволяет отдавать превью в размере > 1280
    """
    album = public_album_get(req)
    return album.items


def public_album_check(req):
    """
    Получить минимальную информацию об альбоме.

    Возвращает название альбома и обложка

    Query string аргументы:
      * public_key [обязательный]
      * uid - идентификатор пользователя запрашивающего публичный альбом
    """
    album = get_public_album(req.public_key, req.uid)
    _prepare_formatter(album, req)
    return album


def public_album_save(req, async=False):
    """
    Сохранить элементы публичного альбома себе в Диск.

    Query string аргументы:
      * public_key [обязательный]
      * uid [обязательный] - идентификатор пользователя сохраняющего публичный альбом
      * item_id - идентификат элемента который следует сохранить
      * path - путь по которому следует сохранить элементы альбома в Диске
      * item_name - имя под которым следует сохранён элемент
    """
    album = get_public_album(req.public_key, req.uid)
    fs = Bus(request=req)

    if req.item_id:
        item = album.items.get(id=req.item_id)
        # если элемент не найден или у владельца не достаточно прав чтоб опубликовать его, то швыряем исключение
        if not item or (item.uid != req.uid and not item.is_public):
            raise ResourceNotFound()

        new_names = [req.item_name] if req.item_name is not None else [item.object.address.name]
        resources = [item.object]
    else:
        # если пришел владелец, то сохраняем все элементы, иначе только публичные элементы
        if album.uid == req.uid:
            items = album.items
        else:
            items = album.bulk_load_items(only_public_items=True)
        resources = [item.object for item in items if item.obj_type in (AlbumItem.RESOURCE, AlbumItem.SHARED_RESOURCE)]
        resources = filter(None, resources)
        auto_suffixator = AutoSuffixator()
        new_names = [auto_suffixator(r.address.name) for r in resources]

    path = req.path
    if path is None:
        if req.item_id:
            # сохраняем один файл в Downloads
            download_dir_address = fs.get_downloads_address(req.uid)
            fs.mksysdir(req.uid, type='downloads')
            path = download_dir_address.path
            dst_address = download_dir_address.get_child(new_names[0])
            dst_address = fs.autosuffix_address(dst_address)
            new_names = [dst_address.name]
        else:
            # сохраняем весь альбом в корень
            dst_dir_address = Address.Make(req.uid, "/disk/%s" % album.title)
            dst_dir_address = fs.autosuffix_address(dst_dir_address)
            path = dst_dir_address.path

    if async:
        operation = fs.async_copy_resources(req.uid, path, resources, new_names=new_names,
                                            force_djfs_albums_callback=True)
        client_path = os.path.join(path, new_names[0]) if req.item_id else path
        return {'oid': operation.id, 'type': operation.type, 'path': client_path}
    else:
        return fs.copy_resources(req.uid, path, resources, new_names=new_names)


def async_public_album_save(req):
    """
    Асинхронно сохранить элементы публичного альбома себе в Диск.

    Query string аргументы:
      * public_key [обязательный]
      * uid [обязательный] - идентификатор пользователя сохраняющего публичный альбом
      * item_id - идентификат элемента который следует сохранить
      * path - путь по которому следует сохранить элементы альбома в Диске
      * item_name - имя под которым следует сохранён элемент
    """
    return public_album_save(req, async=True)


def public_album_item_download_url(req):
    """
    Получить публичный урл на элемент публичного альбома.

    Query string аргументы:
      * public_key [обязательный]
      * uid - идентификатор пользователя, делающего запрос
      * item_id [обязательный] - идентификат элемента, который следует сохранить
      * inline - 1 или 0, меняет Disposition на inline или attach
    """
    album = get_public_album(req.public_key, req.uid)
    item = album.items.get(id=req.item_id)
    fs = Bus(request=req)
    resource = fs.resource(PUBLIC_UID, item.object.address.id)
    return resource.get_url(inline=req.inline)


def public_album_download_url(req):
    """
    Формирования url на заберун для скачивания альбома zip архивом.

    Query string аргументы:
      * public_key [обязательный]
      * uid - идентификатор пользователя, делающего запрос
    """
    album = get_public_album(req.public_key, req.uid)
    return {'url': Zaberun().generate_zip_album_url(req.uid or PUBLIC_UID,
                                                    album.public_key,
                                                    album.title or ALBUM_DEFAULT_NAME)}


def public_album_item_info(req):
    """
    Получить информацию об элементе альбома.

    Query string аргументы:
      * public_key [обязательный]
      * item_id [обязательный] - идентификатор элемента
      * uid - идентификатор пользователя, делающего запрос
    """
    album = get_public_album(req.public_key, req.uid)
    item = album.items.get(id=req.item_id)
    if not item:
        raise ResourceNotFound()
    _prepare_formatter(item, req)
    return item


def public_album_item_video_url(req):
    """
    Получить url для видеостриминга.

    Query string аргументы:
      * public_key [обязательный]
      * item_id [обязательный] - идентификатор элемента
      * uid - идентификатор пользователя, делающего запрос
      * yandexuid - yandexuid обращающегося, если определен
    """
    album = get_public_album(req.public_key, req.uid)
    item = album.items.get(id=req.item_id)
    if not item:
        raise ResourceNotFound()

    consumer = PUBLIC_UID if album.is_public else album.uid
    resource = item.object
    return video.generate_url(
        consumer, resource.uid, resource.file_mid(), resource.hid,
        public=album.is_public, video_info=resource.get_video_info(),
        consumer_yandexuid=req.yandexuid
    )


def public_album_social_wall_post(req):
    """
    Опубликовать альбом пользователя в соц. сеть через social proxy

    Query string аргументы:
      * provider [обязательный] - название соцсети для импорта (vkontakte, facebook, odnoklassniki, mailru)
      * public_key [обязательный] - публичный ключ альбома
      * uid [обязательный] - идентификатор пользователя, публикующего в соц. сети альбом
    """
    album = get_public_album(req.public_key, req.uid)

    # Забаненый альбом даже владелец не может публиковать. Ибо нефиг!
    if album.is_blocked is True:
        raise ResourceNotFound()

    return album.send_social_wall_post(req.provider)


def public_album_search_bulk_info(req):
    from mpfs.core.base import _search_resources_info_by_file_ids
    album = get_public_album(req.public_key)
    uid = album.uid
    fields = req.search_meta.split(',')
    file_ids = req.file_ids.split(',')

    # фильтруем элементы, которых нет в альбоме и которые не можем загрузить
    in_album_items = AlbumItem.controller.find_items_by_file_ids(uid, album.id, file_ids)
    existing_items = AlbumItemsController().bulk_load_objects_for_items(in_album_items)
    file_ids = [item.obj_id for item in existing_items]

    return _search_resources_info_by_file_ids(
        uid,
        file_ids,
        sort_field=req.sort,
        asc_order=bool(req.order),
        fields=fields
    )


def async_public_album_social_wall_post(req):
    """
    Асинхронно опубликовать альбом пользователя в соц. сеть через social proxy

    Query string аргументы:
      * provider [обязательный] - название соцсети для импорта (vkontakte, facebook, odnoklassniki, mailru)
      * uid [обязательный] - идентификатор пользователя, публикующего в соц. сети альбом
      * public_key - публичный ключ альбома
      * album_id - идентификатор альбома
    """

    album = None

    if req.public_key is not None:
        album = get_public_album(req.public_key, req.uid)
    elif req.album_id is not None:
        album = Album.controller.get(uid=req.uid, id=req.album_id)

    if album is None or album.is_blocked is True:
        raise ResourceNotFound()

    if not album.is_public:
        raise albums_errors.AlbumsIsNotPublicError()

    if req.provider not in SocialProxy().get_wall_post_valid_providers():
        raise albums_errors.AlbumsProviderNotFound()

    operation = manager.create_operation(
        req.uid,
        'social',
        'post_album',
        odata={
            'provider': req.provider,
            'album_id': album.id,
        }
    )

    return {'oid': operation.id, 'type': operation.type}


def fotki_album_public_url(req):
    album = Album.controller.get(uid=req.owner_uid, fotki_album_id=req.fotki_album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()
    check_user_permissions_for_album(album, uid=req.uid)
    return {'url': album.short_url}


def fotki_album_item_public_url(req):
    resource = factory.get_resource(req.owner_uid, req.path)
    obj_id = resource.meta['file_id']
    album = Album.controller.get(uid=req.owner_uid, fotki_album_id=req.fotki_album_id)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()
    check_user_permissions_for_album(album, uid=req.uid)
    item = _get_album_item(uid=req.owner_uid, obj_id=obj_id, album_id=album.id)
    return {'url': item.build_short_url()}
