# -*- coding: utf-8 -*-

from mpfs.common.errors import ResourceNotFound, ResourceBlocked
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.public_urls import any_url_to_private_hash
from mpfs.config import settings
from mpfs.core.address import ResourceId
from mpfs.core.albums import errors as albums_errors
from mpfs.core.albums.models import Album
from mpfs.core.filesystem.base import FEATURE_TOGGLES_DISABLE_OLD_PREVIEWS
from mpfs.core.queue import mpfs_queue
from mpfs.core.user.base import User

ALBUM_MAX_DJFS_RESOURCES_BULK_SIZE = settings.album['max_djfs_resources_bulk_size']


def get_album_by_public_key(public_key):
    """
    Возвращает опубликованный альбом.

    :exception: `ResourceNotFound` если альбома с таким публичным ключём нет.

    :param public_key: Публичный ключ альбома.

    :rtype: Album
    """
    album = get_album_by_public_key_unchecked(public_key)
    if not album:
        raise albums_errors.PhotoAlbumNotFoundError()
    return album


def get_album_by_public_key_or_url(public_key_or_url):
    public_key = any_url_to_private_hash(public_key_or_url)
    return get_album_by_public_key(public_key)


def get_public_album(public_key, req_uid=None):
    """
    Возвращает опубликованный альбом, если у пользователя есть доступ к нему или выбрасывает исключение.

    :exception: `ResourceNotFound` если альбома с таким публичным ключём нет или если альбом заблокирован.
    :exception: `albums_errors.AlbumsIsNotPublicError`

    :param str public_key: Публичный ключ альбома.
    :param str req_uid: UID пользователя от имени которого происходит доступ к альбому.

    :rtype: Album
    """
    album = get_album_by_public_key(public_key)
    check_user_permissions_for_album(album, uid=req_uid)
    return album


def check_user_permissions_for_album(album, uid):
    """Проверяет доступ пользователя к альбому."""
    user = User(album.uid)

    # https://st.yandex-team.ru/CHEMODAN-36240
    # https://st.yandex-team.ru/CHEMODAN-42695
    if user.is_missing_or_blocked_in_passport():
        raise ResourceNotFound()

    # владелец должен видеть даже заблокированный альбом
    if uid != album.uid and (album.is_blocked or user.is_blocked()):
        raise ResourceNotFound()

    if album.uid != uid and not album.is_public:
        # если пришёл не владелец и альбом не публичный, то отдаём ошибку
        raise albums_errors.AlbumsIsNotPublicError()


def get_album_by_public_key_unchecked(public_key):
    """Возвращает альбом по публичному ключу.

    Если альбом не найден, то вернет None.
    """
    owner_uid, album_id = Album.parse_public_key(public_key)
    if owner_uid is not None and album_id is not None:
        return Album.controller.get(uid=owner_uid, id=album_id)


def get_resource_from_album_item(album, item_id, request_uid=None):
    """Возвращает ресурс альбому и ID элемента в альбоме.

    Производит проверки доступа пользователя с `request_uid` к альбому и элементу альбома.
    """
    check_user_permissions_for_album(album, uid=request_uid)

    item = album.items.get(id=item_id)
    if not item:
        raise ResourceNotFound()
    if request_uid != album.uid and item.object.is_blocked():
        raise ResourceBlocked()

    return item.object


def send_djfs_albums_callback(uid, root_resource):
    """
    Отправляем callback в djfs для гео и личных альбомов
    :param uid: uid пользователя
    :param root_resource: ресурс
    :param method: метод от которого
    :return:
    """
    if not experiment_manager.is_feature_active('djfs_albums_callbacks'):
        return
    resources_buffer = list()

    # Тут получаем закешированное(!) поддерево ресурсов.
    all_children = root_resource.get_full_index()

    for _, resource_data in all_children.iteritems():
        file_id = resource_data['meta'].get('file_id')
        if not file_id:
            continue
        resource_id = ResourceId(uid, file_id).serialize()
        media_type = resource_data.get('media_type')
        has_preview = (resource_data['meta'].get('pmid') or
                       resource_data['meta'].get('previews') or
                       resource_data['meta'].get('screenshot'))

        if resource_data['meta'].get('ext_coordinates'):
            resources_buffer.append(resource_id)
        elif media_type == 'video' or (media_type == 'image' and has_preview):
            resources_buffer.append(resource_id)
        else:
            continue

        if len(resources_buffer) == ALBUM_MAX_DJFS_RESOURCES_BULK_SIZE:
            mpfs_queue.put({'uid': uid, 'resource_ids': resources_buffer}, 'file_changed_operation')
            resources_buffer = list()

    if len(resources_buffer):
        mpfs_queue.put({'uid': uid, 'resource_ids': resources_buffer}, 'file_changed_operation')


def borders_is_valid(left, right, top, bottom):
    return left >= 0 and top >= 0 and right <= 1 and bottom <= 1 and left < right and top < bottom


def shift_borders(border_min, border_max):
    need_resize = False
    if border_max - border_min >= 1:
        need_resize = True
        border_min, border_max = 0, 1.0
    elif border_min < 0:
        border_min, border_max = 0, border_max - border_min
    elif border_max > 1:
        border_max, border_min = 1.0, border_min - (border_max - 1)
    return border_min, border_max, need_resize


def get_new_horizontal_borders(y_rel, face_height, top_offset, bottom_offset):
    top = y_rel - top_offset
    bottom = y_rel + face_height + bottom_offset
    return top, bottom


def get_new_vertical_borders(x_rel, face_width, avatar_width_relative):
    width_offset = (avatar_width_relative - face_width) / 2.0
    left = x_rel - width_offset
    right = x_rel + face_width + width_offset
    return left, right


def get_album_item_face_coordinates(item):
    """
        ЕСЛИ ПРАВИШЬ ТУТ ПОПРАВЬ И В DJFS_ALBUMS
        Брат близнец для.
        https://a.yandex-team.ru/arc/trunk/arcadia/disk/djfs/core/src/main/java/ru/yandex/chemodan/app/djfs/core/album/AlbumItemFaceCoordinates.java#L28

        Суть алгоритма:
            1. Определяем сторону с которой начнем манипуляции с размерами рамки, если она не влезает в границы изображения;
            2. Определяем размеры рамки на основе координат лица;
            3. Ресайзим и сдвигаем ширину/высоту рамки, если они вышли за границы изображения;
    """

    face_info = item.face_info
    if not face_info:
        return
    try:
        x_rel = float(face_info['face_coord_x'])
        y_rel = float(face_info['face_coord_y'])
        face_width = float(face_info['face_width'])
        face_height = float(face_info['face_height'])
        photo_width_px = int(face_info['width'])
        photo_height_px = int(face_info['height'])
    except KeyError:
        return

    # Магические константы
    top_offset_rel = 0.35
    bottom_offset_rel = 0.1
    vert_offset_rel = top_offset_rel + bottom_offset_rel
    width_to_height = 3.0 / 4.0
    height_to_width = 4.0 / 3.0
    photo_height_to_width = float(photo_height_px) / photo_width_px
    photo_width_to_height = float(photo_width_px) / photo_height_px

    resize_width_first = False
    if (photo_width_px / 4.0) < (photo_height_px / 3.0):
        avatar_max_height_rel = photo_width_to_height * width_to_height
        resize_width_first = True
    else:
        avatar_max_width_rel = photo_height_to_width * height_to_width

    top_offset, bottom_offset = top_offset_rel * face_height, bottom_offset_rel * face_height
    top, bottom = get_new_horizontal_borders(y_rel, face_height, top_offset, bottom_offset)
    avatar_height_px = (bottom - top) * photo_height_px
    avatar_width_px = height_to_width * avatar_height_px
    avatar_width_relative = avatar_width_px / photo_width_px
    left, right = get_new_vertical_borders(x_rel, face_width, avatar_width_relative)

    if resize_width_first:
        left, right, need_resize = shift_borders(left, right)
        if need_resize:
            height_offset = avatar_max_height_rel - face_height
            bottom_offset = height_offset * bottom_offset_rel / vert_offset_rel
            top_offset = height_offset - bottom_offset
            if height_offset < 0:
                bottom_offset, top_offset = top_offset, bottom_offset
            top, bottom = get_new_horizontal_borders(y_rel, face_height, top_offset, bottom_offset)
        top, bottom, _ = shift_borders(top, bottom)
    else:
        top, bottom, need_resize = shift_borders(top, bottom)
        if need_resize:
            left, right = get_new_vertical_borders(x_rel, face_width, avatar_max_width_rel)
        left, right, _ = shift_borders(left, right)

    return {
        "face_left": left,
        "face_top": top,
        "face_right": right,
        "face_bottom": bottom,
        "width": photo_width_px,
        "height": photo_height_px,
    }
