# -*- coding: utf-8 -*-
import hashlib
import time
import urllib

from mpfs.config import settings
from mpfs.common.errors import DecryptionError, ZeroUpdateControllerMPFSError, KladunNoResponse
from mpfs.common.util.crypt import CryptAgent
from mpfs.core.albums import errors
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.core.queue import mpfs_queue
from mpfs.core.services.previewer_service import Previewer
from mpfs.core.services.mulca_service import Mulca
from mpfs.core.user.base import User
from mpfs.core.filesystem.resources.disk import DiskFile, DiskFolder, MPFSFile
from mpfs.core.albums.errors import AlbumsUnableToUseAlbumAsCoverError, AlbumsUnableToUseFolderAsCoverError, \
    AlbumsWrongLayoutError, AlbumsUnableToPublishUnsavedError, AlbumsCoverOffsetWrongTypeError, \
    AlbumsUpdateSocialCoverError, AlbumTitleTooLong
from mpfs.core.models import Model, ObjectIdField, ModelField, EmbeddedModelField
from mpfs.core.services.clck_service import clck
from mpfs.core.services.logreader_service import LogreaderService
from mpfs.core.services.zaberun_service import Zaberun
from mpfs.engine.process import get_default_log, get_error_log
from mpfs.core.albums.events import AlbumPostToSocialEvent
from mpfs.core.services.socialproxy_service import SocialProxy
from mpfs.metastorage.postgres.exceptions import UniqueConstraintViolationError, DatabaseConstraintError
from mpfs.metastorage.postgres.schema import AlbumType, ItemsSortingMethod

default_log = get_default_log()
error_log = get_error_log()
logreader = LogreaderService()
ALBUM_MAX_TITLE_LENGTH = settings.album['max_title_length']


class Album(Model):
    LAYOUT_ROWS = 'rows'
    LAYOUT_SQUARES = 'squares'
    LAYOUT_WATERFALL = 'waterfall'
    LAYOUT_FIT_WIDTH = 'fit_width'

    LAYOUT_CHOICES = [LAYOUT_ROWS, LAYOUT_SQUARES, LAYOUT_WATERFALL, LAYOUT_FIT_WIDTH]
    primary_key_field = 'id'
    GENERATED_PUBLIC_FIELDS = ('public_url', 'public_key', 'short_url')
    SOCIAL_COVER_URL_SIZE = '1200x630'
    UPDATE_SOCIAL_COVER_TIMEOUT = 19
    """Таймаут генерации превью в секундах.

    Нужен для быстрой обратоки сохранения изменений альбома.
    И при необходимости ставить асинхронную задачу для генерации обложки.
    """

    id = ObjectIdField(source='_id')
    uid = ModelField(required=True)
    title = ModelField()
    description = ModelField()
    _cover = EmbeddedModelField(lambda: AlbumItem, source='cover')
    cover_offset_y = ModelField()
    is_public = ModelField()
    public_key = ModelField()
    public_url = ModelField()
    short_url = ModelField()
    layout = ModelField()
    flags = ModelField()
    mtime = ModelField()
    ctime = ModelField()
    social_cover_stid = ModelField() # stid обложки для соц. сетей
    is_blocked = ModelField()  # у альбомов ни разу не блокировавшихся этого атрибута нет
    block_reason = ModelField()  # у альбомов ни разу не блокировавшихся этого атрибута нет

    # https://st.yandex-team.ru/CHEMODAN-39011
    fotki_album_id = ModelField()  # уникальный идентификатор альбома из Я.Фотки (Int32)

    album_type = ModelField()
    album_items_sorting = ModelField()
    is_desc_sorting = ModelField()
    _PUBLIC_KEY_SEPARATOR = ':'
    _crypt_agent = CryptAgent()
    """Используется для вычисления публичного ключа альбома и для вычисления uid и album_id по публичному ключу."""

    def __init__(self, **kwargs):

        if kwargs.get('fotki_album_id'):
            kwargs['fotki_album_id'] = int(kwargs['fotki_album_id'])
        super(Album, self).__init__(**kwargs)
        self.album_type = kwargs.get('album_type')
        self.items_count = None
        self._public_views_count = None
        self.force_sync_update_social_cover = False
        if not self.pk:
            if self.album_type == AlbumType.PERSONAL.value or self.album_type is None:
                self.album_items_sorting = ItemsSortingMethod.BY_DATE_CREATED
                self.is_desc_sorting = False
            elif self.album_type == AlbumType.FAVORITES.value:
                self.album_items_sorting = ItemsSortingMethod.BY_DATE_CREATED
                self.is_desc_sorting = True

    @classmethod
    def from_db(cls, db_object):
        obj = super(Album, cls).from_db(db_object)
        # https://st.yandex-team.ru/CHEMODAN-26072
        for attr_name in cls.GENERATED_PUBLIC_FIELDS:
            if not getattr(obj, attr_name):
                obj.save()
                break
        return obj

    def has_cover(self):
        return self.cover and self.cover.object

    def ensure_cover(self, itemset_changed=False):
        if (
            (self.album_type == AlbumType.FAVORITES.value and itemset_changed) or
            (not self.has_cover() and self.album_type != AlbumType.GEO.value)
        ):
            self.select_new_cover()

    @property
    def social_cover_url(self):
        """Публичная ссылка на спец. превью для социальных сетей"""
        if self.is_public and self.has_cover() and self.social_cover_stid:
            mime_type = self.cover.object.get_preview_mime()
            return Zaberun().generate_preview_url('0', self.social_cover_stid, self.cover.object.name,
                                               inline=True, eternal=True, content_type=mime_type, size=self.SOCIAL_COVER_URL_SIZE)
        return None

    def select_new_cover(self):
        items = self.bulk_load_items(amount=40,
                                     only_public_items=False,
                                     obj_type=[AlbumItem.RESOURCE, AlbumItem.SHARED_RESOURCE])
        self.cover_offset_y = 0

        if items:
            try:
                self.cover = items[0]
                self.save()
            except DatabaseConstraintError:
                error_log.exception("Unable to set album cover for uid %s album %s" % (self.uid, self.id))
        else:
            self.cover = None

    @property
    def cover(self):
        if self._cover is not None:
            self._cover.album_id = self.id
            self._cover.album = self
        return self._cover

    @cover.setter
    def cover(self, item):
        self._cover = item
        if self._cover is not None:
            self._cover.album_id = self.id
            self._cover.album = self

    @property
    def user(self):
        if not hasattr(self, '_user'):
            setattr(self, '_user', User(self.uid))
        return getattr(self, '_user', None)

    @property
    def items(self):
        if hasattr(self, '_items'):
            return self._items
        return AlbumItem.controller.filter(uid=self.uid, album_id=self.id)

    @items.setter
    def items(self, items):
        items_controller = self.items
        items_controller.setup_result_cache(items)
        self._items = items_controller

    def validate(self):
        super(Album, self).validate()
        # проверяем ковёр только если он был изменён, чтоб лишний раз не доставать ресурс из БД
        if self.cover and '_cover' in self.changed_fields:
            if self.cover.obj_type not in (AlbumItem.RESOURCE, AlbumItem.SHARED_RESOURCE):
                raise AlbumsUnableToUseAlbumAsCoverError()
            if self.cover.object.type != 'file':
                raise AlbumsUnableToUseFolderAsCoverError()

        if self.layout is not None and self.layout not in self.LAYOUT_CHOICES:
            raise AlbumsWrongLayoutError()

        if not isinstance(self.cover_offset_y, (int, long, float, type(None))):
            raise AlbumsCoverOffsetWrongTypeError()

        if len(self.title) > ALBUM_MAX_TITLE_LENGTH:
            raise AlbumTitleTooLong()

    @classmethod
    def build_public_key(cls, uid, album_id):
        """
        Формирует публичный ключ альбома по uid владельца и album_id.

        :return: Публичный ключ альбома.
        :rtype: str
        """
        key = cls._PUBLIC_KEY_SEPARATOR.join([uid, album_id])
        public_key = cls._crypt_agent.encrypt(key)
        return public_key

    @classmethod
    def parse_public_key(cls, public_key):
        """
        Вытаскивает из публичного ключа альбома uid владельца и album_id.

        Если вытащить uid и album_id по каким-то причинам не удалось, то возвращает (None, None).

        :param str public_key: Публичный ключ альбома.
        :return: uid владельца и album_id в виде (<uid>, <album_id>)
        :rtype: tuple
        """
        try:
            key = cls._crypt_agent.decrypt(public_key)
            uid, album_id = key.split(cls._PUBLIC_KEY_SEPARATOR, 1)
        except (ValueError, DecryptionError):
            uid = album_id = None
        return uid, album_id

    @staticmethod
    def build_public_url(public_key):
        return settings.system['public_urls']['album'] % (urllib.quote(public_key, safe=''),)

    @classmethod
    def build_short_url(cls, public_url):
        key, link = clck.generate(public_url, mode='album')
        return link

    def publish(self):
        """Опубликовать альбом."""
        if not self.uid or not self.id:
            raise AlbumsUnableToPublishUnsavedError()
        if self.album_type == AlbumType.FAVORITES.value:
            raise AlbumsUnableToPublishUnsavedError("Favorites album can't be published")
        self.is_public = True
        self.save()

    def unpublish(self):
        """Снять альбом с публикации."""
        self.is_public = False
        self.save()

    def get_public_views_count(self, load_info_from_counters=True):
        if self._public_views_count is None:
            self._public_views_count = 0
            if self.public_key and load_info_from_counters:
                self._public_views_count = logreader.get_one_counter(self.public_key)

        return self._public_views_count

    def as_json_dict(self, load_info_from_counters=True):
        result = super(Album, self).as_json_dict()

        # Эмулируем старое поведение, когда публичные признаки хранились в отдельной модели PublicAlbum:
        # вставляем объект public и перемещаем в него все публичные признаки из объекта альбома.
        public = {
            'public_key': self.public_key,
            'public_url': self.public_url,
            'short_url': self.short_url,
            'views_count': self.get_public_views_count(load_info_from_counters=load_info_from_counters),
        }
        # удаляем публичные атрибуты из тела альбома, чтоб не вводить ни кого в заблуждение
        for k in public.keys():
            result.pop(k, None)

        # фильтруем отсутствующие значения
        public = dict([(k, v) for k, v in public.iteritems() if v is not None])

        result['public'] = public
        result['user'] = self.user.public_info()
        result['cover'] = result.pop('_cover', None)
        social_cover_url = self.social_cover_url
        if social_cover_url:
            result['social_cover_url'] = social_cover_url
        if not self.has_cover():
            result['cover'] = None
        if self.items_count is not None:
            result['items_count'] = self.items_count

        return result

    def copy(self):
        items = list(AlbumItem.controller.filter(uid=self.uid, album_id=self.id).order_by('order_index'))
        album = Album(**self._instance_data)
        album.pk = None
        album.public_key = None
        album.public_url = None
        album.short_url = None
        album.social_cover_stid = None
        album.save()
        if album.cover:
            album.cover.album_id = album.id
        album.save(update_fields=['cover'])

        if album.is_public:
            album.publish()

        if items:
            for index, item in enumerate(items):
                item.pk = None
                item.album_id = album.id
            AlbumItem.controller.bulk_create(items)
        return album

    def delete(self):
        """Удаляет альбом со всеми зависимостями."""
        AlbumItem.controller.filter(uid=self.uid, album_id=self.id).delete()
        super(Album, self).delete()

    def get_last_item_index(self):
        """Вернёт индекс последнего элемента в альбоме или -1, если элементов нет."""
        last_items = list(AlbumItem.controller.filter(uid=self.uid, album_id=self.id).order_by('-order_index')[:1])
        if last_items:
            last_item = last_items[0]
            return last_item.order_index
        else:
            return -1

    def save(self, *args, **kwargs):
        now = time.time()
        if not self.ctime:
            self.ctime = now
        if self.album_type != AlbumType.GEO.value:
            # CHEMODAN-70609: Для геоальбомов меняем mtime только в djfs
            self.mtime = now

        if self.layout is None:
            self.layout = self.LAYOUT_CHOICES[0]

        if self.id and self.uid:
            new_public_key = self.build_public_key(self.uid, self.id)
            have_all_public_fields = all([getattr(self, a) for a in self.GENERATED_PUBLIC_FIELDS])
            if self.public_key != new_public_key or not have_all_public_fields:
                self.public_key = self.build_public_key(self.uid, self.id)
                self.public_url = self.build_public_url(self.public_key)
                self.short_url = self.build_short_url(self.public_url)

        # готовим данные для генерации соц. обложки до сохранения альбома в базёнку,
        # т.к. после сохранения не понятно будет какие поля обновились и новый альбом или обновился старый
        update_fields = self.get_update_fields(*args, **kwargs)
        update_social_cover = update_fields is None or bool({'_cover', 'title', 'uid'} & set(update_fields))
        is_new = self.is_new(*args, **kwargs)

        super(Album, self).save(*args, **kwargs)

        # если альбом новый, то пытаемся сгенерить соц. обложку синхронно
        if is_new or self.force_sync_update_social_cover:
            try:
                self.force_sync_update_social_cover = False
                self.update_social_cover(old_social_cover_stid=self.social_cover_stid,
                                         generate_cover_timeout=self.UPDATE_SOCIAL_COVER_TIMEOUT,
                                         retry=False)
            except AlbumsUpdateSocialCoverError as e:
                default_log.info(str(e))
            except KladunNoResponse:
                default_log.info(
                    'Failed to generate cover in %s seconds: create async task' % self.UPDATE_SOCIAL_COVER_TIMEOUT
                )
                data = {
                    'uid': self.uid,
                    'album_id': self.id,
                    'old_social_cover_stid': self.social_cover_stid,
                }
                mpfs_queue.put(data, 'albums_update_social_cover')
        # если альбом не новый, но обложка обновлена, то ставим асинхронную задачу обновить обложку
        elif update_social_cover:
            data = {
                'uid': self.uid,
                'album_id': self.id,
                'old_social_cover_stid': self.social_cover_stid,
            }
            mpfs_queue.put(data, 'albums_update_social_cover')

    def __str__(self):
        return '%s(uid=%s, id=%s)' % (self.__class__.__name__, self.uid, self.id)

    def update_social_cover(self, old_social_cover_stid=None, generate_cover_timeout=None, retry=True):
        """
        Обновляет stid социальной обложки альбома.

        Старый stid удаляет из мульки или помещает в deleted_stids.

        :raise AlbumsUpdateSocialCoverError: Если old_social_cover_stid != social_cover_stid.

        :param old_social_cover_stid: Старый stid соц. обложки, для зщиты от конкуретного обновления.
        """
        # если кто-то уже успел сменить обложку до того как мы добрались до неё
        if self.social_cover_stid != old_social_cover_stid:
            msg = 'Unable to update social cover. Reason: %s current social_cover_stid %s != %s' % (
                self, self.social_cover_stid, old_social_cover_stid)
            raise AlbumsUpdateSocialCoverError(msg)

        if (self.cover and self.cover.obj_type in [AlbumItem.RESOURCE, AlbumItem.SHARED_RESOURCE]
            and self.cover.object and hasattr(self.cover.object, 'meta') and self.cover.object.meta.get('pmid', None)):

            # генерим новую обложку и записывем её stid в альбом
            self.social_cover_stid = Previewer().generate_album_preview(
                self.cover.object.meta['pmid'],
                self.user.public_info()['login'],
                self.title,
                timeout=generate_cover_timeout,
                retry=retry,
            )
        else:
            # иначе считаем, что у альбома нет обложки и нужно обнулить у альбома атрибут.
            default_log.debug('%s have no image cover. Set social_cover_stid to None' % self)
            self.social_cover_stid = None

        stids_to_remove = []
        if old_social_cover_stid:
            stids_to_remove.append(old_social_cover_stid)

        try:
            # в любом случае, если обновили обложку, то сохраняем альбом
            self.save()
        except ZeroUpdateControllerMPFSError:
            # при сохранении не нашли альбом. Удаляем новую обложку
            if self.social_cover_stid:
                stids_to_remove.append(self.social_cover_stid)

        # после того, как старый стид гарантированно уже ни где не значится -- дропаем его из стораджа
        for stid in stids_to_remove:
            try:
                Mulca().remove(stid)
            except Mulca.api_error as e:
                # если не получилось, то пишем ошибку в error_log и добавляем стид в deleted_stids,
                # чтоб его потом почистила файловая чистка мульки
                error_log.error(str(e))
                DeletedStid.controller.bulk_create(
                    [DeletedStid(stid=stid, stid_source=DeletedStidSources.UPDATE_SOCIAL_COVER)]
                )

    def bulk_load_items(self, last_item_order_index=None, amount=None, only_public_items=False, obj_type=None):
        """
        Возвращает элементы публичного альбома, пригодные для отображения посторонним.

        Пагинация осуществляется на основе параметров last_item_id и amount.

        :param last_item_order_index: order_index последнего элемента альбома полученного на предыдущей странице.
        :param amount: Количество элементов, которое должна содержать страница.
        :param only_public_items: Признак выгрузки только публичных элементов.
        :param obj_type: фильтрация по obj_type. Допустимо передавать список или строку.
        """
        result = []
        if amount == 0:
            return result

        # Возвращать меньше элементов, чем запрошено не консистентно
        # до тех пор пока мы не достигли конца списка элементов.
        # Поэтому если часть элементов была отброшена из-за недостаточности прав владельца альбома,
        # то добираем недостающие элементы дополнительными запросами в БД.
        album_items = self.items

        if last_item_order_index:
            album_items = album_items.filter(
                order_index={'$lt': last_item_order_index} if self.is_desc_sorting else {'$gt': last_item_order_index}
            )

        if isinstance(obj_type, basestring):
            album_items = album_items.filter(obj_type=obj_type)
        elif isinstance(obj_type, list):
            album_items = album_items.filter(obj_type={'$in': obj_type})

        iter_count = 0
        album_items = album_items.order_by('-order_index' if self.is_desc_sorting else 'order_index')[:amount]
        while len(album_items) > 0 and (amount is None or len(result) < amount) and iter_count < len(self.items):
            # идём в БД и загружаем элементы вместе с объектами
            loaded_items = AlbumItem.controller.bulk_load_objects_for_items(album_items)

            # фильтруем заблокированные по HID'у файлы https://st.yandex-team.ru/CHEMODAN-25146
            if only_public_items:
                loaded_items = AlbumItem.controller.filter_blocked(loaded_items)

            # если последний запрос вернул элементы и лимит либо не установлен,
            # либо меньше количества отобранных элементов,
            for item in loaded_items:
                if not only_public_items or item.is_public:
                    result.append(item)
                    if len(result) == amount:
                        break

            cur_last_item_order_index = list(album_items)[-1].order_index
            iter_count += len(album_items)

            # обновляем фильтр элементов, так чтобы в него попали элементы следующие сразу за последним обработанным
            # и загружаем эти элементы вместе с объектами
            album_items = album_items.filter(order_index={'$gt': cur_last_item_order_index})

        for album_item in result:
            album_item.album = self
            if isinstance(album_item.object, MPFSFile):
                album_item.object.print_to_listing_log()

        return result

    def send_social_wall_post(self, provider):
        resp = SocialProxy().wall_post(self.uid, provider, self.short_url)

        if resp['state'] == 'success':
            AlbumPostToSocialEvent(album=self, provider=provider).send()

        return resp

    def _update(self, update_fields, upsert):
        data = self.as_dict()
        if update_fields:
            data = dict([(k, data[k]) for k in update_fields if k in data])
        self.controller.filter(**self._get_object_query()).update(upsert=upsert, **data)

    def _save(self, force_insert=False, update_fields=None, upsert=False):
        """
        Внутренний метод сохранения. Выполняет самые общие действия. Поэтому переопределять его не стоит.
        """
        if self.controller is None:
            raise AttributeError('Unable to save model without controller.')
        # если у объекта нет первичного ключа или явно указано, что нужно добавить новый объект,
        # то добавляем объект в БД
        if self.is_new(force_insert=force_insert) and not upsert:
            if self.album_type == AlbumType.FAVORITES.value:
                self.pk = hashlib.md5("%s_%s" % (self.uid, self.album_type)).hexdigest()[:24]
            try:
                self.pk = self.controller._insert(self.as_db_object())
            except UniqueConstraintViolationError:
                raise errors.AlbumAlreadyExists(extra_msg='%s album id: %s'%(self.album_type, self.pk),
                                                data={'album_id': self.pk})
        else:
            self._update(update_fields, upsert)


class AlbumItem(Model):
    is_sharded = True
    primary_key_field = 'id'

    ALBUM = 'album'
    RESOURCE = 'resource'
    SHARED_RESOURCE = 'shared_resource'

    id = ObjectIdField(source='_id')
    uid = ModelField(required=True)
    album_id = ObjectIdField(required=True)
    order_index = ModelField(required=True)
    obj_type = ModelField(required=True)
    obj_id = ModelField(required=True)
    group_id = ModelField()
    description = ModelField()
    date_created = ModelField()
    face_info = ModelField()

    @property
    def object(self):
        """
        Возвращает ресурс или альбом на который ссылается элемент.

        :rtype: mpfs.core.filesystem.resources.disk.DiskFile | mpfs.core.filesystem.resources.disk.DiskFolder | Album
        | mpfs.core.filesystem.resources.share.SharedFolder | mpfs.core.filesystem.resources.share.SharedFile
        :
        """
        if not hasattr(self, '_object'):
            if self.obj_type == self.ALBUM:
                self.object = Album.controller.get(uid=self.uid, id=self.obj_id)
            else:
                self.object = self.get_resource()
        return getattr(self, '_object', None)

    @property
    def is_shared(self):
        return self.obj_type == self.SHARED_RESOURCE

    @object.setter
    def object(self, value):
        setattr(self, '_object', value)

    @classmethod
    def get_shared_resource_owner_uid(cls, uid, group_id):
        from mpfs.core.social.share.link import LinkToGroup
        from mpfs.core.social.share.group import Group
        from mpfs.common.errors.share import ShareNotFound, GroupNotFound
        try:
            LinkToGroup.load(uid=uid, gid=group_id)
            group = Group.load(gid=group_id)
            return group.owner
        except (ShareNotFound, GroupNotFound):
            # если доступа нет, то получить объект мы не можем
            return None

    def get_resource(self):
        album_items = AlbumItem.controller.bulk_load_objects_for_items([self])
        if album_items:
            return album_items[0].object
        return None

    def as_json_dict(self):
        ret = super(AlbumItem, self).as_json_dict()
        if isinstance(self.object, Album):
            ret['object'] = self.object.as_json_dict()
        elif isinstance(self.object, (DiskFile, DiskFolder)):
            ret['object'] = self.object.dict()
        return ret

    @property
    def is_available(self):
        """Доступен ли элемент пользователю."""
        return bool(self.object)

    @property
    def is_public(self):
        """Отображается ли элемент в публичном альбоме."""
        # убрали функционал ограничения доступа к элементам публичного альбома если они из ОП с правами RO
        # https://st.yandex-team.ru/CHEMODAN-70041
        return self.is_available

    @property
    def album(self):
        if not hasattr(self, '_album'):
            album = Album.controller.get(uid=self.uid, id=self.album_id)
            self._album = album
        return getattr(self, '_album', None)

    @album.setter
    def album(self, value):
        if isinstance(value, Album):
            setattr(self, '_album', value)
        else:
            raise TypeError("'Album' object required")

    def delete(self):
        """Удаляет элемент, а если он cover, то и обложку"""
        cover = getattr(self.album, 'cover', None)
        if (cover is not None and
            (self.id == cover.id or
             (cover.id is None and cover.obj_id == self.obj_id))):
            self.album.cover = None
        self.album.save()
        super(AlbumItem, self).delete()

    def build_short_url(self):
        return "%s/%s" % (self.album.short_url, self.id)

