# -*- coding: utf-8 -*-
"""

MPFS
CORE

Ядро системы

"""
import datetime
import traceback
import os
import random
import itertools
import time

import mpfs.engine.process
from mpfs.common import errors
from mpfs.common.errors import (
    APIError,
    BadRequestError,
    NotFound,
    ResourceNotFound,
    PassportUnsupportedEventTypeError,
    UserAlreadyReindexedForQuickMove, UserCannotBeReindexedForQuickMove)
from mpfs.common.errors import share as share_errors
from mpfs.common.static.tags.experiment_names import ATTACH_UPLOAD_TO_DISK_QUOTA
from mpfs.common.util.crypt import CryptAgent
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.common.util.user_agent_parser import UserAgentParser
from mpfs.common.util.video_unlim import user_unlim_experiment_data, get_user_country, RKUB
from mpfs.common.util.ycrid_parser import YcridParser, YcridPlatformPrefix

from mpfs.config import settings
from mpfs.core.albums.errors import AlbumItemMissed
from mpfs.core.albums.logic.common import (
    get_resource_from_album_item,
    get_album_by_public_key_unchecked,
)
from mpfs.core.albums.models import Album
from mpfs.core.db_rps_limiter import HANDLE_SHARE_INVITE_USER_RL
from mpfs.core.filesystem.cleaner.worker import DbChecker, CheckingStid
from mpfs.core.filesystem.dao.folder import FolderDAO
from mpfs.core.filesystem.resources.group import GroupResource
from mpfs.core.office.util import get_editor, ONLY_OFFICE_FEATURE, ONLINE_EDITOR_FEATURE
from mpfs.core.operations.base import SaveOnDiskOperation
from mpfs.core.photoslice.albums.data_structures import PhotosliceAlbumTypeDataStructure
from mpfs.core.services.attach_service import Attach
from mpfs.core.services.index_service import SearchIndexer
from mpfs.core.services.orchestrator_service import orchestrator_service
from mpfs.core.services.smartcache_service import smartcache
from mpfs.core.services.zaberun_service import Zaberun
from mpfs.core.services.kladun_service import Kladun
from mpfs.core.user import base as user
from mpfs.core.user.dao.user import UserDAO
from mpfs.core.queue import mpfs_queue
from mpfs.core import social
from mpfs.core.bus import Bus
from mpfs.core import factory
from mpfs.common.util import from_json, filter_uid_by_percentage, safe_to_int, type_to_bool, to_json
from mpfs.common.util.filetypes import getGroupNumber, getMediaTypeByName
from mpfs.common.static.tags import LENTA_BLOCK_ID, COMMIT_FILE_INFO, COMMIT_FILE_UPLOAD, COMMIT_FINAL
from mpfs.core.comments.interface import (
    comments_permissions,
    comments_owner,
    info_by_comment_id,
)
from mpfs.core.deletion_from_local_device.interface import can_delete
from mpfs.core.operations import manager
from mpfs.core.filesystem import hardlinks
from mpfs.core.user import constants, invites
from mpfs.core.user.base import User
from mpfs.core.metastorage.decorators import *
from mpfs.core.metastorage.decorators import _check_user_exists_and_not_blocked
from mpfs.core.services.disk_service import Disk
from mpfs.core.services.search_service import SearchDB
from mpfs.core.services.passport_service import Passport
from mpfs.core.services.mail_service import MailTVM, MailObject
from mpfs.common.static.tags import PROVIDER_DIR
from mpfs.core.event_history.logger import (
    CatchHistoryLogging,
    SAVE_FILE_FROM_MAIL_LOG_MESSAGE_RE, EXTRACT_FILE_FROM_ARCHIVE_LOG_MESSAGE_RE,
    SAVE_FILE_FROM_WEB_LOG_MESSAGE_RE
)
from mpfs.core.social.share import ShareProcessor
from mpfs.core.social.share.group import Group
from mpfs.core.filesystem.symlinks import Symlink
from mpfs.core.social.publicator import Publicator
from mpfs.core.billing.processing import pushthelimits
from mpfs.core.rostelecom_unlim.interface import (
    rostelecom_activate,
    rostelecom_deactivate,
    rostelecom_freeze,
    rostelecom_unfreeze,
    rostelecom_list_services,
)
from mpfs.core.filesystem.indexer import DiskDataIndexer
from mpfs.core.filesystem.resources.share import SharedResource
from mpfs.core.filesystem.events import FilesystemStoreEvent
from mpfs.core.lenta.job_handlers import lenta_delete_empty_block
from mpfs.core.lenta.utils import EventLogMessagesTransformer
from mpfs.core.operations.filesystem.copy import CopyAviaryDisk, CopyMailDisk, CopyWebDisk
from mpfs.core.operations.filesystem.store import ExtractFileFromArchive
from mpfs.core.organizations.interface import organization_event
from mpfs.core.address import ResourceId, PublicAddress, LentaResourceId, Address
from mpfs.core.filesystem.resources.disk import MPFSFolder, MPFSFile, append_meta_to_office_files
from mpfs.core.services.rate_limiter_service import rate_limiter
from mpfs.core.services.lenta_loader_service import lenta_loader
from mpfs.core.user.standart import StandartUser
from mpfs.core.versioning.interface import (
    versioning_get_checkpoints,
    versioning_get_folded,
    versioning_restore,
    versioning_save,
    versioning_get_version,
)
from mpfs.core.versioning.logic.version_manager import ResourceVersionManager
from mpfs.core.user.constants import (
    PUBLIC_UID,
    PHOTOUNLIM_AREA_PATH,
    ATTACH_AREA,
    ATTACH_AREA_PATH,
    DEFAULT_FOLDERS_NAMES,
    LNAROD_AREA,
    TELEMOST_FOS_BODY_TEMPLATE,
    TELEMOST_FOS_SUBJECT_TEMPLATE,
)
from mpfs.core.yateam.interface import (
    passport_user_2fa_changed,
    staff_make_yateam_admin,
    staff_reset_yateam_admin,
    staff_user_changed_callback,
)
from mpfs.core.yateam.logic import is_yateam_root, user_has_nda_rights, recreate_yateam_dir, check_yateam_subtree_access
from mpfs.core.last_files.interface import new_get_last_files
from mpfs.core.wake_up.interfaces import wake_up_push_start, wake_up_push_stop

from mpfs.core.services.video_service import VideoStreaming

# Импорты интерфейсных методов
from mpfs.core.albums.interface import (
    album_append_items, album_copy, album_get, album_item_move, album_item_remove,
    album_item_set_attr, album_publish, album_set_attr, album_unpublish,
    albums_create, albums_list, async_public_album_save, public_album_get,
    public_album_save, album_append_item, album_remove, albums_create_with_items,
    public_album_item_download_url, public_album_item_info, public_album_item_video_url,
    album_item_check, album_item_info, public_album_check, public_album_items_list,
    albums_create_from_folder, public_album_download_url, public_album_social_wall_post,
    public_album_block, public_album_unblock, async_public_album_social_wall_post,
    fotki_album_public_url, fotki_album_item_public_url, album_video_streams, albums_exclude_from_generated,
    album_find_in_favorites, public_album_search_bulk_info
)
from mpfs.core.lenta.interface import (
    lenta_block_list, lenta_list_group_invites, lenta_public_log_visit,
    lenta_create_album_from_block, lenta_block_public_link
)
from mpfs.core.office.interface import (office_store, office_action_data, office_action_check, office_download_redirect,
                                        office_lock, office_unlock, office_rename, office_hancom_lock,
                                        office_hancom_unlock, office_hancom_store, office_enable_hancom,
                                        office_disable_hancom, office_generate_online_editor_url,
                                        office_only_office_callback, office_set_editor_type, office_set_access_state,
                                        office_get_access_state, office_info_by_office_doc_short_id, office_get_file_filters,
                                        office_get_file_urls, office_set_selection_strategy, office_show_change_editor_buttons,
                                        bulk_info_by_office_online_sharing_urls, office_switch_to_onlyoffice)
from mpfs.core.notes.interface import notes_init
from mpfs.core.pushnotifier.interface import (push_subscribe, async_push_subscribe, mobile_subscribe,
                                              mobile_unsubscribe,
                                              push_unsubscribe)
from mpfs.core.snapshot.interface import snapshot, deltas, indexer_snapshot
from mpfs.core.bulk_download.interface import bulk_download_prepare, bulk_download_list, public_bulk_download_prepare
from mpfs.core.filesystem.cleaner.models import DeletedStid, DeletedStidSources
from mpfs.core.file_recovery.interface import restore_file
from mpfs.core.promo_codes.interface import promo_code_activate
from mpfs.core.locks.interface import (
    locks_set,
    locks_list,
    locks_delete,
)
from mpfs.metastorage.mongo.collections.filesystem import is_reindexed_for_quick_move, set_reindexed_for_quick_move
from mpfs.core.global_gallery.interface import (
    add_source_ids,
    check_source_id,
    check_source_ids,
    read_deletion_log,
)
from mpfs.core.inactive_users_flow.interface import (
    inactive_users_flow_add,
    inactive_users_flow_add_from_yt,
    inactive_users_flow_update
)
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

SYSTEM_ATTACH_SYS_FOLDERS = settings.system['attach_sys_folders']
MAX_FOS_TEXT_LEN = settings.feedback['max_fos_text_len']
RATE_LIMITER_GROUP_NAMES_FULL_DIFF = settings.rate_limiter['group_names']['full_diff']
RATE_LIMITER_GROUP_NAMES_VERSION_DIFF = settings.rate_limiter['group_names']['version_diff']
RATE_LIMITER_GROUP_NAMES_PATH_DIFF = settings.rate_limiter['group_names']['path_diff']
RATE_LIMITER_GROUP_NAMES_BULK_CHECK_STIDS = settings.rate_limiter['group_names']['bulk_check_stids']
RATE_LIMITER_GROUP_NAMES_LIST_WEBDAV = settings.rate_limiter['group_names']['list_webdav']
RATE_LIMITER_BULK_INFO_BY_RESOURCE_IDS_ENABLED = settings.rate_limiter['bulk_info_by_resource_ids']['enabled']
RATE_LIMITER_BULK_INFO_BY_RESOURCE_IDS_GROUP_NAME = settings.rate_limiter['bulk_info_by_resource_ids']['group_name']
FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED = settings.feature_toggles['secondary_preferred_enabled']
FEATURE_TOGGLES_ENABLE_OPTIMIZATION_FOR_LENTA_WORKER = settings.feature_toggles['enable_optimization_for_lenta_worker']
FEATURE_TOGGLES_CHECK_TRASH_APPEND_BEFORE_TRASH_DROP = settings.feature_toggles['check_trash_append_before_trash_drop']
FEATURE_TOGGLES_CORRECT_SPACE_CHECKS_FOR_SHARED_FOLDERS = \
    settings.feature_toggles['correct_space_checks_for_shared_folders']
FEATURE_NOTIFY_PHOTOSLICE_INDEX_WHEN_FILE_NOT_FOUND = \
    settings.feature_toggles['notify_photoslice_index_when_file_not_found']

SPEED_LIMITS_BY_MEDIA_TYPE = settings.speed_limits['by_media_type']
SPEED_LIMITS_BY_AUTH_METHOD = settings.speed_limits['by_auth_method']
SPEED_LIMITS_BY_UID = settings.speed_limits['by_uid']

JAVA_DJFS_API_TASKS_COPY_ENABLED = settings.java_djfs_api['tasks']['copy']['enabled']
JAVA_DJFS_API_TASKS_COPY_ENABLED_UIDS = set(settings.java_djfs_api['tasks']['copy']['enabled_uids'])
JAVA_DJFS_API_TASKS_COPY_ENABLED_UIDS_PERCENT = settings.java_djfs_api['tasks']['copy']['enabled_uids_percent']

VALID_KLADUN_CALLBACK_TYPES = {COMMIT_FILE_INFO, COMMIT_FILE_UPLOAD, COMMIT_FINAL}

VALID_UNLIMITED_AUTOUPLOAD_REASON = ['by_user', 'by_overdue', 'by_experiment']

TELEMOST_FEEDBACK_FOS_EMAILS = settings.feedback['telemost_fos_email']
FEEDBACK_FOS_TEMPLATE = settings.feedback['fos_template']
UNAUTHORIZED_USER_NAME = 'Unauthorized'
ADVERTISING_FEATURE = 'advertising_enabled'
UNLIMITED_PHOTO_AUTOUPLOADING_FEATURE = 'unlimited_photo_autouploading_allowed'
UNLIMITED_VIDEO_AUTOUPLOADING_FEATURE = 'unlimited_video_autouploading_allowed'
PUBLIC_SETTINGS_FEATURE = 'public_settings_enabled'
crypt_agent = CryptAgent()
RESOURCE_ID_SRC_SEPARATOR = ':'
PUBLIC_PASSWORD_HEADER = 'Disk-Public-Password'
PUBLIC_PASSWORD_TOKEN_HEADER = 'Disk-Public-Password-Token'


@allow_user_init_in_progress
@disallow_pdd
@allow_empty_disk_info
@CatchHistoryLogging(disable_logging=True)
def user_init(req):
    add_services = []
    if req.add_services:
        add_services = req.add_services.split(',')
    user.Create(
        req.uid,
        type='standart',
        locale=req.locale,
        noemail=req.noemail,
        shard=req.shard,
        b2b_key=req.b2b_key,
        source=req.source,
        add_services=add_services
    )

    try:
        usr = user.User(req.uid)
        ycrid = mpfs.engine.process.get_cloud_req_id()
        usr.update_activity_by_ycrid(ycrid)
    except Exception:
        error_log.exception('Cant update activity date for uid %s' % req.uid)


def can_init_user(req):
    return {'can_init': str(int(StandartUser.can_init_user(req.uid)))}


def user_set_pdd_domain(req):
    user = User(req.uid)
    passport = Passport()
    user_info = passport.userinfo(user.uid)
    if not passport.is_from_pdd(uid=user.uid):
        raise errors.UserIsNotPDDError()
    user.update_pdd_domain(user_info['pdd_domain'])
    return User(user.uid).info()


def user_make_b2b(req):
    User(req.uid).make_b2b(req.b2b_key)
    return User(req.uid).info()


def user_reset_b2b(req):
    user = User(req.uid)
    if not user.is_b2b():
        raise errors.UserIsNotB2bError()
    user.reset_b2b(req.noemail)
    return User(req.uid).info()


@allow_user_init_in_progress
@disallow_pdd
@allow_empty_disk_info
def user_check(req):
    need_init = mpfs.engine.process.usrctl().user_init_in_progress(req.uid) or user.NeedInit(req.uid)
    # Исторически сложилось, что отдаем строку "1" или "0"
    return str(int(need_init))


@allow_empty_disk_info
def user_remove(req):
    raise NotImplementedError()


@allow_empty_disk_info
def async_user_remove(req):
    return {'oid': 'turn_off_user_remove', 'type': 'user', 'at_version': '1'}

    operation = manager.create_operation(
        req.uid,
        'user',
        'remove',
        odata={
            'req_type': req.get_type(),
        },
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@slave_read_ok
@user_exists_and_not_blocked
def user_info(req):
    ycrid = mpfs.engine.process.get_cloud_req_id()
    usr = user.User(req.uid, req.project)
    # костыль для того, чтобы ПО не показывало пуш про установку мобильного приложения, если его уже установили

    try:
        usr.update_activity_for_mobile(ycrid)
    except Exception:
        error_log.exception('Cant update activity date for uid %s' % req.uid)
    return usr.info()


@slave_read_ok
def user_feature_toggles(req):
    user_features = (('versioning_extended_period', False),
                     ('antifo', True),
                     ('advertising', True),
                     ('disk_pro', False),
                     ('disk_pro_without_ps_billing', False),
                     ('online_editor', False),
                     ('priority_support', False),
                     ('desktop_folder_autosave', False),
                     ('unlimited_video_autouploading', False),
                     ('unlimited_photo_autouploading', False),
                     ('promote_mail360', True),
                     ('public_settings', False))
    try:
        user_obj = user.User(req.uid)
        if not user_obj.is_blocked():
            user_features = (('versioning_extended_period', user_obj.versioning_extended_period_enabled),
                             ('antifo', user_obj.antifo_enabled),
                             ('advertising', user_obj.advertising_enabled),
                             ('disk_pro', user_obj.disk_pro_enabled),
                             ('disk_pro_without_ps_billing', user_obj.only_disk_pro_without_ps_billing_enabled),
                             ('online_editor', bool(get_editor(user_obj))),
                             ('priority_support', user_obj.priority_support_enabled),
                             ('desktop_folder_autosave', user_obj.desktop_folder_autosave_enabled),
                             ('unlimited_video_autouploading', user_obj.is_unlimited_video_autouploading_allowed()),
                             ('unlimited_photo_autouploading', user_obj.is_unlimited_photo_autouploading_allowed()),
                             ('promote_mail360', not user_obj.has_mail360),
                             ('public_settings', bool(user_obj.get_public_settings_enabled())))
    except errors.StorageInitUser:
        pass

    result = {}
    for field_name, enabled in user_features:
        result[field_name] = {'enabled': enabled}
    return result


@slave_read_ok
@user_exists_and_not_blocked
def awaps_user_info(req):
    return user.User(req.uid).awaps_info()


@slave_read_ok
def user_invite_info(req):
    return invites.report(req.uid, req.hash)


@slave_read_ok
def user_info_attributes(req):
    attributes = Passport().userinfo_summary(uid=req.uid, fields=('is_app_password_enabled', 'is_2fa_enabled'))
    return attributes


@user_exists_and_not_blocked
def user_invite_hash(req):
    code = invites.personal_hash(req.uid)
    return code.hash


@user_exists_and_not_blocked
def user_activity_info(req):
    return user.User(req.uid).get_activity_info()


@slave_read_ok
@user_exists_and_not_blocked
@user_is_writeable
def async_user_invite_friend(req):
    operation = invites.invite_friend(req.uid, req.provider, req.address, req.info)
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@slave_read_ok
@user_exists_and_not_blocked
def user_invite_activated(req):
    return invites.list_activated(req.uid)


@slave_read_ok
@user_exists_and_not_blocked
@user_is_writeable
def user_invite_sent(req):
    return invites.list_sent(req.uid)


@user_exists_and_not_blocked
@user_is_writeable
def set_user_var(req):
    obj = user.User(req.uid, req.project)

    methods = {
        'settings': obj.set_setting,
        'states': obj.set_state,
    }
    return methods.get(req.type)(
        req.key,
        req.value,
        req.namespace
    )


@user_exists_and_not_blocked
@user_is_writeable
def remove_user_var(req):
    obj = user.User(req.uid, req.project)

    methods = {
        'settings': obj.remove_setting,
        'states': obj.remove_state,
    }
    return methods.get(req.type)(
        req.key,
        req.namespace
    )


@user_exists_and_not_blocked
@user_is_writeable
def user_install_device(req):
    return user.User(req.uid, req.project).install_device(req.type, req.id, req.info)


@user_exists_and_not_blocked
@user_is_writeable
def user_uninstall_device(req):
    return user.User(req.uid, req.project).uninstall_device(req.type, req.id)


def _set_hasfolders(base_folder):
    """
    Функция проставление признака наличия подпапок у `base_folder`, а так же у ее подпапок(children->folders)
    """
    if not isinstance(base_folder, MPFSFolder):
        return

    subfolders = [(i.visible_address.path, i) for i in base_folder.children_items['folders']]
    if not subfolders:
        base_folder.hasfolders = 0
        return

    # сортируем список по path (по первому элементу в tuple) и берем итератор
    subfolders.sort(key=lambda i: i[0])
    subfolders_iterator = iter(subfolders)

    base_folder.hasfolders = 1
    base_uid = base_folder.visible_address.uid
    base_path = base_folder.visible_address.path

    subdirs_iterator = SearchDB().list_subresources_iterator(
        base_uid,
        base_path,
        amount=None,  # выгребаем все
        offset=0,  # с самого начала
        resource_types=['dir'],
        sort_field='path',
        asc_order=True,
        fields=['path'],
    )

    search_pathes_iterator = itertools.imap(lambda x: x['path'], subdirs_iterator)
    try:
        search_path = next(search_pathes_iterator)
        for subfolder_path, subfolder_item in subfolders_iterator:
            child_count = 0
            while search_path.startswith(subfolder_path):  # считаем сколько папок из поиска является дочерними
                if search_path != subfolder_path:
                    # пропускаем папку саму по себе (но не всегда делаем это, т.к. поиск может нам ее не вернуть)
                    child_count += 1
                search_path = next(search_pathes_iterator)
            if child_count:
                subfolder_item.hasfolders = 1
    except StopIteration:
        pass


@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
@user_exists_and_not_blocked
def content(req):
    fs = Bus(request=req)
    load_source_ids = req.meta and 'source_ids' in req.meta

    ycrid = mpfs.engine.process.get_cloud_req_id()
    if YcridParser.get_platform(ycrid) == 'dav' and Address(req.path).path.startswith('/disk'):
        if rate_limiter.is_limit_exceeded(RATE_LIMITER_GROUP_NAMES_LIST_WEBDAV, req.uid):
            raise errors.RequestsLimitExceeded429()

    result = fs.content(
        req.uid,
        req.path,
        req.args,
        load_source_ids=load_source_ids
    )
    if req.meta and 'hasfolders' in req.meta:
        base_folder_resource = req.form.model
        _set_hasfolders(base_folder_resource)
    # этот result далее никто не обрабатывает
    return result


def dir_list(req):
    content(req)
    # Прости меня читающий сей код. Не я такой - жизнь такая.
    base_folder_resource = req.form.model
    if not isinstance(base_folder_resource, MPFSFolder):
        raise errors.FolderNotFound()
    # Убираем файлы, в этой ручке они не нужны
    base_folder_resource.children_items['files'] = []


def _dir_size(uid, address):
    folder_resource = factory.get_resource(uid, address)
    if not isinstance(folder_resource, MPFSFolder):
        raise errors.FolderNotFound()
    return SearchDB().folder_size(uid, folder_resource.visible_address.path, folder_resource.resource_id)


@user_exists_and_not_blocked
def dir_size(req):
    """Возвратить размер папки."""
    # Проверяем наличие ресурса в БД
    address = Address.Make(req.uid, req.path)
    if address.storage_name == LNAROD_AREA:
        raise errors.UrlPathError(req.path)
    return _dir_size(req.uid, address)


def public_dir_size(req):
    """Возвратить размер публичной папки по публичному хешу.

    Принимаемые GET-параметры:
    * private_hash [обязательный] - публичный (он же приватный :)) хеш директории.
    """
    publicator = Publicator(request=req)
    publicator_address = publicator.get_address(
        req.private_hash,
        password_info=dict(
            password=req.request_headers.get(PUBLIC_PASSWORD_HEADER, None),
            token=req.request_headers.get(PUBLIC_PASSWORD_TOKEN_HEADER, None)
        )
    )
    raw_address = publicator_address['rawaddress']
    uid = publicator_address['uid']
    return _dir_size(uid, Address(raw_address))


def list_installers(req):
    fs = Bus(request=req)
    return fs.content(
        req.uid,
        Address.Make(req.uid, '/share/dist/').id
    )


def _list_subresources(uid, path, mimetypes=None, mediatypes=None, offset=0, amount=10, sort_field=None, asc_order=True,
                       extended_response=False):
    search_result = SearchDB().list_subresources(
        uid, path, mimetypes=mimetypes,
        mediatypes=mediatypes, offset=offset, amount=amount,
        sort_field=sort_field, asc_order=asc_order
    )

    mpfs_items = None
    if extended_response:
        if not search_result['items']:
            mpfs_items = []
        else:
            addresses = [Address.Make(uid, i['path']) for i in search_result['items']]
            mpfs_items = factory.get_resources(uid, addresses)

    return {
        'total': search_result['total'],
        'search_items': search_result['items'],
        'mpfs_items': mpfs_items,
    }


@user_exists_and_not_blocked
def list_subresources(req):
    mediatypes = []
    # внутри оперируем числовым представлением mediatypes
    if req.mediatypes:
        raw_mediatypes = req.mediatypes.split(',')
        mediatypes = [getGroupNumber(i) for i in raw_mediatypes]

    result = _list_subresources(
        req.uid,
        req.path,
        mimetypes=req.mimetypes.split(',') if req.mimetypes else [],
        mediatypes=mediatypes,
        offset=req.offset,
        amount=req.amount,
        sort_field=req.sort,
        asc_order=bool(req.order),
        extended_response=bool(req.extended_response)
    )

    if req.extended_response:
        return {
            'total': result['total'],
            'items': result['mpfs_items']
        }
    else:
        return {
            'total': result['total'],
            'items': result['search_items']
        }


def _search_resources_info_by_file_ids(uid, file_ids, sort_field, asc_order, fields):
    file_ids = filter(None, file_ids)
    if not file_ids:
        return {
            'total': 0,
            'items': []
        }
    return SearchDB().resources_info_by_file_ids(
        uid,
        file_ids,
        sort_field=sort_field,
        asc_order=asc_order,
        fields=fields
    )


@user_exists_and_not_blocked
def search_sizes_per_mediatype(req):
    return SearchDB().get_sizes_per_mediatype(req.uid, '/disk/')


@user_exists_and_not_blocked
def search_bulk_info(req):
    return _search_resources_info_by_file_ids(
        req.uid,
        req.file_ids.split(','),
        sort_field=req.sort,
        asc_order=bool(req.order),
        fields=req.search_meta.split(',')
    )


def public_search_bulk_info(req):
    publicator = Publicator(request=req)
    resource = publicator.get_public_not_blocked_resource(req.private_hash)

    fields = req.search_meta.split(',')
    if 'path' not in fields:
        fields.append('path')
    sr = _search_resources_info_by_file_ids(
        resource.address.uid,
        req.file_ids.split(','),
        sort_field=req.sort,
        asc_order=bool(req.order),
        fields=fields
    )
    return {
        'total': sr['total'],
        'items': sr['items']
    }


@slave_read_ok
@user_exists_and_not_blocked
def tree(req):
    fs = Bus(request=req)
    return fs.tree(
        req.uid,
        req.path,
        req.deep_level,
        req.args['bounds']['sort'],
        req.args['bounds']['order'],
        parents=req.parents,
    )


@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
@slave_read_ok
@user_exists_and_not_blocked
def fulltree(req):
    fs = Bus(request=req)
    return fs.fulltree(
        req.uid,
        req.path,
        deep_level=req.deep_level,
        relative=req.relative,
    )


@slave_read_ok
@user_exists_and_not_blocked
def services(req):
    fs = Bus(request=req)
    return fs.services(
        req.uid,
        req.path,
        req.deep_level,
        req.args['bounds']['sort'],
        req.args['bounds']['order'],
    )


@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
@user_exists_and_not_blocked
def info(req):
    uid = req.uid
    raw_address = req.path
    load_source_ids = req.meta and 'source_ids' in req.meta

    fs = Bus(request=req)
    try:
        result = fs.info(
            uid,
            raw_address,
            req.unzip_file_id,
            load_source_ids=load_source_ids,
        )
    except errors.ResourceNotFound:
        if FEATURE_NOTIFY_PHOTOSLICE_INDEX_WHEN_FILE_NOT_FOUND:
            DiskDataIndexer.queue_check_missing_files_and_update_photoslice(uid, [Address(raw_address).path])
        raise

    return result


def _check_limit_by_shard(uids):
    pg_query_executer = PGQueryExecuter()
    for uid in uids:
        if not pg_query_executer.is_user_in_postgres(uid):
            continue
        shard_id = pg_query_executer.get_shard_id(uid)

        if rate_limiter.is_limit_exceeded(RATE_LIMITER_BULK_INFO_BY_RESOURCE_IDS_GROUP_NAME, shard_id):
            raise errors.RequestsLimitExceeded429()


@user_exists_and_not_blocked
def bulk_info_by_resource_ids(req):
    load_source_ids = req.meta and 'source_ids' in req.meta
    raw_resource_ids = from_json(req.http_req.data)
    if not isinstance(raw_resource_ids, list):
        raise errors.BadArguments()
    resource_ids = [ResourceId.parse(i) for i in raw_resource_ids]
    enable_service_ids = req.enable_service_ids.split(',')

    user_agent = req.request_headers.get('user-agent', None)
    is_lenta_worker = UserAgentParser.is_lenta_worker(user_agent)
    if RATE_LIMITER_BULK_INFO_BY_RESOURCE_IDS_ENABLED and is_lenta_worker:
        owner_uids = set([r.uid for r in resource_ids])
        _check_limit_by_shard(owner_uids)
    enable_optimization = FEATURE_TOGGLES_ENABLE_OPTIMIZATION_FOR_LENTA_WORKER and is_lenta_worker

    fs = Bus(request=req)
    filtered_resources = fs.resources_by_resource_ids(
        req.uid,
        resource_ids,
        enable_service_ids=enable_service_ids,
        load_source_ids=load_source_ids,
        enable_optimization=enable_optimization
    )
    if req.meta is not None and ('office_online_url' in req.meta or 'office_online_sharing_url' in req.meta or 'office_access_state' in req.meta or not req.meta):
        for resource in filtered_resources:
            append_meta_to_office_files(resource, req)

    return filtered_resources

@user_exists_and_not_blocked
def bulk_info(req):
    load_source_ids = req.meta and 'source_ids' in req.meta
    uid = req.uid
    fs = Bus(request=req)
    paths = from_json(req.http_req.data)
    if not isinstance(paths, list):
        raise errors.BadArguments()
    result = fs.bulk_info(uid, paths, load_source_ids=load_source_ids)
    if FEATURE_NOTIFY_PHOTOSLICE_INDEX_WHEN_FILE_NOT_FOUND:
        missing_paths = set(paths) - set(resource.path for resource in result)
        if missing_paths:
            DiskDataIndexer.queue_check_missing_files_and_update_photoslice(uid, list(missing_paths))
    return result


@user_exists_and_not_blocked
def info_by_file_id(req):
    fs = Bus(request=req)
    load_source_ids = req.meta and 'source_ids' in req.meta
    resource = fs.resource_by_file_id(
        req.uid,
        req.file_id,
        req.owner_uid,
        enable_service_ids=('/disk', PHOTOUNLIM_AREA_PATH),
        load_source_ids=load_source_ids,
    )
    if req.meta is not None:
        if 'views_counter' in req.meta or not req.meta:
            # not req.meta = передать всю мету
            resource.load_views_counter()
    return resource.info()


@user_exists_and_not_blocked
def image_dimensions_by_file_id(req):
    fs = Bus(request=req)
    resource = fs.resource_by_file_id(req.uid, req.file_id, enable_service_ids=('/disk', PHOTOUNLIM_AREA_PATH))

    height = resource.meta.get('height')
    width = resource.meta.get('width')

    if height is not None and width is not None:
        return {
            'width': width,
            'height': height,
        }

    search_response = SearchDB().resources_info_by_file_ids(
        uid=req.uid,
        file_ids=(req.file_id,),
        fields=('width', 'height')
    )

    if not search_response['items']:
        return {
            'width': None,
            'height': None
        }

    found_item = search_response['items'][0]
    width = found_item['width']
    height = found_item['height']
    return {
        'width': int(width) if width is not None else width,
        'height': int(height) if height is not None else height,
    }


@user_exists_and_not_blocked
def image_metadata_by_file_id(req):
    fs = Bus(request=req)
    resource = fs.resource_by_file_id(req.uid, req.file_id, enable_service_ids=('/disk', PHOTOUNLIM_AREA_PATH))

    height = resource.meta.get('height')
    width = resource.meta.get('width')
    angle = resource.meta.get('angle')

    if height is not None and width is not None:
        if req.orientation and angle in (90, 270):
            height, width = width, height
        return {
            'width': width,
            'height': height,
        }

    search_response = SearchDB().resources_info_by_file_ids(
        uid=req.uid,
        file_ids=(req.file_id,),
        fields=('width', 'height', 'orientation'),
    )

    if not search_response['items']:
        return {
            'width': None,
            'height': None,
        }

    found_item = search_response['items'][0]
    width = found_item.get('width')
    height = found_item.get('height')
    if width is not None and height is not None:
        width = int(width)
        height = int(height)
        orientation = found_item.get('orientation')  # landscape, portrait
        if req.orientation and \
            ((orientation == 'portrait' and width > height) or
             (orientation == 'landscape' and width < height)):
            height, width = width, height

    return {
        'width': width,
        'height': height,
    }


@user_exists_and_not_blocked
def info_by_resource_id(req):
    resource_ids = [ResourceId.parse(req.resource_id)]
    fs = Bus(request=req)

    resources = fs.resources_by_resource_ids(req.uid, resource_ids, enable_service_ids=('/disk', PHOTOUNLIM_AREA_PATH))
    if not resources:
        raise errors.ResourceNotFound()
    resource = resources[0]
    if req.meta is not None:
        if 'views_counter' in req.meta or not req.meta:
            # not req.meta = передать всю мету
            resource.load_views_counter()
    return resource.info()


@user_exists_and_not_blocked
def url(req):
    fs = Bus(request=req)
    return fs.url(req.uid, req.path, req.params)


@user_exists_and_not_blocked
def gdpr_takeout_electrichki(req):
    fs = Bus(request=req)
    params = {'expire_seconds': 4 * 60 * 60, 'owner_uid': req.uid}
    return fs.public_download_url(req.uid, '%s:/settings/Favorites.arch' % req.uid, params)


########## Видео стриминг ##########
@user_exists_and_not_blocked
def video_streams(req):
    address = Address.Make(req.uid, req.path)
    resource = factory.get_resource(req.uid, address)
    if not isinstance(resource, MPFSFile):
        raise errors.FileNotFound()
    return VideoStreaming().get_video_info(
        resource.uid, resource.file_mid(),
        use_http=bool(req.use_http),
        user_ip=req.user_ip,
        client_id=req.client_id
    )


def public_video_streams(req):
    consumer = req.uid if req.uid is not None else PUBLIC_UID
    public_addr = PublicAddress(req.private_hash)

    album = get_album_by_public_key_unchecked(public_addr.hash)
    if album:
        if not public_addr.has_relative():
            raise AlbumItemMissed()
        resource = get_resource_from_album_item(album, public_addr.path.strip('/'), consumer)
    else:
        resource = Publicator().get_fully_public_resource(public_addr, uid=consumer)
    if not isinstance(resource, MPFSFile):
        raise errors.FileNotFound()
    return VideoStreaming().get_public_video_info(
        resource.address.uid,
        resource.file_mid(),
        use_http=bool(req.use_http),
        user_ip=req.user_ip,
        consumer_uid=consumer,
        client_id=req.client_id
    )


@user_exists_and_not_blocked
def video_url(req):
    fs = Bus(request=req)
    return fs.video_url(req.uid, req.path)


@allow_empty_disk_info
@slave_read_ok
def public_video_url(req):
    publicator = Publicator(request=req)
    return publicator.public_video_url(req.uid, req.private_hash, req.yandexuid)


####################################


@user_exists_and_not_blocked
@user_is_writeable
def mkdir(req):
    fs = Bus(request=req)
    fs.check_address(req.path, strict=True)
    fs.check_lock(req.path)

    # https://st.yandex-team.ru/CHE-174
    # Костыль на время миграции, необходимый из-за True опции folders./attach.allow_folders, после миграции запретить
    # создание папок и удалить этот кусок кода
    address = Address(req.path, is_folder=True)
    if address.path.startswith('/attach') and not Attach.is_inside_allowed_folders(address):
        # адрес не попадает в разрешенный список
        raise errors.MkdirNotPermitted(req.path)

    # Если параметр visible передан, то приводим его к intу, если не приводится - падаем в HTTP 500 SERVER ERROR
    visible_key = 'visible'
    req.changes[visible_key] = int(req.changes.get(visible_key, 1))
    for field_name in ('ctime', 'mtime'):
        try:
            req.changes[field_name] = int(req.changes[field_name])
        except:
            pass

    result = fs.mkdir(req.uid, req.path, data=req.changes)
    return result


@user_exists_and_not_blocked
@user_is_writeable
def fotki_mkfile(req):
    """Создать файл без колбэков, имея все необходимые данные.

    По умолчанию оповещает поиск о созданном файле.
    В случае отсутствия стида превью, мы не генерируем превью
    (https://st.yandex-team.ru/CHEMODAN-38802#1510576528000).
    """
    fs = Bus(request=req)
    address = Address.Make(uid=req.uid, path=req.path)
    if not Attach.is_inside_allowed_folders(address):
        raise errors.BadRequestError('Got not allowed path %s.' % req.path)
    fs.check_address(address.id, strict=True)
    fs.check_lock(address.id)
    non_required_data = {}

    if hasattr(req, 'source'):
        non_required_data['source'] = req.source

    if hasattr(req, 'type'):
        non_required_data['type'] = req.type

    meta = {
        'file_mid': req.file_mid,
        'digest_mid': req.digest_mid,
        'md5': req.md5,
        'sha256': req.sha256,
        'size': req.size,
        'mimetype': req.mime_type,
    }
    if hasattr(req, 'preview_mid'):
        meta['pmid'] = req.preview_mid

    if hasattr(req, 'etime'):
        # etime приходит сюда строковым, в миллисекундах
        # ctime и mtime тоже, но в итоге не используются, поэтому не конвертируются
        # https://st.yandex-team.ru/CHEMODAN-40145
        meta['etime'] = int(req.etime) / 1000

    fs.mkfile(
        req.uid,
        address.id,
        data=meta,
        notify_search=True,
        # обязательные параметры
        ctime=req.ctime,
        mtime=req.ctime,
        **non_required_data
    )

    return True


@user_exists_and_not_blocked
@user_is_writeable
def user_set_readonly(req):
    mpfs.engine.process.usrctl().set_readonly(req.uid)


@user_exists_and_not_blocked
def user_unset_readonly(req):
    mpfs.engine.process.usrctl().set_writeable(req.uid)


@user_exists_and_not_blocked
@user_is_writeable
def rm(req):
    fs = Bus(request=req)
    fs.check_lock(req.path)
    fs.check_address(req.path)
    fs.check_existing_md5(req.uid, req.path, req.md5)
    result = fs.rm(req.uid, req.path)
    if is_yateam_root(Address(req.path).path) and user_has_nda_rights(req.uid):
        recreate_yateam_dir(req.uid)
    return result


@user_exists_and_not_blocked
@user_is_writeable
def rm_by_resource_id(req):
    # TODO: удаление выполняется после получения путей из ресурсов, правильнее было бы удалять файл по resource_id
    resource_ids = [ResourceId.parse(req.resource_id)]
    fs = Bus(request=req)
    filter_values = {}
    if req.md5:
        filter_values['md5'] = req.md5
    if req.sha256:
        filter_values['sha256'] = req.sha256
    if req.size:
        filter_values['size'] = int(req.size)

    filter_duplicates = not req.rm_all
    resources = fs.resources_by_resource_ids_filtered(req.uid, resource_ids,
                                                      enable_service_ids=('/disk', PHOTOUNLIM_AREA_PATH),
                                                      filter_duplicates=filter_duplicates, filter_values=filter_values)
    if not resources:
        raise errors.ResourceNotFound()

    full_paths = []
    for resource in resources:
        if req.files_only and resource.type == 'dir':
            raise errors.FolderDeletionByResourceIdForbiddenError()
        full_paths.append(resource.visible_address.id)

    result = {'list': []}
    for full_path in full_paths:
        fs.check_lock(full_path)
        fs.check_address(full_path)
        fs.check_existing_md5(req.uid, full_path, req.md5)
        result['list'].append(fs.rm(req.uid, full_path))
        if is_yateam_root(Address(full_path).path) and user_has_nda_rights(req.uid):
            recreate_yateam_dir(req.uid)

    return result


@user_exists_and_not_blocked
@user_is_writeable
def async_rm(req):
    fs = Bus(request=req)
    fs.check_address(req.path)
    fs.check_lock(req.path)
    fs.check_existing_md5(req.uid, req.path, req.md5)
    operation = manager.create_operation(
        req.uid,
        'remove',
        Address(req.path).storage_name,
        odata=dict(
            path=req.path,
            connection_id=req.connection_id,
            callback=req.callback,
        ),
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_rm_by_resource_id(req):
    # TODO: удаление выполняется после получения путей из ресурсов, правильнее было бы удалять файл по resource_id
    resource_ids = [ResourceId.parse(req.resource_id)]
    fs = Bus(request=req)
    filter_values = {}
    if req.md5:
        filter_values['md5'] = req.md5
    if req.sha256:
        filter_values['sha256'] = req.sha256
    if req.size:
        filter_values['size'] = int(req.size)

    filter_duplicates = not req.rm_all
    resources = fs.resources_by_resource_ids_filtered(req.uid, resource_ids,
                                                      enable_service_ids=('/disk', PHOTOUNLIM_AREA_PATH),
                                                      filter_duplicates=filter_duplicates, filter_values=filter_values)
    if not resources:
        raise errors.ResourceNotFound()

    full_paths = []
    for resource in resources:
        if req.files_only and resource.type == 'dir':
            raise errors.FolderDeletionByResourceIdForbiddenError()
        full_path = resource.visible_address.id
        full_paths.append(resource.visible_address.id)
        fs.check_address(full_path, strict=True)
        fs.check_lock(full_path)
        fs.check_existing_md5(req.uid, full_path, req.md5)

    operation = manager.create_operation(
        req.uid,
        'remove',
        'bulk',
        odata=dict(
            paths=full_paths,
            connection_id=req.connection_id,
            callback=req.callback,
        ),
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_trash_drop_all(req):
    fs = Bus(request=req)
    fs.check_trash_lock_for(Address.Make(req.uid, '/trash').id)
    if fs.is_trash_append_process_running(req.uid) and FEATURE_TOGGLES_CHECK_TRASH_APPEND_BEFORE_TRASH_DROP:
        raise errors.TrashAppendIsRunningError()
    operation = manager.create_operation(
        req.uid,
        'trash',
        'drop',
        odata=dict(
            connection_id=req.connection_id,
            callback=req.callback,
        ),
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_trash_append(req):
    fs = Bus(request=req)
    fs.add_to_trash_cleaner_queue(req.uid)
    fs.check_address(req.path, strict=True)
    fs.check_lock(req.path)
    fs.check_trash_lock_for(req.path)
    fs.check_existing_md5(req.uid, req.path, req.md5)
    operation = manager.create_operation(
        req.uid,
        'trash',
        'append',
        odata=dict(
            path=req.path,
            connection_id=req.connection_id,
            callback=req.callback,
        )
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_trash_append_by_resource_id(req):
    resource_ids = [ResourceId.parse(req.resource_id)]
    fs = Bus(request=req)
    fs.add_to_trash_cleaner_queue(req.uid)
    filter_values = {}
    if req.md5:
        filter_values['md5'] = req.md5
    if req.sha256:
        filter_values['sha256'] = req.sha256
    if req.size:
        filter_values['size'] = int(req.size)

    filter_duplicates = not req.append_all
    resources = fs.resources_by_resource_ids_filtered(req.uid, resource_ids,
                                                      enable_service_ids=('/disk', PHOTOUNLIM_AREA_PATH),
                                                      filter_duplicates=filter_duplicates, filter_values=filter_values)
    if not resources:
        raise errors.ResourceNotFound()

    full_paths = []
    for resource in resources:
        if req.files_only and resource.type == 'dir':
            raise errors.FolderDeletionByResourceIdForbiddenError()
        full_path = resource.visible_address.id
        full_paths.append(resource.visible_address.id)
        fs.check_address(full_path, strict=True)
        fs.check_lock(full_path)
        fs.check_trash_lock_for(full_path)
        fs.check_existing_md5(req.uid, full_path, req.md5)

    operation = manager.create_operation(
        req.uid,
        'trash',
        'append_bulk',
        odata=dict(
            paths=full_paths,
            connection_id=req.connection_id,
            callback=req.callback,
        )
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_trash_drop(req):
    fs = Bus(request=req)
    fs.check_lock(req.path)
    fs.check_trash_lock_for(req.path)
    operation = manager.create_operation(
        req.uid,
        'trash',
        'droppath',
        odata=dict(
            path=req.path,
            connection_id=req.connection_id,
            callback=req.callback,
        )
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def trash_drop(req):
    fs = Bus(request=req)
    fs.check_lock(req.path)
    fs.check_trash_lock_for(req.path)
    return fs.trash_drop_element(req.uid, req.path)


@user_exists_and_not_blocked
@user_is_writeable
def async_trash_restore(req):
    fs = Bus(request=req)
    fs.check_lock(req.path)
    fs.check_trash_lock_for(req.path)
    operation = manager.create_operation(
        req.uid,
        'trash',
        'restore',
        odata=dict(
            path=req.path,
            connection_id=req.connection_id,
            callback=req.callback,
            name=req.name,
            force=req.force,
        )
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_social_contacts(req):
    """
    Постановка задачи на выдачу контактов и френдов пользователя
    """
    operation = social.list_contacts(req.uid, req.groups)
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_invite_contacts(req):
    """
    Постановка задачи на выдачу контактов со статусами отправленности инвайтов
    """
    operation = invites.list_contacts(req.uid)
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
def social_rights(req):
    """
    Выдача социальных прав пользователя
    """
    return social.user_rights(req.uid, req.scenario)


@user_exists_and_not_blocked
def social_get_albums(req):
    """
    Выдача фотоальбомов пользователя
    """
    return social.get_albums(req.uid, req.provider)


@user_exists_and_not_blocked
@user_is_writeable
def social_create_album(req):
    """
    Создание фотоальбома пользователя в соц. сети
    """
    return social.create_album(req.uid, req.provider, req.title, req.privacy)


@user_exists_and_not_blocked
@user_is_writeable
def async_social_export_photos(req):
    """
    Постановка задачи на экспорт фото в соцсеть
    """
    operation = social.export_photos(req.uid, req.provider, req.albumid, req.photos)
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_social_import_photos(req):
    """
    Постановка задачи на импорт фото из соцсети
    """
    operation = social.import_photos(req.uid, req.provider, req.type, req.connection_id)
    return {
        'oid': operation.id,
        'type': operation.type,
        'provider_dir': operation.data.get(PROVIDER_DIR),
        'at_version': operation.at_version
    }


@user_exists_and_not_blocked
@user_is_writeable
def async_store_external(req):
    """
    Сохранение файла из веба в диск
    """
    fs = Bus(request=req)
    fs.check_address(req.target, strict=True)
    fs.check_lock(req.target)

    odata = {
        'connection_id': req.connection_id,
        'target': req.target,
        'service_file_url': req.external_url,
        'provider': 'web',
        'disable_redirects': req.disable_redirects,
        'disable_retries': req.disable_retries,
    }
    operation = manager.create_operation(req.uid, 'external_copy', 'web_disk', odata)
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_is_writeable
def extract_file_from_archive(req):
    """
    Извлечение файла из архива
    """
    if req.src_archive:
        src_archive_addr = Address.Make(req.uid, req.src_archive)
    elif req.private_hash:
        publicator = Publicator(request=req)
        src_archive_addr = Address(publicator.get_address(req.private_hash)['rawaddress'])
        check_yateam_subtree_access(req.uid, src_archive_addr.path, src_archive_addr.uid)
    else:
        raise errors.BadArguments('"src_archive" or "private_hash" required.')

    fs = Bus(request=req)
    dst_addr = Address.Make(req.uid, req.dst)
    src_file = os.path.normpath(req.src_file)

    dst_addr = fs.autosuffix_address(dst_addr)

    fs.check_source(src_archive_addr.id)
    fs.check_target(dst_addr.id)

    operation = manager.create_operation(
        req.uid,
        'store',
        'extract-file-from-archive',
        odata={
            'dst_addr': dst_addr,
            'archive_addr': src_archive_addr,
            'file_in_archive_path': src_file,
            'connection_id': req.connection_id,
        }
    )
    return {'oid': operation.id, 'type': operation.type}


def check_fos_request_is_correct(req):
    if not req.http_req.data:
        raise errors.ClientBadRequest('No body in request')
    resp_body = from_json(req.http_req.data)
    support_text = resp_body.get('fos_support_text', None)
    if support_text is None:
        raise errors.ClientBadRequest('No support text in requests body')
    if len(support_text) > MAX_FOS_TEXT_LEN:
        raise errors.ClientBadRequest('Support text length greater than max len')
    return True


def _get_speed_limit(req):
    """Определяет нужно ли лимитировать скорость загрузки.

    Приоритет ограничений:
      1. По методу авторизации
      2. По media type
    """
    user_agent = req.request_headers.get('user-agent', None)
    if (UserAgentParser.is_yandex_search_mobile(user_agent) or
        UserAgentParser.is_yandex_mail_mobile(user_agent)):
        return

    uid = getattr(req, 'uid', None)
    if uid and uid in SPEED_LIMITS_BY_UID:
        return SPEED_LIMITS_BY_UID[uid]

    platform_prefix = YcridParser.get_platform(mpfs.engine.process.get_cloud_req_id())
    if platform_prefix not in (YcridPlatformPrefix.DAV,
                               YcridPlatformPrefix.REST):
        return

    excluded_user_agents = settings.speed_limits['exclude_user_agents']
    if user_agent and user_agent in excluded_user_agents:
        return

    auth_method = req.request_headers.get('X-Auth-Method')
    if auth_method:
        auth_method = auth_method.lower()
    if auth_method in SPEED_LIMITS_BY_AUTH_METHOD:
        return SPEED_LIMITS_BY_AUTH_METHOD[auth_method]

    media_type = getMediaTypeByName(req.path)
    if media_type in SPEED_LIMITS_BY_MEDIA_TYPE:
        return SPEED_LIMITS_BY_MEDIA_TYPE[media_type]


def _prepare_store_args(req):
    try:
        address = Address(req.path)
    except errors.AddressError:
        address = None

    allow_uninitialized = address.storage_name == 'attach'
    _check_user_exists_and_not_blocked(req, allow_uninitialized_user=allow_uninitialized)

    if not allow_uninitialized and not mpfs.engine.process.usrctl().is_user_completely_initialized(req.uid):
        raise errors.StorageInitUser()

    available_kwargs_names = ['force', 'changes', 'size', 'md5', 'sha256',
                              'replace_md5', 'client_type', 'user_ip',
                              'use_https', 'callback', 'connection_id', 'skip_check_space', 'tld',
                              'live_photo_md5', 'live_photo_sha256', 'live_photo_size', 'live_photo_type',
                              'live_photo_operation_id', 'source_id', 'force_deletion_log_deduplication',
                              'photostream_destination']
    kwargs = {}
    if address.storage_name == 'client':
        available_kwargs_names.extend(['fos_app_version', 'fos_reply_email', 'fos_expire_seconds', 'fos_os_version',
                                       'fos_support_text', 'fos_subject', 'fos_recipient_type'])
        if check_fos_request_is_correct(req):
            kwargs['fos_support_text'] = from_json(req.http_req.data).get('fos_support_text', None)

    for kwarg_name in available_kwargs_names:
        try:
            kwargs[kwarg_name] = getattr(req, kwarg_name)
        except AttributeError:
            pass

    photoslice_album_type_data = PhotosliceAlbumTypeDataStructure.cons_by_params(
        req.device_original_path, req.device_collections, getattr(req, 'etime', None), req.user_agent, req.path,
        req.changes
    )
    kwargs['photoslice_album_type_data'] = photoslice_album_type_data

    # параметр не нужен в самой функции store, поэтому далее не прокидываем
    skip_speed_limit = getattr(req, 'skip_speed_limit', None)
    if not skip_speed_limit:
        speed_limit = _get_speed_limit(req)
        if speed_limit is not None:
            kwargs['uploader_speed_limit'] = speed_limit

    return address, kwargs


@allow_user_init_in_progress
@user_is_writeable
def store(req):
    """
    Дефолтная загрузка файла
    """
    fs = Bus(request=req)

    _, kwargs = _prepare_store_args(req)

    operation = fs.store(req.uid, req.path, etime_from_client=getattr(req, 'etime', None),
                         mtime_from_client=getattr(req, 'mtime', None), ctime_from_client=getattr(req, 'ctime', None),
                         user_agent=req.user_agent, **kwargs)

    if isinstance(operation, dict):
        return operation
    else:
        return {
            'upload_url': operation['upload_url'],
            'oid': operation.id,
            'type': operation.type,
            'at_version': operation.at_version
        }


@allow_user_init_in_progress
@user_is_writeable
def attach_store(req):
    """
    Загрузка файла для аттачевых разделов
    """

    fs = Bus(request=req)

    address, kwargs = _prepare_store_args(req)
    if 'uploader_speed_limit' in kwargs:
        del kwargs['uploader_speed_limit']

    path = req.path
    if (experiment_manager.is_feature_active(ATTACH_UPLOAD_TO_DISK_QUOTA)
        and address.storage_name == ATTACH_AREA):
        # заменяем /attach на дефолтную папку для аттачей в /disk
        # Пример: /attach/mail-attachment.ext -> /disk/Почтовые вложения/mail-attachment.ext
        if user.NeedInit(req.uid):
            user.Create(req.uid, source='mail-attach')

        locale = User(req.uid).get_supported_locale()
        fs.mksysdir(req.uid, type=DEFAULT_FOLDERS_NAMES.ATTACH, locale=locale)
        address.change_parent(fs.get_sysdir_address(req.uid, DEFAULT_FOLDERS_NAMES.ATTACH, locale=locale),
                              old_parent=ATTACH_AREA_PATH)
        address = fs.autosuffix_address(address)
        if kwargs['changes']:
            kwargs['changes']['public'] = 1
        else:
            kwargs['changes'] = {'public': 1}
        path = address.id

    operation = fs.store(req.uid, path, etime_from_client=getattr(req, 'etime', None),
                         mtime_from_client=getattr(req, 'mtime', None), ctime_from_client=getattr(req, 'ctime', None),
                         user_agent=req.user_agent, **kwargs)
    if isinstance(operation, dict):
        return operation
    else:
        return {
            'upload_url': operation['upload_url'],
            'oid': operation.id,
            'type': operation.type,
            'at_version': operation.at_version
        }


@user_exists_and_not_blocked
@user_is_writeable
def astore(req):
    """
    Докачка файла по указанию пути и md5
    """
    fs = Bus(request=req)
    fs.check_address(req.path)

    from mpfs.common.util.limits.logic.upload_traffic_limit import check_upload_limits
    check_upload_limits(req.uid, 0)

    operation = manager.get_opened_store_operation(
        req.uid,
        req.path,
        req.md5,
        req.user_agent,
        callback=req.callback
    )

    if not Kladun().check_service_is_alive(operation['upload_url']):
        raise errors.OperationNotFound()

    return {
        'upload_url': operation['upload_url'],
        'oid': operation.id,
        'type': operation.type
    }


@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
@slave_read_ok
@user_exists_and_not_blocked
def default_folders(req):
    """
    Получение списка стандартных папок с учётом локализации
     - без check_exist возвращает dict такого вида {"downloads":"/disk/Загрузки/"}
     - c check_exist такого {"downloads": {"path": /disk/Загрузки/", "exist" 1}}
    """
    result = {}
    locale = user.User(req.uid).get_supported_locale()
    if req.check_exist:
        folders = {}
        for folder_type, paths in constants.DEFAULT_FOLDERS.iteritems():
            if (folder_type == DEFAULT_FOLDERS_NAMES.ATTACH
                and not experiment_manager.is_feature_active(ATTACH_UPLOAD_TO_DISK_QUOTA)):
                continue
            folders[folder_type] = Address.Make(req.uid, paths.get(locale))
        existence = [resource.address.path for resource in Disk().get_resources(req.uid, folders.values())]
        for key, value in folders.iteritems():
            result[key] = {
                "path": value.path,
                "exist": int(value.path in existence),
            }
    else:
        for key, value in constants.DEFAULT_FOLDERS.iteritems():
            if (key == DEFAULT_FOLDERS_NAMES.ATTACH
                and not experiment_manager.is_feature_active(ATTACH_UPLOAD_TO_DISK_QUOTA)):
                continue
            result[key] = value.get(locale)
    return result


@user_exists_and_not_blocked
@user_is_writeable
def dstore(req):
    """
    Дельта обновление файла
    """
    fs = Bus(request=req)
    fs.check_address(req.path)
    fs.check_lock(req.path)

    file_info = fs.info(req.uid, fs.preprocess_path(req.uid, req.path), is_file=True)['this']
    if req.md5 != file_info['meta']['md5']:
        raise errors.DStoreNotMatch()

    free_space = fs.check_available_space(req.uid, req.path)

    from mpfs.common.util.limits.logic.upload_traffic_limit import check_upload_limits
    check_upload_limits(req.uid, 0)

    operation = manager.create_operation(
        req.uid,
        'dstore',
        Address(req.path).storage_name,
        odata=dict(
            free_space=free_space,
            path=req.path,
            file=file_info,
            callback=req.callback,
            changes=req.changes,
            connection_id=req.connection_id,
        ),
        use_https=req.use_https,
    )
    return {
        'patch_url': operation['patch_url'],
        'oid': operation.id,
        'type': operation.type
    }


@slave_read_ok
def status(req):
    operation = manager.get_operation(req.uid, req.oid, allow_unknown_type=True)
    operation.mpfs_request = req
    return operation.get_status()


@user_exists_and_not_blocked
@user_is_writeable
def copy_resource(req):
    fs = Bus(request=req)
    fs.check_address(req.dst, strict=True)
    fs.check_lock(req.src)
    fs.check_lock(req.dst)
    fs.check_available_space_on_copy_by_raw_address(req.uid, req.dst)
    return fs.copy_resource(
        req.uid,
        req.src,
        req.dst,
        req.force,
        req.force_djfs_albums_callback,
    )


@user_exists_and_not_blocked
@user_is_writeable
def copy_disk(req):
    if req.uid == req.src_uid:
        raise errors.BadTargetUserError()

    fs = Bus(request=req)

    target = Address.Make(req.uid, req.dst)
    source = Address.Make(req.src_uid, '/disk')

    try:
        target_parent_resource = factory.get_resource(req.uid, target.get_parent())
    except errors.ResourceNotFound:
        raise errors.CopyParentNotFound()
    else:
        if isinstance(target_parent_resource, SharedResource) and \
            target_parent_resource.link.group.owner == req.src_uid:
            raise errors.BadTargetUserError()

    fs.check_address(target.id, strict=True)
    fs.check_lock(target.id)
    free_space = fs.check_available_space(req.uid, target.id)

    subtype = '%s_%s' % (source.storage_name, target.storage_name)
    operation = manager.create_operation(
        req.uid,
        'copy',
        subtype,
        odata=dict(
            target=target.id,
            source=source.id,
            free_space=free_space,
            force=0,
            connection_id=req.connection_id,
            src_uid=req.src_uid,
        ),
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def async_copy_resource(req):
    fs = Bus(request=req)
    fs.check_address(req.dst, strict=True)
    fs.check_lock(req.src)
    fs.check_lock(req.dst)

    java_copy = False
    if JAVA_DJFS_API_TASKS_COPY_ENABLED or req.uid in JAVA_DJFS_API_TASKS_COPY_ENABLED_UIDS or filter_uid_by_percentage(
        req.uid, JAVA_DJFS_API_TASKS_COPY_ENABLED_UIDS_PERCENT):
        if Address(req.src).storage_name != 'mail' and Address(req.dst).storage_name != 'attach':
            if req.force:
                try:
                    # force copy is not supported in java yet
                    # https://st.yandex-team.ru/CHEMODAN-55729
                    factory.get_resource(req.uid, req.dst)
                except errors.ResourceNotFound:
                    java_copy = True
            else:
                java_copy = True

    if java_copy:
        if not req.force:
            try:
                factory.get_resource(req.uid, req.dst)
            except errors.ResourceNotFound:
                pass
            else:
                raise errors.CopyTargetExists()

        try:
            factory.get_resource(req.uid, Address(req.dst).get_parent())
        except errors.ResourceNotFound:
            raise errors.CopyParentNotFound()

        operation_data = {'source': req.src, 'destination': req.dst, 'force': req.force, 'callback': req.callback,
                          'connection_id': req.connection_id}
        operation = manager.create_operation(req.uid, 'copy', 'copy', odata=operation_data)
    else:
        free_space = fs.check_available_space_on_copy_by_raw_address(req.uid, req.dst)

        target = req.dst
        source = req.src
        subtype = '%s_%s' % (Address(source).storage_name, Address(target).storage_name)
        operation = manager.create_operation(
            req.uid,
            'copy',
            subtype,
            odata=dict(
                target=target,
                source=source,
                free_space=free_space,
                force=req.force,
                callback=req.callback,
                connection_id=req.connection_id,
            ),
        )
    return {'oid': operation.id, 'type': 'copy', 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def import_attach_to_disk(req):
    """Импортирует аттач из письма в диск.

    Принимаемые GET-параметры:
    * uid [обязательный] - идентификатор пользователя, из под которого происходит вызов
    * dst [обязательный] - путь, куда кладем аттач
    * mail_mid [обязательный] - идентификатор письма (передается почтовиками, мы его ниоткуда взять не можем)
    * mail_hid [обязательный] - идентификатор части тела письма, где лежит аттач
    * overwrite (по умолчанию 0) - 0/1, флаг, сообщающий о том, нужно ли перезаписать файл,
                                   если файл с таким именем уже существует.
    * autosuffix (по умолчанию 0) - 0/1, флаг - нужно ли сохранять файл с уникальным именем,
                                    если файл с таким именем уже существует.
      Если переданы overwrite=0 и autosuffix=0, и файл по такому пути уже существует, будет возвращена ошибка
      Если переданы overwrite=1 и autosuffix=1, будет возвращена ошибка
    """
    if req.autosuffix and req.overwrite:
        raise errors.BadArguments('Request has both autosuffix and overwrite')

    fs = Bus(request=req)
    target = Address.Make(req.uid, req.dst)

    if target.storage_name != 'disk':
        raise errors.StorageBadRootPath()

    if req.autosuffix:
        fs.check_lock(target.get_parent().id)
    else:
        fs.check_address(target.id, strict=True)
        fs.check_lock(target.id)
    free_space = fs.check_available_space(req.uid, target.id)

    operation = manager.create_operation(
        req.uid,
        'copy',
        'import_mail_attach',
        odata=dict(
            target=target.id,
            source_mid=req.mail_mid,
            source_hid=req.mail_hid,
            free_space=free_space,
            force=req.overwrite,
            callback='',
            connection_id='',
            autosuffix=req.autosuffix,
        ),
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def import_attaches_to_disk(req):
    """Импортирует несколько аттачей из письма в диск.

    Принимаемые GET-параметры:
    * uid [обязательный] - идентификатор пользователя, из под которого происходит вызов
    * path [обязательный] - каталог, куда класть аттачи, должен существовать
    * overwrite (по умолчанию 0) - 0/1, флаг, сообщающий о том, нужно ли перезаписать файл,
                                   если файл с таким именем уже существует.
    * autosuffix (по умолчанию 0) - 0/1, флаг - нужно ли сохранять файл с уникальным именем,
                                    если файл с таким именем уже существует.
      Если переданы overwrite=0 и autosuffix=0, и файл по такому пути уже существует, будет возвращена ошибка
      Если переданы overwrite=1 и autosuffix=1, будет возвращена ошибка


    Тело запроса:
    {
        items: [
            {"mid": "123456", "hid": "1.1", "file_name": "file1"},
            {"mid": "123457", "hid": "1.1", "file_name": "file2"}
        ]

    }

    поле items обязательное и не может быть пустым
    """
    if req.autosuffix and req.overwrite:
        raise errors.BadArguments('Request has both autosuffix and overwrite')

    data = from_json(req.http_req.data)
    if not data:
        raise errors.JsonBodyExpectedError()

    items = data.get('items')
    if not items:
        raise errors.BadRequestError('No items in request')

    if not req.autosuffix and not req.overwrite:
        if len(set(item['file_name'] for item in items)) != len(items):
            raise errors.BadRequestError('Duplicate file names in request and both autosuffix and overwrite are off')

    fs = Bus(request=req)
    dest_folder_address = Address.Make(req.uid, req.path)

    if dest_folder_address.storage_name != 'disk':
        raise errors.StorageBadRootPath()

    fs.check_lock(dest_folder_address.id)
    free_space = fs.check_available_space(req.uid, dest_folder_address.id)

    operation = manager.create_operation(
        req.uid,
        'mail_attaches',
        'import_mail_attaches',
        odata=dict(
            path=dest_folder_address.id,
            items=items,
            free_space=free_space,
            force=req.overwrite,
            callback='',
            connection_id='',
            autosuffix=req.autosuffix,
        ),
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def import_file_from_service(req):
    """Импортирует файл из внешнего сервиса в диск в папку "Загрузки".

    Принимаемые GET-параметры:
    * uid [обязательный] - идентификатор пользователя, из под которого происходит вызов
    * service_id [обязательный] - идентификатор внешнего сервиса, откуда надо взять содержимое файла
    * service_file_id [обязательный] - идентификатор файла во внешнем сервисе
    * file_name [обязательный] - имя файла
    * overwrite (по умолчанию 0) - 0/1, флаг - нужно ли перезаписать файл, если файл с таким именем уже существует.
                                   (если указано 0 и файл существует, будет возвращена ошибка)
    * autosuffix (по умолчанию 1) - 0/1, флаг - нужно ли сохранять файл с уникальным именем,
                                    если файл с таким именем уже существует.
                                    (если указано 0 и файл существует, будет возвращена ошибка)
      Если переданы overwrite=1 и autosuffix=1, будет возвращена ошибка
    """
    if req.autosuffix and req.overwrite:
        raise ValueError('Request has both autosuffix and overwrite')

    fs = Bus(request=req)

    downloads_folder_address = fs.get_downloads_address(req.uid)

    if not req.autosuffix:
        dst_address = downloads_folder_address.get_child_file(req.file_name)
        fs.check_address(dst_address.id, strict=True)
        fs.check_lock(dst_address.id)
    else:
        fs.check_lock(downloads_folder_address.id)

    free_space = fs.check_available_space(req.uid, downloads_folder_address.id)

    operation = manager.create_operation(
        req.uid,
        'copy',
        'import_file_from_service',
        odata=dict(
            file_name=req.file_name,
            service_id=req.service_id,
            service_file_id=req.service_file_id,
            free_space=free_space,
            force=req.overwrite,
            autosuffix=req.autosuffix,
            callback='',
            connection_id='',
        ),
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def move_resource(req):
    fs = Bus(request=req)
    fs.check_address(req.dst, strict=True)
    fs.check_lock(req.src)
    fs.check_lock(req.dst)
    fs.check_moveability(req.src, req.dst)
    with CatchHistoryLogging(catch_messages=True) as catcher:
        fs.move_resource(
            req.uid,
            req.src,
            req.dst,
            req.force,
            check_hids_blockings=req.check_hids_blockings,
        )
        if is_yateam_root(Address(req.src).path) and user_has_nda_rights(req.uid):
            recreate_yateam_dir(req.uid)
        messages = catcher.get_messages()
        if req.return_status:
            # новое поведение
            if req.get_lenta_block_id:
                lenta_block_id = EventLogMessagesTransformer.to_lenta_block_id(messages=messages)
                if lenta_block_id:
                    return {'lenta_block_id': lenta_block_id}
                else:
                    return {}
            else:
                return {}
        else:
            # старое поведение
            return ''


@user_exists_and_not_blocked
@user_is_writeable
def async_move_resource(req):
    fs = Bus(request=req)
    fs.check_address(req.dst, strict=True)
    fs.check_lock(req.src)
    fs.check_lock(req.dst)
    fs.check_moveability(req.src, req.dst)

    source = req.src
    target = req.dst
    subtype = '%s_%s' % (Address(source).storage_name, Address(target).storage_name)
    operation = manager.create_operation(
        req.uid,
        'move',
        subtype,
        odata=dict(
            target=target,
            source=source,
            force=req.force,
            callback=req.callback,
            connection_id=req.connection_id,
        ),
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@user_exists_and_not_blocked
@user_is_writeable
def setprop(req):
    fs = Bus(request=req)
    fs.check_address(req.path)
    fs.check_lock(req.path)
    return fs.setprop(
        req.uid,
        req.path,
        req.changes,
        req.deleted,
    )


@replace_thrown_exception(errors.UserBlocked, errors.AuthorizationError())
@user_exists_and_not_blocked
def diff(req):
    load_source_ids = req.meta and 'source_ids' in req.meta
    if req.version:
        group_name = RATE_LIMITER_GROUP_NAMES_VERSION_DIFF
    elif not Address(req.path).is_system:  # check that the path is not storage and not is root
        # st.yandex-team.ru/CHEMODAN-43078
        # не лимитируем мобильные приложения при запросе дифа с фильтром по пути
        group_name = RATE_LIMITER_GROUP_NAMES_PATH_DIFF  # unknown group for rate_limiter
    else:
        group_name = RATE_LIMITER_GROUP_NAMES_FULL_DIFF

    rate_limiter.check_limit_exceeded(group_name, req.uid, error_class=errors.RequestsLimitExceeded429)

    fs = Bus(request=req)
    return fs.diff(
        req.uid,
        req.path,
        req.version,
        req.allow_quick_move_deltas,
        load_source_ids=load_source_ids,
    )


@user_exists_and_not_blocked
def deprecated_search(req):
    fs = Bus(request=req)
    return fs.deprecated_search(
        req.uid,
        req.path,
        req.query,
    )


@user_exists_and_not_blocked
def search(req):
    load_source_ids = req.meta and 'source_ids' in req.meta
    fs = Bus(request=req)
    ret = fs.search(
        req.uid, req.path, req.query,
        force=req.force, count_lost_results=req.count_lost_results, load_source_ids=load_source_ids
    )
    return ret


@user_exists_and_not_blocked
def geo_search(req):
    fs = Bus(request=req)
    ret = fs.geo_search(req.uid, req.latitude, req.longitude, req.start_date, req.end_date, req.distance,
                        count_lost_results=req.count_lost_results)
    return ret


@slave_read_ok
@user_exists_and_not_blocked
def timeline(req):
    fs = Bus(request=req)
    return fs.timeline(
        req.uid,
        req.path,
    )


@slave_read_ok
@user_exists_and_not_blocked
def set_last_files(req):
    fs = Bus(request=req)
    return fs.set_last_files(
        req.uid,
        req.path
    )


@user_exists_and_not_blocked
@user_is_writeable
def hardlink_copy(req):
    fs = Bus(request=req)

    fs.check_address(req.path)
    fs.check_lock(req.path)
    fs.check_preconditions(req.uid, req.path, replace_md5=req.replace_md5)
    fs.check_available_space(req.uid, req.path)

    res = fs.hardlink_copy(
        req.uid,
        req.path,
        req.md5,
        req.size,
        req.sha256,
        changes=req.changes
    )
    usr = user.User(req.uid)
    if req.client_type == 'desktop':
        usr.set_state(
            key='file_uploaded',
            value=1
        )
    usr.set_state(
        key='first_file',
        value=1
    )
    return res


@slave_read_ok
@user_exists_and_not_blocked
def space(req):
    '''
    Возвращает информацию о доступном и занятом юзером месте.

    Принимаемые GET-параметры:
    * uid [обязательный] - идентификатор пользователя, из под которого происходит вызов
    * count_default_folders - идентификатор системной папки, для которой нужно посчитать размер.
                              Это не бесплатная операция, которая может существенно сказываться на производительности ручки.
    Если указана системная папка, а её нет или она отсутствует в индексе Поиска по Диску,
    то ответ просто не будет содержать информации о папке.
    '''
    fs = Bus(request=req)
    ret = fs.space(req.uid)
    folder_type = req.count_default_folders

    if folder_type:
        try:
            default_folder_address_localized = fs.get_sysdir_address(req.uid, folder_type)
        except KeyError:
            default_folder_address_localized = None

        if default_folder_address_localized:
            try:
                dir_size_resp = _dir_size(req.uid, default_folder_address_localized)
            except errors.ResourceNotFound:
                dir_size_resp = None

            if dir_size_resp:
                ret['default_folders'] = {
                    folder_type:  dir_size_resp,
                }
    return ret


@user_exists_and_not_blocked(work_in_location=UserExistsAndNotBlockedParams.WorkInLocation.All,
                             attachuser_fail_strategy=UserExistsAndNotBlockedParams.AttachuserFailStrategy.Skip)
def kladun_callback(req):
    if req.stage not in VALID_KLADUN_CALLBACK_TYPES:
        raise BadRequestError('Unknown callback type: %s' % req.stage)

    try:
        operation = manager.process_operation(req.uid, req.oid, xml_data=req.body, stage=req.stage)
    except errors.OperationNotFound:
        raise errors.KladunOperationNotFound()

    op_data = operation.data

    if op_data.get('hardlinked', None):
        try:
            resource = Bus().resource(operation.uid, operation.get_path())
            from mpfs.core.filesystem.quota import Quota
            Quota().update_upload_traffic_for_operation(operation, bytes_uploaded=resource.size)
        except Exception:
            error_log.exception('Can\'t update upload traffic counter')

        raise errors.KladunHardlinkFound(headers={
            'path': Address(op_data.get('path')).path,
            'resource_id': op_data.get('resource_id')
        })
    elif 'error' in op_data:
        error_data = op_data['error']
        response_code = error_data.get('response')
        if (isinstance(error_data, dict) and
                error_data.get('code') == errors.EmptyFileUploadedForNonEmpyStoreError.code):
            raise errors.EmptyFileUploadedForNonEmpyStoreError
        if 400 <= response_code < 500:
            raise errors.KladunConflict()
        elif response_code >= 500:
            exception = error_data.get('exception')
            if isinstance(exception, Exception):
                raise exception
            elif isinstance(exception, tuple) and isinstance(exception[1], Exception):
                raise exception

    if req.stage == COMMIT_FILE_UPLOAD:
        resource = Bus().resource(operation.uid, operation.get_path())
        address = Address(operation.get_path())
        is_live_photo = isinstance(operation, SaveOnDiskOperation) and operation.is_live_photo_store(operation.data)
        event = FilesystemStoreEvent(
            uid=operation.uid, tgt_resource=resource, tgt_address=address,
            type=operation.type, subtype=operation.subtype,
            set_public=operation.data.get('set_public', False),
            changes=operation.data.get('changes', {}),
            is_live_photo=is_live_photo,
            size=safe_to_int(resource.size),
        )
        if not op_data.get('event_created'):
            if isinstance(operation, CopyMailDisk):
                with CatchHistoryLogging(catch_messages=True) as catcher:
                    event.send_self_or_group(resource=resource)
                    messages = catcher.get_messages(
                        pattern=SAVE_FILE_FROM_MAIL_LOG_MESSAGE_RE
                    )
                    EventLogMessagesTransformer.to_lenta_block_id_for_public_op(
                        messages=messages, operation=operation, save=True
                    )
            elif isinstance(operation, (ExtractFileFromArchive, CopyWebDisk)):
                with CatchHistoryLogging(catch_messages=True) as catcher:
                    event.send_self_or_group(resource=resource)
                    messages = catcher.get_messages(
                        pattern=[
                            EXTRACT_FILE_FROM_ARCHIVE_LOG_MESSAGE_RE,
                            SAVE_FILE_FROM_WEB_LOG_MESSAGE_RE
                        ]
                    )
                    EventLogMessagesTransformer.to_lenta_block_id_for_op(messages=messages, operation=operation, save=True)
            else:
                event.send_self_or_group(resource=resource)
            operation.data['event_created'] = True
            operation.save()
        from mpfs.core.filesystem.quota import Quota
        Quota().update_upload_traffic_for_operation(operation, bytes_uploaded=resource.size)

    result = True
    if 'path' in op_data and 'resource_id' in op_data:
        result = {'_headers': {
            'path': Address(op_data.get('path')).path,
            'resource_id': op_data.get('resource_id'),
        }}
    return result


def passport_callback(req):
    """Обработать колбэк от паспорта."""
    usr = User(req.uid)
    if req.event == 'account.changed':
        try:
            usr.public_info(raise_if_does_not_exist=True)
        except errors.PassportUserDoesNotExistError:
            mpfs_queue.put({
                'uid': req.uid,
            }, 'handle_passport_user_deleted_event')
    else:
        raise NotImplementedError()


def passport_user_enabled_changed(req):
    """Обработать изменение статуса enabled/disabled у пользователя.

    Формат:
    {
        'v': 1,
        'uid': uid,
        'event': event_name,
        'timestamp': event_timestamp,
    }

    event_name может быть 'account.enable', 'account.disable' и 'account.removal_disable'.
    1. account.enable - пользователь разблокирован, как после блокироваки так и после временного удаления
    2. account.disable - пользователь заблокирован
    3. account.removal_disable - пользователь удалил аккаунт, но у него ещё есть месяц, чтобы восстановить. в этом случае придёт account.enable.

    https://st.yandex-team.ru/PASSP-19950
    https://st.yandex-team.ru/CHEMODAN-39870
    """
    data = from_json(req.http_req.data)
    uid = str(data['uid'])
    event = str(data['event'])
    log.info('passport_user_enabled_changed callback for uid %s event %s' % (uid, event))
    if event not in {'account.enable', 'account.disable', 'account.removal_disable'}:
        raise PassportUnsupportedEventTypeError()

    if event == 'account.removal_disable':
        operation = manager.create_operation(uid, 'user', 'preliminary_delete', odata={})
        return {'status': 'ok', 'oid': operation.id}
    return {'status': 'ok'}


@user_is_writeable
def kladun_download_counter_inc(req):
    """
    Инкрементация счётчика скачиваний файла
    """
    address = PublicAddress(req.hash)
    key = crypt_agent.decrypt(address.hash)
    uid = key.split(RESOURCE_ID_SRC_SEPARATOR)[0]
    if User(uid).is_blocked():
        return

    return Publicator(request=req).download_counter_inc(
        req.hash,
        req.bytes_downloaded,
        req.count
    )


@user_is_writeable
def mark_stid_deleted(req):
    """Кладем stid в deleted_stids"""
    if req.stid:
        stid = DeletedStid(stid=req.stid, stid_source=DeletedStidSources.MARK_STID_DELETED)
        DeletedStid.controller.bulk_create([stid], get_size_from_storage=True)


@user_exists_and_not_blocked
def direct_url(req):
    fs = Bus(request=req)
    return fs.direct_url(
        req.uid,
        req.path,
        req.modified,
    )


def public_direct_url(req):
    """
    Формирование прямой ссылки на файл в мульке для Docviewer
    """
    publicator = Publicator(request=req)
    return publicator.public_direct_url(
        req.private_hash,
        req.modified,
        req.uid,
    )


@user_exists_and_not_blocked
def dv_data(req):
    address = Address.Make(req.uid, req.path)
    resource = factory.get_resource(req.uid, address)
    if req.version_id:
        version = ResourceVersionManager.get_version(resource, req.version_id)
        file_stid = version.file_stid
    else:
        file_stid = resource.file_mid()
    return {
        "name": resource.visible_address.name,
        "file_stid": file_stid,
        "mimetype": getattr(resource, 'mimetype', None),
        "media_type": getattr(resource, 'media_type', None),
    }


def public_dv_data(req):
    publicator = Publicator(request=req)
    resource = publicator.get_public_not_blocked_resource(req.private_hash, uid=req.uid)
    return {
        "name": resource.visible_address.name,
        "file_stid": resource.file_mid(),
        "mimetype": getattr(resource, 'mimetype', None),
        "media_type": getattr(resource, 'media_type', None),
    }


@user_exists_and_not_blocked
@user_is_writeable
def trash_append(req):
    fs = Bus(request=req)
    fs.add_to_trash_cleaner_queue(req.uid)
    fs.check_address(req.path, strict=True)
    fs.check_lock(req.path)
    fs.check_trash_lock_for(req.path)
    fs.check_existing_md5(req.uid, req.path, req.md5)
    result = fs.trash_append(req.uid, req.path)
    if is_yateam_root(Address(req.path).path) and user_has_nda_rights(req.uid):
        recreate_yateam_dir(req.uid)
    return result


@user_exists_and_not_blocked
@user_is_writeable
def trash_append_by_resource_id(req):
    resource_ids = [ResourceId.parse(req.resource_id)]
    fs = Bus(request=req)
    fs.add_to_trash_cleaner_queue(req.uid)
    filter_values = {}
    if req.md5:
        filter_values['md5'] = req.md5
    if req.sha256:
        filter_values['sha256'] = req.sha256
    if req.size:
        filter_values['size'] = int(req.size)

    filter_duplicates = not req.append_all
    resources = fs.resources_by_resource_ids_filtered(req.uid, resource_ids,
                                                      enable_service_ids=('/disk', PHOTOUNLIM_AREA_PATH),
                                                      filter_duplicates=filter_duplicates, filter_values=filter_values)
    if not resources:
        raise errors.ResourceNotFound()
    full_paths = []
    for resource in resources:
        if req.files_only and resource.type == 'dir':
            raise errors.FolderDeletionByResourceIdForbiddenError()
        full_paths.append(resource.visible_address.id)

    result = {'list': []}
    for full_path in full_paths:
        fs.check_address(full_path, strict=True)
        fs.check_lock(full_path)
        fs.check_trash_lock_for(full_path)
        fs.check_existing_md5(req.uid, full_path, req.md5)

        result['list'].append(fs.trash_append(req.uid, full_path))
        if is_yateam_root(Address(full_path).path) and user_has_nda_rights(req.uid):
            recreate_yateam_dir(req.uid)

    return result


@user_exists_and_not_blocked
@user_is_writeable
def trash_restore(req):
    fs = Bus(request=req)
    fs.check_lock(req.path)
    fs.check_trash_lock_for(req.path)
    result = fs.trash_restore(req.uid, req.path, new_name=req.name, force=req.force)
    return result


@user_exists_and_not_blocked
@user_is_writeable
def trash_append_file(req):
    """
    Не используется
    """
    fs = Bus(request=req)
    fs.check_address(req.path)
    fs.check_lock(req.path)
    fs.check_trash_lock_for(req.path)
    return fs.trash_append_file(req.uid, req.path)


@user_exists_and_not_blocked
@user_is_writeable
def trash_drop_all(req):
    fs = Bus(request=req)
    fs.check_trash_lock_for(Address.Make(req.uid, '/trash').id)
    if fs.is_trash_append_process_running(req.uid) and FEATURE_TOGGLES_CHECK_TRASH_APPEND_BEFORE_TRASH_DROP:
        raise errors.TrashAppendIsRunningError()
    return fs.trash_drop_all(req.uid, req.uid)


@user_exists_and_not_blocked
@user_is_writeable
def restore_deleted(req):
    fs = Bus(request=req)
    fs.check_address(req.dest)
    fs.check_lock(req.dest)
    return fs.restore_deleted(req.uid, req.path, req.dest, req.force)


@user_exists_and_not_blocked
@user_is_writeable
def set_public(req):
    """
    Публикация ресурса: делаем симлинк и ставим публичный статус
    """
    if is_yateam_root(Address(req.path).path):
        raise errors.YaTeamDirModifyError('Tried to publish YaTeam root directory')

    # В RL нет счетчика
    # if rate_limiter.is_limit_exceeded('cloud_api_user_publish_resource', req.uid):
    #    raise errors.RequestsLimitExceeded()

    fs = Bus(request=req)
    fs.check_address(req.path)
    publicator = Publicator(request=req)
    return publicator.set_public(req.uid, req.path, user_ip=req.user_ip, tld=req.tld)


@user_exists_and_not_blocked
@user_is_writeable
def set_public_settings(req):
    """
    Устанавливаем настройки шаринга
    """
    fs = Bus(request=req)
    fs.check_address(req.path)
    publicator = Publicator(request=req)
    if not req.http_req.data:
        raise errors.BadRequestError('Body required')
    data = from_json(req.http_req.data)
    if not data or not isinstance(data, dict):
        raise errors.BadRequestError('Bad body')
    return publicator.set_public_settings(req.uid, req.path, data)

@user_exists_and_not_blocked
@user_is_writeable
def get_public_settings(req):
    """
    Получаем настройки шаринга
    """
    fs = Bus(request=req)
    fs.check_address(req.path)
    publicator = Publicator(request=req)
    return publicator.get_public_settings(req.uid, req.path)

@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
@allow_empty_disk_info
def get_public_settings_by_hash(req):
    """
    Получаем настройки шаринга по шифрованному ключу
    """
    publicator = Publicator(request=req)
    return publicator.get_public_settings(req.uid, hashed_address=req.private_hash)


@user_exists_and_not_blocked
@user_is_writeable
def set_private(req):
    """
    Убирание публичности ресурса: снимаем статус, удаляем симлинк
    """
    fs = Bus(request=req)
    fs.check_address(req.path)
    publicator = Publicator(request=req)
    return publicator.set_private(req.uid, req.path, return_info=req.return_info)


@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
@allow_empty_disk_info
@slave_read_ok
def public_info(req):
    """
    Получение информации о публичном файле
    """
    publicator = Publicator(request=req)
    return publicator.public_info(
        req.private_hash,
        req.uid,
        password_info=dict(
            password=req.request_headers.get(PUBLIC_PASSWORD_HEADER, None),
            token=req.request_headers.get(PUBLIC_PASSWORD_TOKEN_HEADER, None)
        )
    )


@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
@allow_empty_disk_info
@slave_read_ok
def public_fulltree(req):
    """
    Получение полного дерева публичной папки
    """
    publicator = Publicator(request=req)
    return publicator.public_fulltree(
        req.private_hash,
        req.deep_level,
        req.uid,
        password_info=dict(
            password=req.request_headers.get(PUBLIC_PASSWORD_HEADER, None),
            token=req.request_headers.get(PUBLIC_PASSWORD_TOKEN_HEADER, None)
        )
    )


@enable_secondary_preferred_if(FEATURE_TOGGLES_SECONDARY_PREFERRED_ENABLED)
@allow_empty_disk_info
@slave_read_ok
def public_content(req):
    """
    Получение полного дерева публичной папки
    """
    load_source_ids = req.meta and 'source_ids' in req.meta
    publicator = Publicator(request=req)
    result = publicator.public_content(
        req.private_hash,
        req.uid,
        load_source_ids=load_source_ids,
        password_info=dict(
            password=req.request_headers.get(PUBLIC_PASSWORD_HEADER, None),
            token=req.request_headers.get(PUBLIC_PASSWORD_TOKEN_HEADER, None)
        )
    )
    if req.meta and 'user' in req.meta:
        base_folder_resource = req.form.model
        uid = base_folder_resource.visible_address.uid
        base_folder_resource.meta['user'] = Passport().public_userinfo(uid)
    return result


@allow_empty_disk_info
@slave_read_ok
def search_public_list(req):
    publicator = Publicator(request=req)
    public_addr = PublicAddress(req.private_hash)
    resource = publicator.get_fully_public_resource(public_addr, req.uid)

    uid = resource.visible_address.uid
    fields = list(set(req.search_meta.split(',') + ['path']))
    if isinstance(resource, MPFSFolder):
        if not resource.loaded:
            resource.load()
        mpfs_ordered_resources = itertools.chain([resource], resource.children_items['folders'],
                                                 resource.children_items['files'])
        file_ids = [i.meta['file_id'] for i in mpfs_ordered_resources]
        search_result = SearchDB().resources_info_by_file_ids(
            uid,
            file_ids,
            fields=fields
        )
        for item in search_result['items']:
            relative_addr = PublicAddress.MakeRelative(
                public_addr.hash,
                resource.visible_address.path,
                item['path']
            )
            item['id'] = relative_addr.id
            item['path'] = relative_addr.path
        return search_result['items']
    elif isinstance(resource, MPFSFile):
        file_id = resource.meta['file_id']
        search_result = SearchDB().resources_info_by_file_ids(
            uid,
            [file_id],
            fields=fields
        )
        if len(search_result['items']) != 1:
            raise errors.ResourceNotFound('Search result: %s' % search_result)
        result = search_result['items'][0]
        result['id'] = public_addr.id
        result['path'] = public_addr.path
        return result
    else:
        raise NotImplementedError()


@allow_empty_disk_info
@slave_read_ok
def public_url(req):
    """
    Получение ссылки на публичный файл
    """
    publicator = Publicator(request=req)
    return publicator.public_url(
        req.private_hash,
        req.inline,
        req.uid,
        req.check_blockings,
        password_info=dict(
            password=req.request_headers.get(PUBLIC_PASSWORD_HEADER, None),
            token=req.request_headers.get(PUBLIC_PASSWORD_TOKEN_HEADER, None)
        )
    )


@allow_empty_disk_info
@user_exists_and_not_blocked
@user_is_writeable
def public_notification(req):
    """
    Отправка email
    """
    publicator = Publicator(request=req)
    return publicator.public_notification(
        req.uid,
        req.path,
        req.emails,
        req.message,
        req.locale,
    )


@allow_empty_disk_info
@user_exists_and_not_blocked
@user_is_writeable
def public_copy(req):
    """
    Копирование опубликованного файла к себе в Диск
    """
    user.Check(req.uid, type='standart')
    publicator = Publicator(request=req)
    return publicator.grab(
        req.uid,
        req.private_hash,
        req.name,
        save_path=req.save_path,
        password_info=dict(
            password=req.request_headers.get(PUBLIC_PASSWORD_HEADER, None),
            token=req.request_headers.get(PUBLIC_PASSWORD_TOKEN_HEADER, None)
        )
    )


@allow_empty_disk_info
@user_exists_and_not_blocked
@user_is_writeable
def async_public_copy(req):
    """
    Асинхронное копирование опубликованного чего угодно себе в Диск
    """
    user.Check(req.uid, type='standart')
    publicator = Publicator(request=req)
    return publicator.async_grab(
        req.uid,
        req.private_hash,
        req.name,
        save_path=req.save_path,
        password_info=dict(
            password=req.request_headers.get(PUBLIC_PASSWORD_HEADER, None),
            token=req.request_headers.get(PUBLIC_PASSWORD_TOKEN_HEADER, None)
        )
    )


@allow_empty_disk_info
@user_exists_and_not_blocked
@user_is_writeable
def async_copy_default(req):
    """
    Копирование файла отовсюду к себе в Диск
    """
    user.Check(req.uid, type='standart')
    fs = Bus(request=req)

    new_name = req.name
    source = Address(req.path)

    if source.storage_name == 'mail':
        # адовые костыли для того, чтобы избавиться от похода в поиск из-за возможного лага индексации
        _, attach_identifier = req.path.split('/file:')
        mid, hid = attach_identifier.split(':')
        source_name = MailTVM().get_mime_part_name(uid=req.uid, mid=mid, hid=hid)
        if source_name is None:
            raise ResourceNotFound()
        source_object = MailObject()
        source_object.meta = {'mid': mid, 'hid': hid}
        source_object.name = source_name
    else:
        source_object = factory.get_resource(req.uid, source)

    locale = user.User(req.uid).get_supported_locale()
    incoming_address = fs.get_downloads_address(req.uid, locale)

    if source.storage_name == 'mail':
        # для почтовых аттачей мы пока не знаем откуда получить size кроме поиска
        # но в него ходить не хотим из-за возможного лага индексации
        # проверку доступного места сделает потом кладун в колбэке
        free_space = None
    else:
        free_space = fs.check_available_space(
            req.uid,
            incoming_address.id,
            required_space=source_object.get_size()
        )

    fs.mksysdir(req.uid, type='downloads', locale=locale)

    target = Address(incoming_address.id + source_object.name)
    if new_name:
        target = target.rename(new_name)
    parent_folder = factory.get_resource(req.uid, target.get_parent())
    parent_folder.check_rw()
    target = fs.autosuffix_address(target)
    fs.check_lock(target.id)

    subtype = '%s_%s' % (source.storage_name, target.storage_name)
    odata = {
        'target': target.id,
        'source': source.id,
        'connection_id': req.connection_id,
        'force': req.force,
        'set_public': req.public,
        'force_djfs_albums_callback': req.force_djfs_albums_callback,
    }
    if source.storage_name != 'mail':
        odata.update({'free_space': free_space})

    operation = manager.create_operation(
        req.uid,
        'copy',
        subtype,
        odata=odata,
        source_object=source_object,
    )
    return {'oid': operation.id, 'type': operation.type, 'at_version': operation.at_version}


@slave_read_ok
@user_exists_and_not_blocked
def active_operations(req):
    return list(manager.get_active_operations(req.uid, show_hidden=req.show_hidden))


@user_exists_and_not_blocked
@user_is_writeable
def bulk_action(req):
    """
    Групповое выполнение операций
    """
    operation = manager.create_operation(
        req.uid,
        'bulk',
        'filesystem',
        odata=dict(
            cmd=req.cmd,
            callback=req.callback
        ),
    )
    return {
        'oid': operation.id,
        'type': operation.type,
        'at_version': operation.at_version
    }


def version(req):
    return mpfs.__version__


def ping(req):
    return 'pong'


def ping_slb(req):
    return 'pong'


@user_exists_and_not_blocked
def inspect(req):
    fs = Bus(request=req)
    return fs.inspect(req.uid, req.path)


@user_exists_and_not_blocked
@user_is_writeable
def recount(req):
    fs = Bus(request=req)
    return fs.ask_space_counters_recount(req.uid)


@user_exists_and_not_blocked
@user_is_writeable
def break_counters(req):
    db = mpfs.engine.process.dbctl().database()

    if req.type == 'group':
        groups = db.groups.get_all(owner=req.uid)
        for gr in groups:
            gr['size'] = random.randint(1, 10000)
            db.groups.put_one(gr)
        return
    elif req.type == 'trash':
        db.disk_info.put(req.uid, '/trash_size', random.randint(1, 10000))


@user_exists_and_not_blocked
@user_is_writeable
def share_create_group(req):
    """
    Создание общей группы для каталога
    """
    return ShareProcessor(connection_id=req.connection_id).create_group(req.uid, req.path)


@user_exists_and_not_blocked
@user_is_writeable
def share_invite_user(req):
    if not HANDLE_SHARE_INVITE_USER_RL.check_limit_and_increment_counter(req.uid):
        raise errors.RequestsLimitExceeded429()

    sp = ShareProcessor(connection_id=req.connection_id)

    operation = sp.invite_to_group(
        req.gid,
        req.group_path,
        req.uid,
        req.universe_login,
        req.universe_service,
        req.locale,
        req.user_avatar,
        req.user_name,
        rights=int(req.rights),
        auto_accept=req.auto_accept,
        user_ip=req.ip or req.request_headers.get('X-Real-Ip'),
    )

    result = {
        'oid': operation.id,
        'type': operation.type,
        'gid': operation.data['gid'],
        'at_version': operation.at_version
    }

    if 'hash' in operation.data:
        result['hash'] = operation.data['hash']

    return result


def share_invite_info(req):
    invite_info = ShareProcessor().invite_info(req.user_uid, req.hsh)
    gid = invite_info.pop('gid')
    if req.include_files_count:
        group = Group.load(gid=gid)
        resource_id = group.get_folder().resource_id
        r = SearchDB().folder_size(uid=group.owner, path=group.path, resource_id=resource_id)
        invite_info['files_count'] = int(r['files_count'])
    return invite_info


def share_folder_info(req):
    return ShareProcessor().folder_info(req.uid, req.gid)


@user_exists_and_not_blocked
@user_is_writeable
def share_activate_invite(req):
    sp = ShareProcessor(request=req, connection_id=req.connection_id)
    return sp.activate_group_invite(req.uid, req.hsh)


@user_exists_and_not_blocked
@user_is_writeable
def share_change_rights(req):
    group = Group.load(gid=req.gid)
    # только владелец может менять права
    if group.owner != req.uid:
        raise share_errors.GroupNoPermit()
    share_processor = ShareProcessor(connection_id=req.connection_id)
    try:
        if req.user_uid:
            member_user = User(req.user_uid)
            share_processor.change_member_rights(member_user, group, req.rights)
        else:
            share_processor.change_invite_rights(req.universe_login, req.universe_service, group, req.rights)
    except share_errors.ShareRightsNotChanged:
        # если права не сменились - пофиг. Историческое поведение.
        pass


@user_exists_and_not_blocked
@user_is_writeable
def share_leave_group(req):
    return ShareProcessor(connection_id=req.connection_id).leave_group(
        req.uid,
        req.gid
    )


@user_exists_and_not_blocked
@user_is_writeable
def share_kick_user(req):
    return ShareProcessor(connection_id=req.connection_id).kick_user(
        req.uid,
        req.gid,
        req.user_uid,
    )


class ShareUsersInGroupIterationKey(object):
    LIMIT = 1000

    def __init__(self, offset, limit, stop_iteration):
        self.offset = offset
        self.limit = limit
        self.stop_iteration = stop_iteration

    @classmethod
    def parse(cls, value):
        if value == '':
            return cls(0, cls.LIMIT, False)
        elif value is None:
            return cls(0, None, False)
        elif value.isdigit():
            offset = int(value)
            assert offset >= 0
            return cls(offset, cls.LIMIT, False)
        else:
            raise ValueError("%s" % value)

    def serialize(self):
        if self.stop_iteration:
            return None
        return str(self.offset)


@user_exists_and_not_blocked
def share_users_in_group(req):
    iteration_key = ShareUsersInGroupIterationKey.parse(req.iteration_key)
    group_entities = ShareProcessor.list_users_in_group(
        req.gid, req.uid,
        offset=iteration_key.offset, limit=iteration_key.limit,
        exclude_b2b=req.exclude_b2b
    )
    formated_entities = ShareProcessor.format_group_entities(group_entities, extend_from_passport=True)
    if iteration_key.limit is None or len(group_entities) < iteration_key.LIMIT:
        next_iteration_key = ShareUsersInGroupIterationKey(None, None, True)
    else:
        next_offset = iteration_key.offset + ShareUsersInGroupIterationKey.LIMIT
        next_iteration_key = ShareUsersInGroupIterationKey(next_offset, ShareUsersInGroupIterationKey.LIMIT, False)
    return {'users': formated_entities, 'iteration_key': next_iteration_key.serialize()}


@user_exists_and_not_blocked
def share_uids_in_group(req):
    """Возвращает список пользователей, имеющих доступ к расшаренной папке по `resource_id` или `file_id` и `owner_uid`

    Принимаемые GET-параметры:
    * uid [обязательный] - идентификатор пользователя, из под которого происходит вызов.
    * file_id [необязательный] - идентификатор файла.
    * owner_uid [необязательный] - владелец файла.
    * resource_id [необязательный] - идентификатор ресурса
    * offset [необязательный, значение по умолчанию: 0] - смещение от начала выборки.
    * amount [необязательный, значение по умолчанию: 100000] - максимальное количество возвращаемых пользователей.
    """
    fs = Bus(request=req)
    if req.resource_id:
        resource_id = ResourceId.parse(req.resource_id)
    elif req.file_id and req.owner_uid:
        resource_id = ResourceId(req.owner_uid, req.file_id)
    else:
        raise errors.BadArguments()

    resources = fs.resources_by_resource_ids(req.uid, [resource_id])
    if not resources:
        raise errors.ResourceNotFound()
    resource = resources.pop()
    if not isinstance(resource, (SharedResource, GroupResource)):
        raise share_errors.GroupNotFound()
    group_entities = ShareProcessor.list_users_in_group(
        resource.group.gid, None,
        offset=req.offset, limit=req.amount,
        exclude_b2b=False
    )
    formated_entities = ShareProcessor.format_group_entities(group_entities, extend_from_passport=False)
    return {'users': formated_entities}


@user_exists_and_not_blocked
@user_is_writeable
def share_change_group_owner(req):
    return ShareProcessor(connection_id=req.connection_id).change_owner(
        req.uid,
        req.gid,
        req.user_uid
    )


@user_exists_and_not_blocked
def share_list_all_folders(req):
    return ShareProcessor().list_all_folders(req.uid)


@user_exists_and_not_blocked
def share_list_joined_folders(req):
    return ShareProcessor().list_joined_folders(req.uid)


@user_exists_and_not_blocked
def share_list_owned_folders(req):
    return ShareProcessor().list_owned_folders(req.uid)


@slave_read_ok
@user_exists_and_not_blocked
def share_list_not_approved_folders(req):
    return ShareProcessor().list_not_approved_folders(req.uid)


@user_exists_and_not_blocked
@user_is_writeable
def share_unshare_folder(req):
    return ShareProcessor(connection_id=req.connection_id).unshare_folder(
        req.uid,
        req.gid
    )


@user_exists_and_not_blocked
@user_is_writeable
def share_reject_invite(req):
    return ShareProcessor(connection_id=req.connection_id).reject_invite(
        req.uid,
        req.hsh
    )


@user_exists_and_not_blocked
@user_is_writeable
def share_remove_invite(req):
    return ShareProcessor(connection_id=req.connection_id).remove_invite(
        req.uid,
        req.gid,
        req.universe_login,
        req.universe_service
    )


@user_exists_and_not_blocked
@user_is_writeable
def share_b2b_synchronize_access(req):
    operation = manager.create_operation(
        req.uid,
        'share',
        'b2b_synchronize_access',
        odata={'gid': req.gid}
    )
    return {
        'oid': operation.id,
        'type': operation.type
    }


@user_exists_and_not_blocked
def reindex_search(req):
    mediatypes = filter(lambda x: x, req.mediatype.split(','))

    return DiskDataIndexer().push_reindex(
        req.uid,
        index_body=req.index_body,
        index_type=req.index_type,
        mediatype=mediatypes,
        area='search',
        force=req.force
    )


@user_exists_and_not_blocked
def start_reindex_for_quick_move(req):
    if is_reindexed_for_quick_move(req.uid):
        raise UserAlreadyReindexedForQuickMove()

    if not mpfs.engine.process.usrctl().is_user_in_postgres(req.uid):
        raise UserCannotBeReindexedForQuickMove()

    SearchIndexer().start_reindex_for_quick_move(req.uid)


@user_exists_and_not_blocked
def check_reindexed_for_quick_move(req):
    if not mpfs.engine.process.usrctl().is_user_in_postgres(req.uid):
        return {'is_reindexed': False}
    return {'is_reindexed': is_reindexed_for_quick_move(req.uid)}


@user_exists_and_not_blocked
def reindex_for_quick_move_callback(req):
    set_reindexed_for_quick_move(req.uid)


# на время переноса ярушки закрываем проверку по существованию юзера
# @user_exists_and_not_blocked
@user_is_writeable
def mksysdir(req):
    fs = Bus(request=req)

    if req.type in SYSTEM_ATTACH_SYS_FOLDERS:
        if user.NeedInit(req.uid, type='attach'):
            user.Create(req.uid, type='attach')

    locale = user.User(req.uid).get_supported_locale()
    if req.type in constants.SOCIAL_PROVIDERS_DIRS:
        # создаем локализованную папку /disk/Social Networks/ в случае если её нет
        social_networks_base_path = constants.DEFAULT_FOLDERS['social'][locale]
        try:
            factory.get_resource(req.uid, social_networks_base_path)
        except ResourceNotFound:
            fs.mksysdir(req.uid, 'social', locale)

    return fs.mksysdir(req.uid, req.type, locale)


def add_sony_tablet_serial(req):
    import mpfs.core.user.devices.sony

    mpfs.core.user.devices.sony.add_tablet_serial(req.serial)


def remove_sony_tablet_serial(req):
    import mpfs.core.user.devices.sony

    mpfs.core.user.devices.sony.remove_tablet_serial(req.serial)


@user_is_writeable
def fail_operations(req):
    """
    Удалить все операции
    """
    manager.fail_operations(req.uid)


def hardlink(req):
    """
    Выдаст stid файла, если найдет его по md5-sha256-size
    https://jira.yandex-team.ru/browse/CHEMODAN-12112
    """
    fs = Bus(request=req)
    h = fs.hardlink(req.md5, req.size, req.sha256, hid=req.hid)
    return h.file_part()


@user_exists_and_not_blocked
@user_is_writeable
def aviary_render(req):
    """Редактирование изображения через Aviary"""
    fs = Bus(request=req)
    resource = fs.resource(req.uid, req.path)
    operation = manager.create_operation(
        req.uid,
        'aviary',
        'aviary_render',
        odata=dict(
            resource=resource,
            actionlist=req.actionlist,
            time_suffix=req.time_suffix,
            override_url=req.override_url,
            connection_id=req.connection_id,
            callback=req.callback,
        )
    )

    return {
        'oid': operation.id,
        'type': operation.type,
        'at_version': operation.at_version
    }


@user_exists_and_not_blocked
def aviary_preview_url(req):
    """Получение ссылки на превью для авиари - по ней превью можно скачать без авторизации"""
    fs = Bus(request=req)
    resource = fs.resource(req.uid, req.path)
    return {
        'preview': resource.get_aviary_preview_url()
    }


@user_exists_and_not_blocked
def aviary3_image_info(req):
    """
    Возвращает данные изображения для рендеринга в Aviary:
      * ссылку на превью;
      * ссылку на оригинал;
      * (не реализовано) размеры оригинала.
    """
    fs = Bus(request=req)
    resource = fs.resource(req.uid, req.path)
    return {
        'preview': resource.get_aviary_preview_url(),
        'orig': resource.get_aviary_orig_url()
    }


@user_exists_and_not_blocked
@user_is_writeable
def aviary3_save_image(req):
    """
    Создает операцию сохранения изображения, находящегося по переданной ссылке.
    """
    fs = Bus(request=req)
    resource = fs.resource(req.uid, req.path)
    target_path = CopyAviaryDisk.format_target_path(req.uid, resource.visible_address.path, req.filename_suffix)
    odata = dict(
        uid=req.uid,
        target=Address.Make(req.uid, target_path).id,
        callback=req.callback,
        connection_id=req.connection_id,
        service_file_url=req.url,
        provider='aviary',
        force=True,
        exclude_orientation=True
    )
    odata['mulca-id'] = resource.info()['this']['meta']['file_mid']
    operation = manager.create_operation(
        req.uid,
        'social_copy',
        'aviary_disk',
        odata=odata
    )
    return {
        'oid': operation.id,
        'type': operation.type,
        'at_version': operation.at_version
    }


@user_is_writeable
def pushthelimitsto_10gb(req):
    """Добивает размер диска старого пользователя до 10ГБ подпиской на бонусные сервисы."""
    pushthelimits.pushto_10gb(req.uid)
    return {}


def update_file_hash(req):
    """
        Пересчет всех хэшей файла.
    """
    odata = {
        'path': req.path,
    }
    if req.md5 and req.sha256 and req.size is not None:
        fs = Bus(request=req)
        fs.check_file_hash(req.uid, req.path, req.md5, req.sha256, req.size)
        odata['md5'] = req.md5
        odata['sha256'] = req.sha256
        odata['size'] = req.size
    operation = manager.create_operation(req.uid, 'system', 'update_file_hash', odata=odata)
    return {
        'oid': operation.id,
        'type': operation.type
    }


def bulk_check_stids(req):
    if rate_limiter.is_limit_exceeded(RATE_LIMITER_GROUP_NAMES_BULK_CHECK_STIDS, '0'):
        raise errors.RequestsLimitExceeded()

    if not req.http_req.data:
        raise errors.BadRequestError('Body required')
    data = from_json(req.http_req.data)
    if not data or not isinstance(data, dict):
        raise errors.BadRequestError('Bad body')

    stids = data.get('stids')
    if not stids or not isinstance(stids, list) or len(stids) > 50:
        raise errors.BadRequestError('Bad "stids" field')
    if not stids:
        checking_stids = []
    else:
        db_checker = DbChecker()
        checking_stids = [CheckingStid(s) for s in stids]
        db_checker.is_stids_in_db(checking_stids)
    return {'items': [{'stid': s.stid, 'in_db': s.is_stid_in_db} for s in checking_stids]}


def list_public(req):
    """
    Получить список публичных ресурсов пользователя

    Query string аргументы:
      * uid [обязательный]
      * type -- Фильтр по типу публичных ресурсов. Допустимые значения file или dir.
      * amount
      * offset
    """
    amount = req.amount
    offset = req.offset
    resources = []

    if amount == 0:
        return []

    def symlinks2resources(symlinks):
        addresses = []
        for symlink in symlinks:
            if symlink:
                address = Address(symlink.target_addr())
                addresses.append(address)

        if addresses:
            return factory.get_resources(req.uid, addresses, available_service_ids=['/disk'])
        return []

    symlinks = Symlink.list_all(uid=req.uid, reverse=False, order_by='data.ctime')
    symlink_address_id_to_symlink_map = {symlink.address.id: symlink for symlink in symlinks if symlink}
    resources = symlinks2resources(symlinks)
    if (req.meta and 'public_time' in req.meta) or not req.meta:
        for resource in resources:
            if 'symlink' in resource.meta:
                resource_symlink_address_id = resource.meta['symlink']
                resource_symlink = symlink_address_id_to_symlink_map.get(resource_symlink_address_id)
                if resource_symlink:
                    resource.meta['public_time'] = int(float(resource_symlink.ctime))  # ctime может быть '123.56'

    if req.type in ('dir', 'file'):
        resources = [r for r in resources if r.type == req.type]
    if amount:
        resources = resources[offset:offset + amount]
    elif offset:
        resources = resources[offset:]

    for r in resources:
        r.set_request(req)
    return resources


def show_settings(req):
    """
    Отдаёт текущие настройки MPFS.
    """
    return settings.as_dict()


def enable_unlimited_autouploading(request):
    User(request.uid).enable_unlimited_autouploading()
    try:
        smartcache.initialize_photostream(request.uid)
    except APIError:
        # Возвращаем 200 при неудачных попытках сынициализировать фотосрез
        pass


def disable_unlimited_autouploading(request):
    User(request.uid).disable_unlimited_autouploading()


def set_unlimited_autouploading(request):
    video_status = request.optional_params.get('video_status')
    video_reason = request.optional_params.get('video_reason')
    photo_status = request.optional_params.get('photo_status')
    user = User(request.uid)
    if video_status and not user.is_unlimited_video_autouploading_allowed():
        unlimited_autouploding = user.get_unlimited_autouploading()
        raise errors.ForbiddenVideoUnlim(
            title=to_json({'reason': unlimited_autouploding['unlimited_video_autouploading_reason']})
        )

    # Если пользователь попал в эксперимент
    if photo_status and not user.is_unlimited_photo_autouploading_allowed():
        raise errors.ForbiddenPhotoUnlim()

    unlimited_autouploding = user.set_unlimited_autouploading(video_status, video_reason, photo_status)
    return {'unlimited_video_autoupload_enabled': unlimited_autouploding['unlimited_video_autouploading_enabled'],
            'unlimited_video_autoupload_reason': unlimited_autouploding['unlimited_video_autouploading_reason'],
            'unlimited_photo_autoupload_enabled': unlimited_autouploding['unlimited_photo_autouploading_enabled'],
            'unlimited_photo_autoupload_allowed': unlimited_autouploding['unlimited_photo_autouploading_allowed'],
            'unlimited_video_autoupload_allowed': unlimited_autouploding['unlimited_video_autouploading_allowed']}


@user_exists_and_not_blocked
def is_user_in_unlim_experiment(request):
    _, is_rkub = user_unlim_experiment_data(request.uid)
    return {'is_user_in_unlim_experiment': 1, 'is_RKUB': int(is_rkub)}


@user_exists_and_not_blocked
def set_ps_billing_feature(request):
    feature_name = request.feature_name
    value = bool(int(request.value))
    uid = request.uid
    user = User(uid)
    if feature_name == ONLY_OFFICE_FEATURE:
        if value and not user.b2b_key:
            mpfs_queue.put({'uid': uid}, 'update_b2b_key')
        user.set_only_office_enabled(value)
    elif feature_name == ONLINE_EDITOR_FEATURE:
        user.set_online_editor_enabled(value)
    elif feature_name == ADVERTISING_FEATURE:
        user.set_advertising_enabled(value)
    elif feature_name == PUBLIC_SETTINGS_FEATURE:
        user.set_public_settings_enabled(value)
    elif feature_name == UNLIMITED_PHOTO_AUTOUPLOADING_FEATURE:
        user.set_unlimited_photo_autouploading_allowed(value)
    elif feature_name == UNLIMITED_VIDEO_AUTOUPLOADING_FEATURE:
        user.set_unlimited_video_autouploading_allowed(value)
    else:
        raise BadRequestError("Unknown feature_name")


@user_exists_and_not_blocked
def set_user_overdraft_date(request):
    uid = request.uid
    user = User(uid)
    overdraft_date = request.overdraft_date
    try:
        overdraft_date = datetime.datetime.strptime(overdraft_date, '%Y-%m-%d').date()
    except ValueError:
        raise BadRequestError("Unknown format of overdraft_date, should be 'YYYY-MM-DD'")
    user.set_overdraft_info(overdraft_date, force=bool(request.force))


def generate_zaberun_url(request):
    zaberun = Zaberun()
    if request.url_type == 'file':
        generated_url = zaberun.generate_file_url(request.uid, request.stid, request.file_name,
                                                  **request.optional_params)
    elif request.url_type == 'preview':
        generated_url = zaberun.generate_preview_url(request.uid, request.stid, request.file_name,
                                                     **request.optional_params)
    else:
        raise BadRequestError(extra_msg='Unsupported url type %s' % request.url_type)
    return {'zaberun_url': generated_url}


def _check_folder_content_media_type(resource):
    folder_dao = FolderDAO()
    # для общих папок в resource.address хранится адрес владельца, поэтому тут используем его
    addr = resource.address
    num_subfolders = folder_dao.count_subfolders(addr)
    if num_subfolders > 0:
        only_image_and_video = False
    else:
        num_subfiles = folder_dao.count_subfiles(addr)
        # если файлов очень много, то не нагружаем базу, а сразу отдаем нет
        if num_subfiles > 2000:
            only_image_and_video = False
        else:
            only_image_and_video = not folder_dao.has_another_media_type_files(addr, ('image', 'video'))

    return {
        "only_image_and_video": only_image_and_video
    }


@user_exists_and_not_blocked
def check_folder_content_media_type(req):
    resource = factory.get_resource(req.uid, req.path)
    return _check_folder_content_media_type(resource)


def public_check_folder_content_media_type(req):
    publicator = Publicator(request=req)
    resource = publicator.get_public_not_blocked_resource(
        req.private_hash,
        req.uid,
        password_info=dict(
            password=req.request_headers.get(PUBLIC_PASSWORD_HEADER, None),
            token=req.request_headers.get(PUBLIC_PASSWORD_TOKEN_HEADER, None)
        )
    )
    return _check_folder_content_media_type(resource)


def force_snapshot(req):
    UserDAO().force_user_to_get_snapshot(req.uid)


def telemost_fos(req):
    check_fos_request_is_correct(req)
    if req.uid:
        uid = req.uid
        user_info = passport.userinfo(req.uid)
        login = user_info['login']
        sender_name = user_info.get('username', '')
    else:
        uid = sender_name = login = UNAUTHORIZED_USER_NAME

    resp_body = from_json(req.http_req.data)
    support_text = resp_body.get('fos_support_text')
    debug_info = resp_body.get('debug_info', '')

    subject = TELEMOST_FOS_SUBJECT_TEMPLATE % {
        'os_version': req.os_version,
        'app_version': req.app_version,
        'fos_subject': req.subject
    }
    body = TELEMOST_FOS_BODY_TEMPLATE % {
        'support_text': support_text,
        'login': login,
        'uid': uid,
        'app_version': req.app_version,
        'os_version': req.os_version,
        'debug_info': debug_info,
    }
    data = {
        'email_to': TELEMOST_FEEDBACK_FOS_EMAILS[req.recipient_type],
        'template_name': FEEDBACK_FOS_TEMPLATE,
        'sender_name': sender_name,
        'sender_email': req.reply_email,
        'template_args': {
            'subject': subject,
            'body': body,
        },
    }
    mpfs_queue.put(data, 'send_email')


# ===========================================================================
# Тэги

def elements_for_tag(req):
    fs = Bus(request=req)
    return fs.elements_for_tag(req.uid, req.path, req.data)


def tags_for_element(req):
    fs = Bus(request=req)
    return fs.tags_for_element(req.uid, req.path)


def tags_in_folder_list(req):
    fs = Bus(request=req)
    return fs.tags_in_folder_list(req.uid, req.path)


def tags_in_folder_tree(req):
    fs = Bus(request=req)
    return fs.tags_in_folder_tree(req.uid, req.path)


def tags_set(req):
    fs = Bus(request=req)
    return fs.tags_set(req.uid, req.path, req.scope, req.data)


def tags_photo_timeline(req):
    fs = Bus(request=req)
    return fs.tags_photo_timeline(req.uid, req.path, req.system_tags, req.user_tags)


def tags_photo_list(req):
    fs = Bus(request=req)
    return fs.tags_photo_list(req.uid, req.path, req.filters, req.system_tags, req.user_tags)


def elements_in_folder_list(req):
    fs = Bus(request=req)
    return fs.elements_in_folder_list(req.uid, req.path, req.data)


def elements_in_folder_tree(req):
    fs = Bus(request=req)
    return fs.elements_in_folder_tree(req.uid, req.path, req.data)


def tags_set_photo_all(req):
    fs = Bus(request=req)
    return fs.tags_set_photo_all(req.uid, req.path)

# ===========================================================================
