# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Вспомогательные механизмы для крон-скриптов

"""
import random
import time

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common import errors
from mpfs.core.metastorage.control import billing_locks
from mpfs.core.billing.processing import billing, notify, admin

log = mpfs.engine.process.get_default_log()


class CronTask(object):
    HOSTNAME = mpfs.engine.process.hostname()
    call_map = {
        'billing': billing.push_billing_commands,
        'notify': notify.push_notify_commands,
        'orders': admin.archive_lost_orders,
    }

    def __init__(self, cron_task_name):
        if cron_task_name not in settings.billing_cron_tasks:
            raise NotImplementedError('Not such cron task in config: "%s"' % cron_task_name)
        if cron_task_name not in self.call_map:
            raise NotImplementedError('Not such cron task "CronTask.call_map": "%s"' % cron_task_name)
        conf = settings.billing_cron_tasks[cron_task_name]
        self.cron_task_name = cron_task_name
        self.check_sleep = conf['check_sleep']
        self.lock_name = conf['lock_name']
        self.lock_delay = conf['lock_delay']

    def run(self):
        if not settings.billing_cron_tasks['enabled']:
            log.info('Billing cron tasks turned off')
            return

        self.random_sleep()
        self.set_lock()
        self.call_map[self.cron_task_name]()
        time.sleep(self.check_sleep)
        self.release_lock()

    def random_sleep(self):
        time.sleep(random.randint(0, self.check_sleep))

    def set_lock(self):
        lock = self.lock_name
        delay = self.lock_delay
        host = self.HOSTNAME
        try:
            billing_locks.set_lock(lock, host, delay)
            log.info('set lock %s on %s successfully' % (lock, host))
        except errors.ResourceLocked as e:
            log.error('lock %s already locked' % lock)
            raise e

    def release_lock(self):
        lock = self.lock_name
        host = self.HOSTNAME
        try:
            billing_locks.release_lock(lock, host)
        except errors.ResourceLocked:
            log.error('lock %s already locked' % lock)
        else:
            log.info('released lock %s on %s successfully' % (lock, host))
