# -*- coding: utf-8 -*-
from mpfs.dao.base import BaseDAOItem, BaseDAO, PostgresBaseDAOImplementation
from mpfs.dao.fields import UidField, IntegerField, DateTimeField
from mpfs.dao.migration.migration import BaseCommonCollectionMigration
from mpfs.metastorage.postgres.queries import (
    SQL_UPDATE_OR_CREATE_OVERDRAFT_INFO,
    SQL_SELECT_OVERDRAFT_INFO,
    SQL_SELECT_All_OVERDRAFT_INFO,
    SQL_DELETE_USER_FROM_OVERDRAFT, SQL_UPDATE_ITERATE_FIELD,
)
from mpfs.metastorage.postgres.schema import overdraft


class OverdraftDAOItem(BaseDAOItem):
    mongo_collection_name = 'overdraft'
    postgres_table_obj = overdraft
    is_sharded = False
    is_migrated_to_postgres = True

    @classmethod
    def get_postgres_primary_key(cls):
        return 'uid'

    uid = UidField(mongo_path='_id', pg_path=overdraft.c.uid)
    ctime = DateTimeField(mongo_path='overdraft_date', pg_path=overdraft.c.overdraft_date, default_value=None)
    mtime = DateTimeField(mongo_path='last_email_date', pg_path=overdraft.c.last_email_date, default_value=None)
    count = IntegerField(mongo_path='last_push_date', pg_path=overdraft.c.last_push_date, default_value=1)


class OverdraftDAO(BaseDAO):
    ALLOWED_ITER_FIELDS = ['last_send_email_handle_date', 'last_send_cleaner_handle_date']

    dao_item_cls = OverdraftDAOItem

    def __init__(self):
        super(OverdraftDAO, self).__init__()
        self._pg_impl = PostgresOverdraftDAOImplementation(self.dao_item_cls)

    def update_or_create(self, uid, overdraft_date, last_email_date=None, last_push_date=None):
        """
        Добавить пользователя в коллекцию на пересчёт счётчиков места
        """
        return self._get_impl(None).update_or_create(uid, overdraft_date, last_email_date, last_push_date)

    def get(self, uid):
        return self._get_impl(None).get(uid)

    def get_all_by_chunks(self, field_for_iter, limit):
        self._check_iter_field(field_for_iter)
        return self._get_impl(None).get_all_by_chunks(field_for_iter, limit)

    def update_iter_field(self, uids, field_for_iter):
        self._check_iter_field(field_for_iter)
        self._get_impl(None).update_iter_field(uids, field_for_iter)

    def delete(self, uid):
        return self._get_impl(None).delete(uid)

    def _check_iter_field(self, field):
        if field not in self.ALLOWED_ITER_FIELDS:
            raise RuntimeError('Field %s not allow for iter, allow %s' % (field, self.ALLOWED_ITER_FIELDS))


class PostgresOverdraftDAOImplementation(PostgresBaseDAOImplementation):
    def update_or_create(self, uid, overdraft_date, last_email_date, last_push_date):
        session = self._get_session()
        with session.begin():
            return session.execute(SQL_UPDATE_OR_CREATE_OVERDRAFT_INFO,
                                   {'uid': uid, 'overdraft_date': overdraft_date, 'last_email_date': last_email_date,
                                    'last_push_date': last_push_date})

    def get(self, uid):
        session = self._get_session()
        with session.begin():
            return session.execute(SQL_SELECT_OVERDRAFT_INFO, {'uid': uid}).fetchone()

    def get_all_by_chunks(self, field_for_iter, limit):
        session = self._get_session()
        with session.begin():
            return session.execute(SQL_SELECT_All_OVERDRAFT_INFO % (field_for_iter, field_for_iter), {'limit': limit}).fetchall()

    def update_iter_field(self, uids, field_for_iter):
        session = self._get_session()
        session.execute(SQL_UPDATE_ITERATE_FIELD % field_for_iter, {'uids': tuple(uids)})

    def delete(self, uid):
        session = self._get_session()
        session.execute(SQL_DELETE_USER_FROM_OVERDRAFT, {'uid': uid})


class OverdraftMigration(BaseCommonCollectionMigration):
    dao_cls = OverdraftDAO
