# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Общие классы сущностей

"""
from copy import copy
from uuid import uuid4

import mpfs.engine.process

import mpfs.common.errors.billing as errors
from mpfs.common.util import generator, ctimestamp
from mpfs.common.static.tags.billing import *

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class Entity(object):
    '''
    Базовая сущность, предоставляющая простые доступы к данным, храним в виде dict
    '''
    def __init__(self, *args, **kwargs):
        self._changed = False
        self._data = {}
        self._removed = set()

    def set(self, key, value):
        if key not in self._data or self._data[key] != value:
            self._data[key] = value
            if self._removed and key in self._removed:
                self._removed.remove(key)
            self._changed = True

    def remove(self, key):
        if self._data.get(key) is not None:
            self._data[key] = None
            self._removed.add(key)
            self._changed = True

    def get(self, key):
        return self._data.get(key)

    def __getattr__(self, *args, **kwargs):
        return self.get(args[0])

    def dict(self):
        return self._data

    def _is_changed(self):
        return self._changed


class StoredEntity(Entity):
    '''
    Хранимая сущность.
    Хранится в виде одной записи в одной коллекции
    '''
    table = None
    pkey = None
    empty = False
    nf_error = errors.BillingEntityNotFound

    def __init__(self, *args, **kwargs):
        super(StoredEntity, self).__init__(*args, **kwargs)

        self._data = kwargs.get('data', {})
        self._id = kwargs.get(self.pkey)

        if not self._data or not self._id:
            if self.pkey in kwargs:
                params = {self.pkey: kwargs[self.pkey]}
            else:
                params = kwargs

            loaded = self.table.get_one(**kwargs)
            if loaded:
                self._id = loaded.get(self.pkey)
                self._data = loaded

        if not self._data and not self.empty:
            raise self.nf_error(self._id)

    def save(self):
        self.touch()
        data = copy(self._data)  # table.put удаляет _id из _data, если оно там есть, а оно нам нужно
        self.table.put(data, remove=self._removed)
        # а дальше надо в _data заполнить правильно 'v' (version), иначе оно не обновляется, но мы ж его скопировали
        # в общем, вся магия происходит в BillingKeyValue.put, а я тут только накрутил костылей над этой магией, чтобы
        # все работало, как раньше:
        self._data.update(data)

    def set(self, key, value, save_to_db=True):
        super(StoredEntity, self).set(key, value)
        if save_to_db and self._is_changed():
            self.save()

    def remove(self, key, save_to_db=True):
        super(StoredEntity, self).remove(key)
        if save_to_db and self._is_changed():
            self.save()

    def touch(self):
        if MTIME in self._data:
            self._data[MTIME] = ctimestamp()

    def delete(self):
        self.table.remove(**{
            self.pkey: self._id,
            'uid': self.uid,
        })
        self._changed = False


class UniqEntity(StoredEntity):
    '''
    Надстройка над базовой сущностью.
    Позволяет вызывать конструктор, указывая primary key сущности
    '''
    pkey = DB_ID
    uniq_key_name = None

    def __init__(self, uniq_key, **kwargs):
        kwargs.update({self.pkey: uniq_key})
        super(UniqEntity, self).__init__(**kwargs)

        setattr(self, self.uniq_key_name, str(getattr(self, self.pkey)))

        if self.uniq_key_name in self._data:
            del self._data[self.uniq_key_name]
        if not self.pkey in self._data:
            self._data[self.pkey] = self._id

    @classmethod
    def GenerateID(classname, *args):
        return generator.md5(*args)

    @classmethod
    def GenerateIntegerID(classname, range):
        return generator.integer(range)

    @classmethod
    def GenerateUuidID(classname):
        return uuid4().hex


class EmptyEntity(UniqEntity):
    '''
    Надстройка над базовой сущностью
    Может не существовать и быть пустой
    '''
    empty = True

    def _is_changed(self):
        return (self._changed and self._data)

    def save(self):
        # если в _data только _id и v - удаляем
        if len(self._data) == 2 and self.pkey in self._data and 'v' in self._data:
            self.delete()
        else:
            super(EmptyEntity, self).save()
