# -*- coding: utf-8 -*-
import collections
import datetime


class PurchaseReceipt(object):
    """
    In-app purchase transactions
    """

    def __init__(self, raw_purchase_receipt):
        self.raw_purchase_receipt = raw_purchase_receipt

    @property
    def product_id(self):
        return self.raw_purchase_receipt["product_id"]

    @property
    def original_transaction_id(self):
        return self.raw_purchase_receipt["original_transaction_id"]

    @property
    def transaction_id(self):
        return self.raw_purchase_receipt["transaction_id"]

    @property
    def purchase_ts(self):
        return int(self.raw_purchase_receipt["purchase_date_ms"]) / 1000

    @property
    def purchase_date(self):
        return datetime.datetime.fromtimestamp(self.purchase_ts)

    @property
    def expires_ts(self):
        if self.is_auto_renewable:
            return int(self.raw_purchase_receipt["expires_date_ms"]) / 1000

    @property
    def expires_date(self):
        if self.is_auto_renewable:
            return datetime.datetime.fromtimestamp(self.expires_ts)

    @property
    def web_order_line_item_id(self):
        return self.raw_purchase_receipt["web_order_line_item_id"]

    @property
    def quantity(self):
        return int(self.raw_purchase_receipt["quantity"])

    @property
    def is_in_intro_offer_period(self):
        if self.is_auto_renewable:
            return self.raw_purchase_receipt["is_in_intro_offer_period"].lower() == "true"

    @property
    def is_trial_period(self):
        if self.is_auto_renewable:
            return self.raw_purchase_receipt["is_trial_period"].lower() == "true"

    @property
    def is_cancellation_transaction(self):
        return "cancellation_date_ms" in self.raw_purchase_receipt

    @property
    def is_auto_renewable(self):
        return "expires_date" in self.raw_purchase_receipt

    def get_period(self):
        return (self.purchase_date, self.expires_date)

    def __repr__(self):
        return "<%s %s>" % (self.__class__.__name__, self.transaction_id)


class AppStoreSubscription(object):
    """
    Contains all data from App Store response associated with subscription (original_transaction_id)
    """

    allow_period_missmatch_delta = datetime.timedelta(minutes=10)

    def __init__(self, original_transaction_id, purchase_receipts, renewal_info):
        self.original_transaction_id = original_transaction_id

        self.purchase_receipts = []
        self.cancellation_receipts = []
        for receipt in purchase_receipts:
            assert receipt.original_transaction_id == original_transaction_id
            assert receipt.is_auto_renewable
            if receipt.is_cancellation_transaction:
                self.cancellation_receipts.append(receipt)
            else:
                self.purchase_receipts.append(receipt)

        self.purchase_receipts.sort(key=lambda x: x.purchase_ts)
        self.cancellation_receipts.sort(key=lambda x: x.purchase_ts)

        assert original_transaction_id == renewal_info["original_transaction_id"]
        self.renewal_info = renewal_info

    def get_active_periods(self):
        result = []
        period_start = None
        period_end = None
        for purchase_receipt in self.purchase_receipts:
            purchase_dt, expires_dt = purchase_receipt.get_period()
            if period_start is None:
                period_start = purchase_dt
                period_end = expires_dt
            elif abs(purchase_dt - period_end) > self.allow_period_missmatch_delta:
                result.append((period_start, period_end))
                period_start = purchase_dt
                period_end = expires_dt
            else:
                period_end = expires_dt
        if period_start:
            result.append((period_start, period_end))
        return result

    def get_lapsed_periods(self):
        active_periods = self.get_active_periods()
        if len(active_periods) < 2:
            return []

        result = []
        prev_active_period = None
        for active_period in active_periods:
            if prev_active_period is None:
                prev_active_period = active_period
                continue
            result.append((prev_active_period[1], active_period[0]))
        return result

    def get_first_purchase_receipt(self):
        return self.purchase_receipts[0]

    def get_latest_purchase_receipt(self):
        return self.purchase_receipts[-1]

    def is_expired(self, on_date=None):
        if on_date is None:
            on_date = datetime.datetime.now()
        return on_date >= self.get_latest_purchase_receipt().expires_date

    def is_still_attempting_to_renew(self):
        if "is_in_billing_retry_period" not in self.renewal_info:
            return True
        return self.renewal_info["is_in_billing_retry_period"].lower() in ("1", "true")

    def __repr__(self):
        return "<%s %s>" % (self.__class__.__name__, self.original_transaction_id)


def app_store_response_parser(raw_app_store_response):
    """
    Converts app store verifyReceipt response to objects.

    Supports only iOS7 style app receipts.
    TODO: add support for one-time products
    :rtype: list[AppStoreSubscription]
    """
    if "latest_receipt_info" not in raw_app_store_response:
        return []

    grouped_purchase_receipts = collections.defaultdict(list)
    for raw_purchase_receipt in raw_app_store_response["latest_receipt_info"]:
        purchase_receipt = PurchaseReceipt(raw_purchase_receipt)
        grouped_purchase_receipts[purchase_receipt.original_transaction_id].append(purchase_receipt)

    grouped_renewal_info = {}
    for renewal_info in raw_app_store_response["pending_renewal_info"]:
        grouped_renewal_info[renewal_info["original_transaction_id"]] = renewal_info

    result = []
    for original_transaction_id, purchase_receipts in grouped_purchase_receipts.items():
        renewal_info = grouped_renewal_info[original_transaction_id]
        result.append(AppStoreSubscription(original_transaction_id, purchase_receipts, renewal_info))
    return result
