# -*- coding: utf-8 -*-
from mpfs.dao.base import BaseDAOItem, BaseDAO, PostgresBaseDAOImplementation, MongoBaseDAOImplementation
from mpfs.dao.fields import StringField, DateTimeField, UidField
from mpfs.dao.migration.migration import BaseCommonCollectionMigration
from mpfs.metastorage.postgres.queries import SQL_INSERT_UNPROCESSED_RECEIPT
from mpfs.metastorage.postgres.schema import unprocessed_receipts


class UnprocessedReceiptDAOItem(BaseDAOItem):
    mongo_collection_name = 'unprocessed_receipts'
    postgres_table_obj = unprocessed_receipts
    is_sharded = False
    is_migrated_to_postgres = False

    id = StringField(mongo_path='_id', pg_path=unprocessed_receipts.c.id)
    syncronization_datetime = DateTimeField(mongo_path='syncronization_datetime', pg_path=unprocessed_receipts.c.syncronization_datetime)
    receipt = StringField(mongo_path='receipt', pg_path=unprocessed_receipts.c.receipt)
    traceback = StringField(mongo_path='traceback', pg_path=unprocessed_receipts.c.traceback)
    uid = UidField(mongo_path='uid', pg_path=unprocessed_receipts.c.uid)


class UnprocessedReceiptDAO(BaseDAO):
    dao_item_cls = UnprocessedReceiptDAOItem

    def __init__(self):
        super(UnprocessedReceiptDAO, self).__init__()
        self._mongo_impl = MongoUnprocessedReceiptDAOImplementation(self.dao_item_cls)
        self._pg_impl = PostgresUnprocessedReceiptDAOImplementation(self.dao_item_cls)

    def save(self, item):
        return self._get_impl(None).save(item)


class MongoUnprocessedReceiptDAOImplementation(MongoBaseDAOImplementation):

    def save(self, item):
        self.update({'_id': item.id}, item.get_mongo_representation(), upsert=True)


class PostgresUnprocessedReceiptDAOImplementation(PostgresBaseDAOImplementation):

    def save(self, item):
        session = self._get_session()
        with session.begin():
            return session.execute(SQL_INSERT_UNPROCESSED_RECEIPT, item.as_raw_pg_dict())


class UnprocessedReceiptMigration(BaseCommonCollectionMigration):
    dao_cls = UnprocessedReceiptDAO
