# -*- coding: utf-8 -*-
import json
import traceback

import mpfs.engine.process
from mpfs.common.errors.billing import (
    BillingInAppStoreIdsOfProductAndRequestMismatch,
    BillingInAppInvalidVerification,
    BillingInAppUidMismatch,
)

from mpfs.common.static.tags.billing import STATE, INAPP_GRACE_PERIOD, INAPP_ACTIVE
from mpfs.common.util import ctimestamp
from mpfs.config import settings
from mpfs.core.billing import Service, BTIME, SID, Client, trust_payment_service, ServiceList, USER_DEFAULT_COUNTRY, \
    Market
from mpfs.core.billing.inapp.unprocessed_receipt import save_failed_receipt_to_db
from mpfs.core.billing.inapp.sync_logic import log_to_billing_log
from mpfs.core.billing.processing.common import provide_paid_service_to_client, simple_delete_service
from mpfs.core.billing import Product
from mpfs.core.billing.processing.repair import recalculate_limit_by_services
from mpfs.core.services.passport_service import passport

default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

BILLING_INAPP_GRACE_PERIOD = settings.billing["inapp_grace_period"]


def provide_inapp_service(client, app_store_sub, receipt, store_id):
    if app_store_sub.is_expired():
        default_log.info("Attempt to provide expired in-app service. Ignoring...")
        return
    product = Product(pid=app_store_sub.get_latest_purchase_receipt().product_id)
    if product.store_id != store_id:
        raise BillingInAppStoreIdsOfProductAndRequestMismatch()
    provided_service = provide_paid_service_to_client(
        client,
        product,
        store_id,
        auto=True,
        bb_time=app_store_sub.get_latest_purchase_receipt().expires_ts,
        receipt=receipt,
        original_transaction_id=app_store_sub.get_latest_purchase_receipt().original_transaction_id,
        state=INAPP_ACTIVE,
    )

    default_log.info(
        "New in-app service provided uid=%(uid)s sid=%(sid)s pid=%(pid)s btime=%(btime)s otid=%(otid)s "
        % {
            "uid": provided_service.uid,
            "sid": provided_service.sid,
            "pid": provided_service.pid,
            "btime": provided_service.btime,
            "otid": provided_service.original_transaction_id,
        }
    )
    return provided_service


def change_pid_of_inapp_service(client, app_store_sub, receipt, store_id, service):
    product = Product(pid=app_store_sub.get_latest_purchase_receipt().product_id)
    provided_service = provide_paid_service_to_client(
        client,
        product,
        store_id,
        auto=True,
        bb_time=service.btime,
        receipt=receipt,
        original_transaction_id=app_store_sub.get_latest_purchase_receipt().original_transaction_id,
        state=service.state,
    )
    simple_delete_service(client, service, send_email=False)

    default_log.info(
        "In-app service pid changed. New: %s. Old: %s."
        % (log_format_service(provided_service), log_format_service(service))
    )
    return provided_service


def update_inapp_service(app_store_sub, service):
    old_btime = service.btime
    service.set(BTIME, app_store_sub.get_latest_purchase_receipt().expires_ts, save_to_db=False)
    service.save()
    default_log.info("In-app service btime changed %s old_btime=%s" % (log_format_service(service), old_btime))
    # filter test subscriptions and grace periods
    if service.btime - old_btime > 10 * 86400:
        log_to_billing_log(service, True, is_prolong=True)
    return service


def process_expired_inapp_service(client, app_store_sub, service):
    cur_time = ctimestamp()

    grace_ended = cur_time - app_store_sub.get_latest_purchase_receipt().expires_ts >= BILLING_INAPP_GRACE_PERIOD
    if grace_ended or not app_store_sub.is_still_attempting_to_renew():
        if grace_ended:
            msg = "Grace period ended"
        else:
            msg = "App Store has stopped attempting to renew the subscription"
        simple_delete_service(client, service)
        default_log.info("In-app service removed. %s. %s" % (msg, log_format_service(service)))
        return

    if service.state != INAPP_GRACE_PERIOD:
        old_btime = service.btime
        service.set(STATE, INAPP_GRACE_PERIOD)
        service.set(BTIME, max(service.btime, app_store_sub.get_latest_purchase_receipt().expires_ts))
        service.save()
        default_log.info(
            "In-app service entered grace period %s old_btime=%s" % (log_format_service(service), old_btime)
        )
        return service


def service_by_otid(otid):
    if not otid:
        return
    services = ServiceList(original_transaction_id=[otid])
    if not services:
        return
    return Service(services[0][SID])


def log_format_service(serivce):
    if not serivce:
        return "(No service)"
    return (
        "Service(uid=%(uid)s, sid=%(_id)s, pid=%(pid)s, state=%(state)s, btime=%(btime)s, otid=%(original_transaction_id)s)"
        % (serivce.dict())
    )


def log_format_app_store_sub(app_store_sub):
    renewal_info = json.dumps(app_store_sub.renewal_info)
    latest_trans = app_store_sub.get_latest_purchase_receipt()
    fmt_latest_trans = json.dumps({"pid": latest_trans.product_id, "expires_date": str(latest_trans.expires_date)})
    return (
        "AppStoreSub(otid=%(otid)s, latest_trans=%(latest_trans)s, renewal_info=%(renewal_info)s, trans_num=%(trans_num)i)"
        % {
            "otid": app_store_sub.original_transaction_id,
            "latest_trans": fmt_latest_trans,
            "renewal_info": renewal_info,
            "trans_num": len(app_store_sub.purchase_receipts),
        }
    )


def process_app_store_sub(client, app_store_sub, receipt, store_id):
    found_service = service_by_otid(app_store_sub.original_transaction_id)

    default_log.info(
        "Processing in-app app store subscription "
        "uid=%s %s %s" % (client.uid, log_format_app_store_sub(app_store_sub), log_format_service(found_service))
    )

    if not found_service:
        return provide_inapp_service(client, app_store_sub, receipt, store_id)

    # for existen subscription use same uid
    client = Client(found_service.uid)

    if found_service.pid != app_store_sub.get_latest_purchase_receipt().product_id:
        found_service = change_pid_of_inapp_service(client, app_store_sub, receipt, store_id, found_service)

    cur_time = ctimestamp()
    btime = found_service.btime
    expires_time = app_store_sub.get_latest_purchase_receipt().expires_ts

    if (
        cur_time < btime < expires_time
        or btime < cur_time < expires_time
        or btime == cur_time < expires_time
        or btime < cur_time == expires_time
    ):
        return update_inapp_service(app_store_sub, found_service)
    elif (
        btime < expires_time < cur_time
        or expires_time < btime < cur_time
        or expires_time == btime < cur_time
        or expires_time < btime == cur_time
    ):
        return process_expired_inapp_service(client, app_store_sub, found_service)
    else:
        default_log.info("In-app service doesn't need to be changed %s" % log_format_service(found_service))
        return found_service


def process_receipt(uid, receipt, store_id, currency="RUB"):
    client = Client(uid)
    if not client.attributes.market:
        user_info = passport.userinfo(uid)
        country = user_info.get('country', USER_DEFAULT_COUNTRY)
        market = Market.from_country(country)
        client.bind_market(market)

    active_services = []
    try:
        trust_subscriptions = trust_payment_service.verify_inapp_receipt(receipt, store_id)
    except BillingInAppInvalidVerification:
        default_log.info("Invalid receipt, do nothing receipt=%s" % receipt)
        raise

    if not trust_subscriptions:
        default_log.info("Got empty receipt without subscriptions.")
        return

    for app_store_sub in trust_subscriptions:
        try:
            service = process_app_store_sub(client, app_store_sub, receipt, store_id)
            if service:
                active_services.append(service)
        except Exception as err:
            save_failed_receipt_to_db(uid, traceback.format_exc(), receipt)
            default_log.exception("MVP1 process receipt error")

    recalculate_limit_by_services(uid)

    for active_service in active_services:
        if active_service.uid != uid:
            raise BillingInAppUidMismatch('Service uid: "%s", request uid: "%s"' % (active_service.uid, uid))
