# -*- coding: utf-8 -*-
from mpfs.engine.process import get_default_log
from mpfs.common.errors.billing import BillingInAppUnsupportedStoreIdError
from mpfs.common.static.tags.billing import APPLE_APPSTORE, GOOGLE_PLAY, PID
from mpfs.core.billing import Product
from mpfs.core.billing.inapp.dao.service_inapp_dao import ServiceInAppDAO
from mpfs.core.services.uaas_service import new_uaas

default_log = get_default_log()


def get_current_product(uid, store_id):
    if store_id == APPLE_APPSTORE:
        raw_services = ServiceInAppDAO.get_apple_appstore_raw_services_for_uid(uid)
    elif store_id == GOOGLE_PLAY:
        raw_services = ServiceInAppDAO.get_google_play_raw_services_for_uid(uid)
    else:
        raise BillingInAppUnsupportedStoreIdError(store_id)

    if raw_services:
        return Product(raw_services[0][PID])
    return None


class Tariff(object):
    def __init__(self, tariff_id, month_product, year_product, is_best_offer=False):
        self.tariff_id = tariff_id
        self.is_best_offer = is_best_offer
        self.month = month_product
        self.year = year_product
        if not self.year.is_year() or not self.month.is_month():
            raise ValueError("Product period in tariff mismatch")
        if not month_product and not year_product:
            raise ValueError("month and/or year product required")
        if self.year and self.month:
            if self.year.store_id != self.month.store_id:
                raise ValueError("store_id mismatch for month and year products")

    def copy(self):
        return self.__class__(self.tariff_id, self.month, self.year, is_best_offer=self.is_best_offer)

    @property
    def tid(self):
        return self.tariff_id

    def get_store_id(self):
        if self.month:
            return self.month.store_id
        return self.year.store_id

    def has_both(self):
        return bool(self.month and self.year)

    def get_products(self):
        return [p for p in (self.month, self.year) if p]


class InAppProductCard(object):
    store_id = None
    tariffs = tuple()
    default_tariff_ids = tuple()

    _tariffs_map = {}
    uaas_experiment_flag_prefix = None

    def __init__(self, uid):
        self.uid = uid
        self._user_card = None
        self._current_product = None

    def build_card(self):
        current_product = get_current_product(self.uid, self.store_id)
        self._current_product = current_product
        tariff_ids = self.get_personal_tariff_ids()
        if current_product is None:
            self._user_card = [self._tariffs_map[i] for i in tariff_ids]
        else:
            self._user_card = []
            for tariff_id in tariff_ids:
                tariff = self._tariffs_map[tariff_id]
                tariff_product = tariff.get_products()[0]
                # логика фильтрации тарифов в карточке
                if tariff_product.get_space() < current_product.get_space():
                    # тариф с объёмом меньше текущего продукта откидываем
                    pass
                elif tariff_product.get_space() == current_product.get_space():
                    if current_product.is_year():
                        pass
                    else:
                        # на место месячного продукта с тем же объёмом подставляем текущий продукт
                        # он может совпадать по pid-у, а может быть другой, например акционный.
                        tmp_tariff = tariff.copy()
                        tmp_tariff.month = current_product
                        self._user_card.append(tmp_tariff)
                else:
                    self._user_card.append(tariff)
        return self._user_card

    def get_personal_tariff_ids(self):
        if not self.uaas_experiment_flag_prefix:
            return self.default_tariff_ids

        try:
            experiments = new_uaas.get_experiments(uid=self.uid)
            for experiment in experiments:
                if experiment.has_match_by_flag_prefix(self.uaas_experiment_flag_prefix):
                    tariff_exp = experiment
                    break
            else:
                return self.default_tariff_ids

            tariff_ids = tariff_exp.get_data()
            if not tariff_ids:
                return self.default_tariff_ids
            for tariff_id in tariff_ids:
                if tariff_id not in self._tariffs_map:
                    raise ValueError('Unexpected tariff from uaas %s' % tariff_id)
        except Exception:
            default_log.exception("Error due to tariff get")
            return self.default_tariff_ids
        else:
            return tariff_ids

    @classmethod
    def build_and_validate(cls):
        """Проверяем, что заполнили карточки правильно"""
        assert cls.store_id in (APPLE_APPSTORE, GOOGLE_PLAY)
        cls._tariffs_map = {}
        for tariff in cls.tariffs:
            assert tariff.get_store_id() == cls.store_id
            assert tariff.tid not in cls._tariffs_map
            cls._tariffs_map[tariff.tid] = tariff
        for tariff_id in cls.default_tariff_ids:
            assert tariff_id in cls._tariffs_map

    def format(self):
        if self._user_card is None:
            self.build_card()

        result = {"payment_method": self.store_id, "items": []}
        if self._current_product:
            result["current_product"] = self._format_product(self._current_product)
        for tariff in self._user_card:
            f_tariff = {"id": tariff.tid, "is_best_offer": tariff.is_best_offer, "periods": {}}
            for period_name, product in (("month", tariff.month), ("year", tariff.year)):
                if product:
                    f_tariff["periods"][period_name] = self._format_product(product)
            result["items"].append(f_tariff)
        return result

    @staticmethod
    def _format_product(product):
        return {"product_id": product.pid}


class AppStoreProductCard(InAppProductCard):
    store_id = APPLE_APPSTORE
    tariffs = (
        Tariff(
            "100gb_apple_appstore_2019",
            Product("100gb_1m_apple_appstore_2019"),
            Product("100gb_1y_apple_appstore_2019"),
        ),
        Tariff(
            "1tb_apple_appstore_2019",
            Product("1tb_1m_apple_appstore_2019"),
            Product("1tb_1y_apple_appstore_2019"),
            is_best_offer=True,
        ),
        Tariff(
            "3tb_apple_appstore_2019",
            Product("3tb_1m_apple_appstore_2019"),
            Product("3tb_1y_apple_appstore_2019")
        ),
    )
    default_tariff_ids = ("100gb_apple_appstore_2019", "1tb_apple_appstore_2019", "3tb_apple_appstore_2019")
    uaas_experiment_flag_prefix = 'disk_tariff_ios_'


class GooglePlayProductCard(InAppProductCard):
    store_id = GOOGLE_PLAY
    tariffs = (
        Tariff(
            "100gb_google_play_2019",
            Product("100gb_1m_google_play_2019"),
            Product("100gb_1y_google_play_2019")),
        Tariff(
            "1tb_google_play_2019",
            Product("1tb_1m_google_play_2019"),
            Product("1tb_1y_google_play_2019"),
            is_best_offer=True,
        ),
        Tariff(
            "3tb_google_play_2019",
            Product("3tb_1m_google_play_2019"),
            Product("3tb_1y_google_play_2019")
        ),
    )
    default_tariff_ids = ("100gb_google_play_2019", "1tb_google_play_2019", "3tb_google_play_2019")
    uaas_experiment_flag_prefix = None


AppStoreProductCard.build_and_validate()
GooglePlayProductCard.build_and_validate()


def get_card_class(store_id):
    if store_id == APPLE_APPSTORE:
        return AppStoreProductCard
    elif store_id == GOOGLE_PLAY:
        return GooglePlayProductCard
    else:
        raise BillingInAppUnsupportedStoreIdError(store_id)
