# -*- coding: utf-8 -*-
import json
import time
from copy import copy
from random import randint

import dateutil
import traceback

import mpfs.engine.process
from mpfs.common.errors.billing import (
    BillingTrustUIDSyncError,
    BillingTrustInvalidVerification,
    BillingTrustSyncError,
    BillingInAppUidMismatch,
    TrustInAppSubscriptionStateNotSynchronized,
    TrustInAppSubscriptionStateUnexpected,
)

from mpfs.common.static.tags.billing import STATE, ServiceState, SID, BILLING_INAPP_RESYNC_SUBS, \
    ORIGINAL_TRANSACTION_ID, BUY_NEW, PROLONG_AUTO
from mpfs.common.util import with_retries
from mpfs.core.billing import Service, BTIME, Client, Product
from mpfs.core.billing.inapp.dao.service_inapp_dao import ServiceInAppDAO
from mpfs.core.billing.inapp.unprocessed_receipt import save_failed_receipt_to_db
from mpfs.core.billing.processing.repair import recalculate_limit_by_services
from mpfs.core.billing.service import ServiceList, ServiceCreate
from mpfs.core.services.trust_payments import trust_payment_service
from mpfs.core.queue import mpfs_queue

default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()
stat_billing_payments_log = mpfs.engine.process.get_stat_log("billing-payments")


class TrustInAppSubscription(object):
    def __init__(self, subscription_dict):
        self.subscription = copy(subscription_dict)
        if self.subscription["state"] not in ServiceState.valid_states:
            if self.subscription["state"] == ServiceState.not_synchronized:
                raise TrustInAppSubscriptionStateNotSynchronized(
                    "Trust in app subscription not synchronized: %s" % self.subscription)
            raise TrustInAppSubscriptionStateUnexpected(
                'Unexpected trust in app subscription state: "%s". Expected: %s'
                % (self.subscription["state"], ServiceState.valid_states)
            )
        subs_until_dt = dateutil.parser.parse(self.subscription["subs_until_dt"])
        self.subscription["subs_until_ts"] = int(time.mktime(subs_until_dt.timetuple()))

    def get_subs_until_ts(self):
        return self.subscription["subs_until_ts"]

    def is_finished(self):
        return self.subscription["state"] == ServiceState.finished

    def get_subscription_id(self):
        return self.subscription["id"]

    def get_state(self):
        return self.subscription["state"]

    def get_product_id(self):
        return self.subscription["product_id"]

    def get_store_id(self):
        return self.subscription["store_id"]

    def get_uid(self):
        return str(self.subscription["uid"])

    def get_is_production(self):
        return self.subscription.get("is_production", False)

    def __repr__(self):
        return "TrustInAppSubscription(%s)" % json.dumps(self.subscription)


def get_service_by_inapp_subs_id(otid):
    if not otid:
        return
    services = ServiceList(original_transaction_id=[otid])
    if not services:
        return
    return Service(services[0][SID])


def format_service(service):
    if not service:
        return "(no service)"
    return "InAppService(uid=%s, sid=%s, pid=%s, btime=%s, trust_subs_id=%s, state=%s)" % (
        service.uid,
        service.sid,
        service.pid,
        service.btime,
        service.original_transaction_id,
        service.state,
    )


def provide_inapp_service(client, trust_subscription):
    product = Product(pid=trust_subscription.get_product_id())
    service = ServiceCreate(client, product, auto=False)
    service.set(BTIME, int(trust_subscription.get_subs_until_ts()), save_to_db=False)
    # на самом деле это id in-app подписки в трасте. Поле так названо при разработке mvp1 in-app-ов.
    service.set(ORIGINAL_TRANSACTION_ID, trust_subscription.get_subscription_id(), save_to_db=False)
    service.set(STATE, trust_subscription.get_state(), save_to_db=False)
    service.enable(save_to_db=False)
    service.save()
    return service


def process_receipt(uid, receipt, store_id, currency="RUB"):
    active_services = []
    try:
        sync_items = trust_payment_service.handle_inapp_receipt(uid, receipt, store_id, currency=currency)
    except BillingTrustInvalidVerification as err:
        default_log.info("Invalid receipt, do nothing receipt=%s" % receipt)
        raise

    catch_exception = False
    for sync_item in sync_items:
        try:
            if sync_item["sync_status"] != "success":
                raise BillingTrustSyncError(sync_item)
            trust_sub = TrustInAppSubscription(sync_item["subscription"])
            service = process_trust_sub(trust_sub)
            if service:
                active_services.append(service)
        except Exception as err:
            save_failed_receipt_to_db(uid, traceback.format_exc(), receipt)
            error_log.exception("Got exception during inapp subscription sync. Trust item: %s", sync_item)
            catch_exception = True

    if catch_exception:
        raise

    for active_service in active_services:
        if active_service.uid != uid:
            raise BillingInAppUidMismatch('Service uid: "%s", request uid: "%s"' % (active_service.uid, uid))


def log_to_billing_log(service, is_production, is_prolong):
    if not is_production:
        return
    try:
        product = Product(service.pid)
        log_data = {
            'uid': service.uid,
            'action': PROLONG_AUTO if is_prolong else BUY_NEW,
            'order': service.original_transaction_id,
            'product_id': service.pid,
            'auto': True,
            'status': 'success',
            'price': product.price['RUB'],
            'currency': 'RUB',
            'used': None,
            'limit': None
        }
        stat_billing_payments_log.info(log_data)
    except Exception as e:
        error_log.error('Failed to write `stat_log`: %s:%s', type(e), e)


def process_trust_sub(trust_subscription):
    """Main sync logic with trust in-app subscription"""
    if not isinstance(trust_subscription, TrustInAppSubscription):
        raise TypeError("Expected TrustInAppSubscription obj, got %s" % trust_subscription)

    client = Client(trust_subscription.get_uid())
    service = get_service_by_inapp_subs_id(trust_subscription.get_subscription_id())
    service_before_sync_text = format_service(service)
    default_log.info("Before sync. MPFS: %s, TRUST: %s", service_before_sync_text, trust_subscription)

    if trust_subscription.get_subs_until_ts() < time.time() and trust_subscription.get_state() in (
        ServiceState.active,
        ServiceState.in_grace,
    ):
        default_log.warn('Trust "subs_until_ts" in the past for active state. %s' % trust_subscription)

    do_recalculate_limit = True
    is_production = trust_subscription.get_is_production()
    if not service:
        # no mpfs service
        if trust_subscription.is_finished():
            msg = "Ignore already finished subscription."
            service = None
            do_recalculate_limit = False
        else:
            msg = "Create new service."
            service = provide_inapp_service(client, trust_subscription)
            log_to_billing_log(service, is_production, False)
    else:
        # related service exists in mpfs
        if service.uid != trust_subscription.get_uid():
            raise BillingTrustUIDSyncError(subscription=trust_subscription)

        if trust_subscription.is_finished():
            msg = "Subscription finished. Service removed."
            service.set(STATE, trust_subscription.get_state(), save_to_db=False)
            service.delete(disable=True, send_email=False)
            service = None
        elif service.pid != trust_subscription.get_product_id():
            msg = "Change product."
            old_service = service
            service = provide_inapp_service(client, trust_subscription)
            old_service.delete(disable=True, send_email=False)
            log_to_billing_log(service, is_production, True)
        elif (
            abs(service.btime - trust_subscription.get_subs_until_ts()) > 10
            or service.state != trust_subscription.get_state()  # чтобы исключить различия в несколько секунд
        ):
            msg = "Subscription changed."
            service.set(BTIME, trust_subscription.get_subs_until_ts(), save_to_db=False)
            service.set(STATE, trust_subscription.get_state(), save_to_db=False)
            service.save()
            if trust_subscription.get_state() == ServiceState.active:
                log_to_billing_log(service, is_production, True)
        else:
            msg = "No changes."
            do_recalculate_limit = False

    service_after_sync_text = format_service(service)
    default_log.info(
        "Sync result: %s BEFORE: %s. AFTER: %s. TRUST_SUBS: %s",
        msg,
        service_before_sync_text,
        service_after_sync_text,
        trust_subscription,
    )
    if do_recalculate_limit:
        try:
            recalculate_limit_by_services(client.uid)
        except Exception:
            default_log.exception("Got exception due to limit recalculation")
    return service


class InAppResyncManager(object):
    @staticmethod
    def async_resync_by_trust_subs_id(trust_subs_id, delay=None):
        """Create async task"""
        if not trust_subs_id:
            default_log.warn("Attemp put inapp resync tasks for empty trust_subs_id")
            return
        mpfs_queue.put(
            {"trust_subs_id": trust_subs_id}, BILLING_INAPP_RESYNC_SUBS, deduplication_id="%s" % trust_subs_id,
            delay=delay
        )

    @staticmethod
    def get_trust_inapp_subs(trust_subs_id):
        raw_inapp_subs = trust_payment_service.get_inapp_subscription(trust_subs_id)
        return TrustInAppSubscription(raw_inapp_subs)

    @staticmethod
    def resync_by_trust_subs_id(trust_subs_id):
        from mpfs.config import settings
        default_log.info("Start resync in-app subs %s" % trust_subs_id)

        trust_subscription = with_retries(
            lambda: InAppResyncManager.get_trust_inapp_subs(trust_subs_id), settings.billing_cron_tasks['in_app_resync_timeouts'],
            TrustInAppSubscriptionStateNotSynchronized, lambda: trust_payment_service.resync_inapp_subscription(trust_subs_id))
        process_trust_sub(trust_subscription)

    @classmethod
    def regular_resync_jobs(cls):
        """This method calls in regular bazinga (cron) task BillingProcess"""
        cls.resync_all_on_hold()
        cls.resync_all_in_grace()
        cls.resync_all_expired()

    @classmethod
    def resync_all_on_hold(cls):
        raw_services = ServiceInAppDAO.get_raw_services(states=[ServiceState.on_hold])
        default_log.info("In app resync. Find %i services %s state", len(raw_services), ServiceState.on_hold)
        cls._put_resync_tasks(raw_services)

    @classmethod
    def resync_all_in_grace(cls):
        raw_services = ServiceInAppDAO.get_raw_services(
            btime_lower_bound=time.time(), states=[ServiceState.in_grace]
        )
        default_log.info("In app resync. Find %i services %s state", len(raw_services), ServiceState.in_grace)
        cls._put_resync_tasks(raw_services)

    @classmethod
    def resync_all_expired(cls):
        raw_services = ServiceInAppDAO.get_raw_services(
            btime_upper_bound=time.time(), states=[ServiceState.in_grace, ServiceState.active]
        )
        default_log.info("In app resync. Find %i expired services", len(raw_services))
        cls._put_resync_tasks(raw_services)

    @classmethod
    def _put_resync_tasks(cls, raw_services):
        from mpfs.config import settings
        for raw_service in raw_services:
            trust_subs_id = raw_service.get("original_transaction_id")
            if trust_subs_id:
                cls.async_resync_by_trust_subs_id(
                    trust_subs_id,
                    delay=randint(0, settings.queue2['regular_resync_jobs']['async_resync_by_trust_subs_id']['delay']))
            else:
                default_log.warn("No 'original_transaction_id' for in-app subscription. %s" % raw_services)
