# -*- coding: utf-8 -*-
from mpfs.common.errors import BadRequestError
from mpfs.common.static.tags.billing import CTIME
from mpfs.common.util import from_json, to_json
from mpfs.common.util.crypt import AesCbcCryptAgent, DEFAULT_SECRET


class IterationKey(object):
    DEFAULT_PAGE_SIZE = 100
    _crypt_agent = AesCbcCryptAgent(DEFAULT_SECRET)

    def __init__(self, uid, first_ctime, last_ctime, filters, is_archive):
        self.uid = uid
        self.first_ctime = first_ctime
        self.last_ctime = last_ctime
        self.filters = filters
        self.is_archive = is_archive

    @classmethod
    def parse(cls, raw_iteration_key):
        if not raw_iteration_key:
            return None
        try:
            raw_json = cls._crypt_agent.decrypt(raw_iteration_key)
            data = from_json(raw_json)
            iter_key = IterationKey(
                uid=data['uid'],
                first_ctime=data['first_ctime'],
                last_ctime=data['last_ctime'],
                filters=data['filters'],
                is_archive=data['is_archive'],
            )
        except Exception:
            raise BadRequestError(extra_msg='Unable to parse iteration key, probably malformed')
        return iter_key

    def serialize(self):
        data = {
            'uid': self.uid,
            'first_ctime': self.first_ctime,
            'last_ctime': self.last_ctime,
            'filters': self.filters,
            'is_archive': self.is_archive,
        }
        raw_json = to_json(data)
        raw_iteration_key = self._crypt_agent.encrypt(raw_json)
        return raw_iteration_key

    @classmethod
    def get_limit(cls):
        return cls.DEFAULT_PAGE_SIZE + 1

    @classmethod
    def check_if_more_results_exists_and_truncate(cls, orders, used_limit):
        # так как мы в результате отбрасываем один элемент, то проверяем на строгое неравенство
        if len(orders) < used_limit:
            return False, orders
        if orders[-1][CTIME] != orders[-2][CTIME]:
            return True, orders[:-1]
        i = len(orders) - 2
        while i >= 0 and orders[i][CTIME] == orders[-1][CTIME]:
            i -= 1
        if i < 0:
            # TODO: если элементов с одинаковым ctime больше, чем размер страницы, то не вернем те, что не поместились
            # TODO: в окно
            return True, orders
        return True, orders[:i + 1]
