# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Интерфейсы работы с заказами и подписками

"""
import mpfs.engine.process
from mpfs.common.errors import billing as errors, BadRequestError
from mpfs.common.static.tags.billing import *
from mpfs.common.util import ctimestamp
from mpfs.config import settings
from mpfs.core.billing.client import Client
from mpfs.core.billing.interface.iteration_keys.order_list_filtered import IterationKey
from mpfs.core.billing.market import Market
from mpfs.core.billing.order import Order, OrderList, ArchiveOrder, look_for_order
from mpfs.core.billing.payment import Payment
from mpfs.core.billing.processing import billing, common
from mpfs.core.billing.product import Product, get_default_products_line
from mpfs.core.billing.product.catalog import Catalog
from mpfs.core.queue import mpfs_queue as queue
from mpfs.engine.queue2.utils import get_callback_host
from mpfs.metastorage import DESCENDING

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

catalog = Catalog()

BILLING_GROUP_UIDS_LIMIT = settings.billing['group_uids_limit']
BILLING_AVAILABLE_TEMPLATE_TAGS = settings.billing['available_template_tags']
VALID_BB_DOMAINS = settings.system['valid_billing_top_level_domains']
THIRD_TARIFF_PLUS_PIDS = settings.billing['plus_pids']
NO_DISCOUNT_EMAIL_DELAY = settings.email_sender_campaigns['no_discount_follow_up_email']['email_delay']
WITH_DISCOUNT_EMAIL_DELAY = settings.email_sender_campaigns['with_discount_follow_up_email']['email_delay']


def get_bb_callbacks():
    callback_conf = settings.billing['bb_callback']
    callback_host = callback_conf['host']
    if not callback_host:
        callback_host = 'http://' + get_callback_host()

    order_callback_url = "%s%s" % (callback_host, callback_conf['order_path'])
    subscription_callback_url = "%s%s" % (callback_host, callback_conf['subscription_path'])
    return (order_callback_url, subscription_callback_url)


def order_place(req):
    '''
    Создать заказ
    '''
    if req.tld in VALID_BB_DOMAINS:
        bb_tld = req.tld
    else:
        bb_tld = None
    order, data = _order_place(Client(req.uid, req.ip),
                         req.pid,
                         req.payment_method,
                         req.auto, req.locale,
                         bb_tld,
                         req.group_uids,
                         req.group_name)
    return order.number


def order_make_payment(req):
    '''
    Оплата заказа

    Дергаем Биллинг за старт оплаты
    Пользователю отдаем ссылку на нужную страницу оплаты
    '''
    if req.tld in VALID_BB_DOMAINS:
        bb_tld = req.tld
    else:
        bb_tld = None

    return _order_make_payment(Client(req.uid, req.ip),
                               Order(req.number),
                               req.locale,
                               bb_tld,
                               req.return_path,
                               req.template_tag,
                               req.pid)


def buy(req):
    client = Client(req.uid, req.ip)

    # разрешаем покупать продукты только из самой дешевой доступной линейки
    from mpfs.core.promo_codes.logic.discount_manager import DiscountManager
    discount = DiscountManager.get_cheapest_available_discount(req.uid)
    line = get_default_products_line()
    discount_template_id = None
    if discount is not None:
        line = discount.provided_line
        discount_template_id = discount.discount_template_id

    THIRD_PIDS = ['3tb_1m_2019', '3tb_1y_2019',
                  '3tb_1m_2019_discount_10', '3tb_1m_2019_discount_20', '3tb_1m_2019_discount_30',
                  '3tb_1y_2019_discount_10', '3tb_1y_2019_discount_20', '3tb_1y_2019_discount_30'] + THIRD_TARIFF_PLUS_PIDS

    # TODO: CHEMODAN-68575 Переделать
    if req.pid not in THIRD_PIDS and req.pid not in catalog.get_pids(Market(client.attributes.market), line):
        raise errors.BillingCannotBuyProductFromNotCheapestLineAvailable()

    order, data = _order_place(client,
                               req.pid,
                               req.payment_method,
                               req.auto,
                               req.locale,
                               req.bb_tld,
                               req.group_uids,
                               req.group_name,
                               discount_template_id,
                               req.return_path,
                               req.template_tag,
                               from_=req.from_)
    result = _order_make_payment(client,
                                 order,
                                 req.locale,
                                 req.bb_tld,
                                 req.return_path,
                                 req.template_tag,
                                 req.pid)
    result.update(**data)
    result.pop('order_id', None)
    result['order_number'] = order.number

    # https://st.yandex-team.ru/CHE-434
    data = {'uid': req.uid}
    for campaign_name in ('no_discount_follow_up_email', 'with_discount_follow_up_email'):
        task_name = campaign_name
        delay = settings.email_sender_campaigns[campaign_name]['email_delay']
        queue.put(
            data,
            task_name,
            delay=delay,
            deduplication_id='%s__%s' % (task_name, req.uid),
            robust=False  # https://st.yandex-team.ru/CHEMODAN-69191: c True не поддерживается delay
        )
    return result


def _order_place(client, pid, payment_method, auto, locale, bb_tld, group_uids, group_name,
                 discount_template_id=None, return_path=None, template_tag=None, from_=None):
    '''
    Создать заказ
    '''
    if not client.attributes.market:
        raise errors.BillingClientNotBindedToMarket(client.uid)

    product = Product(pid)
    market = Market(client.attributes.market)

    if payment_method not in market.payment_methods:
        raise errors.BillingPaymentMethodNotAllowed(payment_method)

    if auto and not product.auto:
        raise errors.BillingProductAutoProlongateNotAllowed(product.pid)

    if group_uids:
        group_uids = [u.strip() for u in group_uids.split(',')]
        if len(set(group_uids)) < len(group_uids):
            raise errors.BillingGroupOrderDuplicateUids(product.pid)
        if len(group_uids) > BILLING_GROUP_UIDS_LIMIT:
            raise errors.BillingGroupOrderLimitFailed(product.pid)

    order, data = Order.Create(
        client,
        product,
        payment_method,
        auto=auto,
        locale=locale,
        group_uids=group_uids,
        group_name=group_name,
        bb_tld=bb_tld,
        discount_template_id=discount_template_id,
        return_path=return_path,
        template_tag=template_tag
    )
    common.stat(action=ORDER_NEW, client=client, product=product, order=order, from_=from_)

    return order, data


def _order_make_payment(client, order, locale, bb_tld, return_path, template_tag, pid):
    if template_tag not in BILLING_AVAILABLE_TEMPLATE_TAGS:
        raise errors.BillingUnexpectedTemplateTag()

    order_callback_url, subscription_callback_url = get_bb_callbacks()
    callback = subscription_callback_url if order.auto else order_callback_url

    payment = Payment(
        client,
        order.payment_method,
        payment_basis=order,
        callback=callback % (order.number, client.uid),
        return_path=return_path,
    )
    return payment.make(bb_tld=bb_tld, template_tag=template_tag, locale=locale, pid=pid)


def order_process_callback(req):
    """
    Обработка коллбека от ББ (заказ обычный).
    """
    queue_order_process_callback(req.uid, req.number, req.status, req.status_code, req.mode, req.trust_refund_id)


def queue_order_process_callback(uid, number, status, status_code, mode, trust_refund_id, pid=None):
    """
    Кладет в очередь задачу обработки обычного заказа.
    """
    try:
        order = Order(number)
        order.touch()
    except errors.BillingOrderNotFound, e:
        pass

    data = {
        UID: uid,
        NUMBER: number,
        STATUS: status,
        STATUS_CODE: status_code,
        MODE: mode,
        TRUST_REFUND_ID: trust_refund_id,
    }

    queue.put(data, BILLING_ORDER_CALLBACK)
    log.info(
        'callback: order %s:%s, status "%s (%s)" (mode "%s") trust_refund_id: %s' %
        (uid, number, status, status_code, mode, trust_refund_id)
    )


def subscription_process_callback(req):
    """
    Обработка коллбека от ББ (заказ с подпиской)
    """
    queue_subscription_process_callback(req.uid, req.number, req.status, req.status_code, req.mode, req.trust_refund_id)


def queue_subscription_process_callback(uid, number, status, status_code, mode, trust_refund_id):
    """
    Кладет в очередь задачу обработки подписки.
    """
    data = {
        UID: uid,
        NUMBER: number,
        STATUS: status,
        STATUS_CODE: status_code,
        MODE: mode,
        TRUST_REFUND_ID: trust_refund_id,
    }

    queue.put(data, BILLING_SUBSCRIPTION_CALLBACK)
    log.info(
        'callback: subscription %s:%s, status "%s (%s)" (mode "%s") trust_refund_id: %s' %
        (uid, number, status, status_code, mode, trust_refund_id)
    )


def order_list_current(req):
    params = {CLIENT: Client(req.uid)}
    return OrderList(**params)


def order_list_history(req):
    params = {CLIENT: Client(req.uid), ARCHIVE: True}
    return OrderList(**params)


def order_list_filtered(req):
    if req.iteration_key:
        iteration_key = IterationKey.parse(req.iteration_key)
        if req.uid != iteration_key.uid:
            raise BadRequestError(extra_msg='Uid from qs and iteration key mismatched')
        first_ctime = iteration_key.first_ctime
        last_ctime = iteration_key.last_ctime
        filters = iteration_key.filters
        is_archive = iteration_key.is_archive
    else:
        first_ctime = ctimestamp()
        last_ctime = first_ctime
        filters = [x for x in req.filters.split(',') if x]
        is_archive = False
    spec = {
        CTIME_LT: last_ctime,
        UID: req.uid,
    }
    if filters:
        spec[STATES] = filters

    spec['sort'] = (CTIME, DESCENDING)

    orders = []
    if not is_archive:
        spec['limit'] = IterationKey.get_limit()
        has_more_results, orders = IterationKey.check_if_more_results_exists_and_truncate(OrderList(**spec),
                                                                                          spec['limit'])
        if has_more_results:
            new_iter_key = IterationKey(uid=req.uid, first_ctime=first_ctime, last_ctime=orders[-1][CTIME],
                                        filters=filters, is_archive=False)
            return _serialize_order_list(orders, new_iter_key)
        if len(orders) >= IterationKey.DEFAULT_PAGE_SIZE:
            new_iter_key = IterationKey(uid=req.uid, first_ctime=first_ctime, last_ctime=first_ctime,
                                        filters=filters, is_archive=True)
            return _serialize_order_list(orders, new_iter_key)
        spec[CTIME_LT] = first_ctime

    spec['limit'] = IterationKey.get_limit() - len(orders)
    spec[ARCHIVE] = True
    has_more_results, archived_orders = IterationKey.check_if_more_results_exists_and_truncate(
        OrderList(**spec), spec['limit'])

    new_iter_key = None
    if has_more_results:
        new_iter_key = IterationKey(uid=req.uid, first_ctime=first_ctime, last_ctime=archived_orders[-1][CTIME],
                                    filters=filters, is_archive=True)
    return _serialize_order_list(orders + archived_orders, new_iter_key)


def _serialize_order_list(orders, iteration_key=None):
    for i in orders:
        pid = i[PID]
        i['product'] = {
            PID: i.pop(PID, None),
            BB_PID: i.pop(BB_PID, None),
            NAMES: Product(pid).get_names(i.get(AUTO)),
        }
    result = {'items': orders}
    if iteration_key:
        result['iteration_key'] = iteration_key.serialize()
    return result


def order_recall(req):
    '''
    Ручное перепроведение зафейленного из-за наших ошибок платежа
    Сработает только на lost-заказы
    '''
    order = ArchiveOrder(req.number)
    if order.state in (LOST, FAIL):
        billing.process_first_payment(order, req.status, req.status_code)


def order_set_payment_processing_state(req):
    """
    Установить ожидающему колбека заказу статус payment_processing
    """
    order = look_for_order(req.number)
    if order is None or order.uid != req.uid:
        raise errors.BillingOrderNotFound()
    if isinstance(order, ArchiveOrder):
        raise errors.BillingUnableToChangeOrderState()
    order.set('state', PAYMENT_PROCESSING)
