# coding: utf-8
import mpfs.engine.process

from mpfs.common.errors import billing as errors
from mpfs.common.static.tags.billing import TRUST_REFUND_ID, REFUND_STATUS, SUCCESS
from mpfs.core.billing.order import ArchiveOrder
from mpfs.core.services.trust_payments import trust_payment_service


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def refund_order(req):
    order = ArchiveOrder(req.number)  # успешно оплаченного заказа не может быть в другом месте

    if order.get(REFUND_STATUS) == SUCCESS:
        raise errors.BillingRefundRetryError('Order %s:%s already refunded' % (req.uid, order.number))

    refund_id = order.get(TRUST_REFUND_ID)

    if refund_id:
        refund_status = trust_payment_service.get_refund_status(req.uid, refund_id)
        if not refund_status.can_retry:
            raise errors.BillingRefundRetryError('Cant retry refund %s for %s:%s' % (refund_id, req.uid, order.number))

    payment_id = trust_payment_service.get_last_payment_id(req.uid, order)

    if not payment_id:
        raise errors.BillingPaymentNotFoundError('Payment not found for %s:%s' % (req.uid, order.number))

    payment_info = trust_payment_service.check_payment_by_id(req.uid, payment_id)

    log.info('Found payment %s for %s:%s' % (payment_id, req.uid, order.number))

    if not payment_info.is_payment_refundable:
        raise errors.BillingPaymentNotRefundableError(
            'Payment %s for %s:%s is not refundable' % (payment_id, req.uid, order.number))

    refund_id = trust_payment_service.create_refund(req.uid, payment_id, order.number, req.reason)
    refund_status = trust_payment_service.start_refund(req.uid, refund_id)

    log.info('Refund %s created and started for payment %s for %s:%s' % (refund_id, payment_id, req.uid, order.number))

    order.set(TRUST_REFUND_ID, refund_id)
    return {
        'refund_id': refund_id,
        'status': refund_status.status,
        'status_desc': refund_status.status_description,
        'fiscal_receipt_url': refund_status.fiscal_receipt_url,
        'can_retry': refund_status.can_retry,
    }


def refund_order_status(req):
    order = ArchiveOrder(req.number)
    refund_id = order.get(TRUST_REFUND_ID)

    if refund_id:
        refund_status = trust_payment_service.get_refund_status(req.uid, refund_id)
        return {
            'refund_id': refund_id,
            'status': refund_status.status,
            'status_desc': refund_status.status_description,
            'fiscal_receipt_url': refund_status.fiscal_receipt_url,
            'can_retry': refund_status.can_retry,
        }

    raise errors.BillingRefundNotFoundError('Refund not found for order %s:%s' % (req.uid, order.number))
