# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Интерфейсы по работе с услугами

"""
import mpfs.engine.process

from mpfs.common.errors import billing as errors
from mpfs.common.static.tags.billing import *
from mpfs.common.util import ctimestamp
from mpfs.core.billing import Product, OrderList
from mpfs.core.billing.client import Client
from mpfs.core.billing.constants import PAID_ORGANIZATION_SERVICE_ID
from mpfs.core.billing.order import ProlongateOrder
from mpfs.core.billing.order.subscription import Subscription
from mpfs.core.billing.product.catalog import PRIMARY_PRODUCTS_IDS
from mpfs.core.billing.product.catalog.inapp import InAppProductsHolder
from mpfs.core.billing.service import Service, ServiceList, ServiceListHistory
from mpfs.core.billing.market import Market
from mpfs.core.billing.processing import billing
from mpfs.core.organizations.dao.organizations import OrganizationDAO
from mpfs.core.queue import mpfs_queue as queue

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def service_list(req):
    '''
    Листинг услуг
    '''
    params = {CLIENT: Client(req.uid, req.ip)}
    services = ServiceList(**params)
    primary_services = [x for x in services if x[PID] in PRIMARY_PRODUCTS_IDS]
    primary_service_sid_to_order = {
        x[SID]: x
        for x in OrderList(uid=req.uid, sid=[ps[SID] for ps in primary_services], archive=True)
    }
    for service in primary_services:
        if service[PID] in InAppProductsHolder.all_pids:
            service[PAYMETHOD] = Product(service[PID]).store_id
        elif service[SID] in primary_service_sid_to_order:
            order = primary_service_sid_to_order[service[SID]]
            service[PAYMETHOD] = order[PAYMETHOD]

    if req.from_db:
        return services

    from mpfs.core.organizations.logic import B2B_IGNORED_SERVICES, B2B_FAKE_ORGANIZATION_SPACE_SERVICE_NAME, B2B_SHARED_ORGANIZATION_SPACE_ENABLED
    if not B2B_SHARED_ORGANIZATION_SPACE_ENABLED:
        return services

    from mpfs.core.user.base import User
    user = User(req.uid)
    if not user.b2b_key:
        return services

    organization = OrganizationDAO().find_by_id(user.b2b_key)
    if not organization:
        return services
    if not organization.is_paid:
        return services

    services = [x for x in services if x[PRODUCT].id not in B2B_IGNORED_SERVICES]
    current_time = ctimestamp()
    fake_product = Product(B2B_FAKE_ORGANIZATION_SPACE_SERVICE_NAME)
    fake_product.attributes.amount = organization.quota_free
    services.append({
        AUTO: fake_product.auto,
        BTIME: None,
        CHILD_SIDS: None,
        CTIME: current_time,
        DELETETIME: None,
        ENABLED: True,
        GROUP: False,
        GROUP_NAME: None,
        LASTBTIME: current_time,
        MTIME: current_time,
        PARENT_SID: None,
        PID: B2B_FAKE_ORGANIZATION_SPACE_SERVICE_NAME,
        PRODUCT: fake_product,
        SID: PAID_ORGANIZATION_SERVICE_ID,
        STATE: None,
        UID: req.uid,
        V: 0,
    })
    return services


def service_list_history(req):
    '''
    Листинг завершенных услуг
    '''
    params = {CLIENT: Client(req.uid, req.ip)}
    return ServiceListHistory(**params)


def service_prolongate(req):
    '''
    Ручное продление услуги

    Срабатывает только
    Отдает заказ на оплату
    '''
    client = Client(req.uid, req.ip)

    if not client.attributes.market:
        raise errors.BillingClientNotBindedToMarket(req.uid)

    market = Market(client.attributes.market)

    if req.payment_method not in market.payment_methods:
        raise errors.BillingPaymentMethodNotAllowed(req.payment_method)

    service = Service(req.sid)
    product = service.product

    if not product.prolongable:
        raise errors.BillingProductNotProlongable(product.pid)

    if service.auto:
        # защита от случая когда пытаемся вручную продлить автопродлеваемую услугу
        # а то чревато списанием денег несколько раз
        raise errors.BillingServiceHasAutoProlongate(req.sid)

    subscription = None
    try:
        subscription = Subscription(service=req.sid)
    except errors.BillingSubscriptionNotFound:
        pass

    if subscription is not None:
        error_log.info(
            'Service %s has subscription %s, something strange is happening...' % (req.sid, subscription.number)
        )
        raise errors.BillingError('Unexpected subscription for service %s.' % req.sid)

    order, data = ProlongateOrder.Create(
        client,
        service.product,
        req.payment_method,
        service=service,
        locale=req.locale
    )

    return order.number


def service_unsubscribe(req):
    '''
    Отказ от подписки
    '''
    client = Client(req.uid, req.ip)
    service = Service(req.sid)
    subscription = Subscription(service=service.sid)

    if service.parent_sid:  # если это услуга не покупателя, то не должно быть возможности отменить подписку
        return

    billing.subscription_unsubscribe(client, service, subscription)


def service_process_billing(req):
    '''
    Обработка постановки услуги на биллинг (интерфейс)
    '''
    data  = {SID: req.sid, UID: req.uid}
    queue.put(data, BILLING_SERVICE_STANDART_BILLING)


def service_set_attribute(req):
    '''
    Установка атрибутов
    '''
    client = Client(req.uid, req.ip)
    service = Service(req.sid)
    service.set_attribute(req.key, req.value)


def service_remove_attribute(req):
    '''
    Удаление атрибутов
    '''
    client = Client(req.uid, req.ip)
    service = Service(req.sid)
    service.remove_attribute(req.key)
