# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Заказ

"""
import mpfs.engine.process
from mpfs.common.errors import billing as errors
from mpfs.common.static.tags.billing import *
from mpfs.common.util import ctimestamp
from mpfs.config import settings
from mpfs.core.billing.client import Client
from mpfs.core.billing.entity import UniqEntity
from mpfs.core.billing.market import Market
from mpfs.core.billing.product import Product
from mpfs.core.billing.product.catalog import get_group_product_id_by_qty
from mpfs.core.metastorage.control import billing_orders, billing_orders_history
from mpfs.core.services.billing_service import BB

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

THIRD_TARIFF_PLUS_PIDS = settings.billing['plus_pids']

class Order(UniqEntity):
    '''
    Класс
    конкретного заказа на
    конкретный продукт
    конкретного рынка с
    конкретным видом оплаты

    Заказ может быть заархивирован
    '''

    order_type = BUY_NEW
    table = billing_orders
    nf_error = errors.BillingOrderNotFound
    uniq_key_name = NUMBER

    @classmethod
    def Create(classname, *args, **kwargs):
        '''
        При успешном обращении в ББ создаст запись в базе
        '''
        client, product, payment_method = args
        auto   = kwargs.get('auto', False)
        locale = kwargs.get('locale', None)
        clid   = kwargs.get('clid', None)
        group_uids = kwargs.get('group_uids', None)
        bb_tld = kwargs.get('bb_tld', None)
        return_path = kwargs.get('return_path', '')
        template_tag = kwargs.get('template_tag', 'desktop/form')

        if group_uids:
            group_pid = get_group_product_id_by_qty(product.pid, len(group_uids))
            product = Product(group_pid)
            args = client, product, payment_method  # отличный способ передавать параметры в функцию :)

        market = Market(client.attributes.market)
        price  = product.get_price(market)
        bb_pid = product.bb_pid(payment_method=payment_method, auto=auto)
        tariff_name = None
        if locale is not None:
            tariff_name = product.get_names(auto=auto).get(locale)

        number = classname.GenerateUuidID()
        data = classname._construct_db_record(number, *args, **kwargs)

        billing_orders.insert(data)
        order_place_result = BB.order_place(client.uid, number, bb_pid, market.region,
                                            locale=locale, clid=clid, bb_tld=bb_tld,
                                            tariff_name=tariff_name, ip_address=client.ip, pid=product.pid,
                                            return_path=return_path, template_tag=template_tag)
        if product.pid in THIRD_TARIFF_PLUS_PIDS:
            number = str(order_place_result['order_id'])
            billing_orders.remove(_id=data['_id'])
            data = classname._construct_db_record(number, *args, **kwargs)
            try:
                billing_orders.insert(data)
            except Exception as e:
                billing_orders.update({'_id': data['_id']}, data)
            return classname(number, data=data), order_place_result
        return classname(number, data=data), {}

    @classmethod
    def create_and_save(cls, client, product, payment_method, number=None, **kwargs):
        """
        создать заказ и сохранить в базу

        Этот метод создан отдельно от Create, поскольку тот делаем поход в биллинг. Этот же метод попросту создает заказ
        в базе
        """
        group_uids = kwargs.get('group_uids', None)

        if group_uids:
            group_pid = get_group_product_id_by_qty(product.pid, len(group_uids))
            product = Product(group_pid)

        if number is None:
            number = cls.GenerateUuidID()
        if not isinstance(number, str):
            number = str(number)
        data = cls._construct_db_record(number, client, product, payment_method, **kwargs)

        billing_orders.insert(data)
        return cls(number, data=data)

    @classmethod
    def _construct_db_record(classname, number, *args, **kwargs):
        client, product, payment_method = args
        auto   = kwargs.get('auto', False)
        locale = kwargs.get('locale', None)
        clid   = kwargs.get('clid', None)
        group_uids = kwargs.get('group_uids', None)
        group_name = kwargs.get('group_name', None)
        discount_template_id = kwargs.get('discount_template_id', None)

        current_time = ctimestamp()

        market = Market(client.attributes.market)
        price  = product.get_price(market)
        bb_pid = product.bb_pid(payment_method=payment_method, auto=auto)

        data = {
            DB_ID: number,
            UID: client.uid,
            PID: product.pid,
            BB_PID: bb_pid,
            PRICE: price,
            MARKET: market.code,
            CURRENCY: market.currency,
            CTIME: current_time,
            MTIME: current_time,
            STATE: NEW,
            AUTO: auto,
            PAYMETHOD: payment_method,
            ORDER_TYPE: classname.order_type
        }

        if locale:
            data[LOCALE] = locale

        if group_uids:
            data[GROUP_UIDS] = group_uids
            data[GROUP_NAME] = group_name

        if discount_template_id:
            data[DISCOUNT_TEMPLATE_ID] = discount_template_id

        return data

    def close(self, status, status_code=None):
        '''
        Установить состояние
        '''
        self.set(STATE, status)
        self.set(STATUS_CODE, status_code)

    def archive(self):
        '''
        Передвинет заказ в историю
        '''
        self.touch()

        history_record = self.dict()
        history_record[Order.uniq_key_name] = self.number
        del history_record[Order.pkey]

        billing_orders_history.insert(history_record)
        self.delete()

    def get_state(self):
        '''
        Возвращает статус заказа
        '''
        return self._data.get(STATE)

    def get_discount_template_id(self):
        return self._data.get(DISCOUNT_TEMPLATE_ID)


class ProlongateOrder(Order):

    order_type = PROLONG_MANUAL

    @classmethod
    def _construct_db_record(classname, number, *args, **kwargs):
        data = super(ProlongateOrder, classname)._construct_db_record(number, *args, **kwargs)
        service = kwargs['service']
        data[SID] = service.sid
        return data


class ArchiveOrder(Order):

    table = billing_orders_history
    pkey = NUMBER

    def archive(self):
        pass

    def restore_from_archive(self):
        self.touch()
        history_record = self.dict()
        history_record[Order.pkey] = history_record.pop(Order.uniq_key_name)
        billing_orders.insert(history_record)
        self.delete()

    def set_refund_status(self, status, trust_refund_id):
        self.set(TRUST_REFUND_ID, trust_refund_id, save_to_db=False)
        self.set(REFUND_STATUS, status, save_to_db=False)
        self.save()


def OrderList(**kwargs):
    '''
    Листинги заказов по конкретным запросам
    '''
    result = []
    params = {}
    table = billing_orders

    if CLIENT in kwargs:
        params[UID] = kwargs[CLIENT].uid
    elif UID in kwargs:
        params[UID] = kwargs[UID]

    if ARCHIVE in kwargs:
        table = billing_orders_history

    ctime = {}
    if CTIME_LT in kwargs:
        ctime['$lt'] = kwargs[CTIME_LT]
    if CTIME_GT in kwargs:
        ctime['$gt'] = kwargs[CTIME_GT]
    if ctime:
        params[CTIME] = ctime
    if SID in kwargs:
        sid_kw = kwargs[SID]
        if isinstance(sid_kw, list):
            if sid_kw:
                params[SID] = {'$in': sid_kw}
            else:
                # $in: [] в Mongo всегда возвращает пустой результат.
                # Маленькая оптимизация тут, чтобы меньше ходить в базу,
                # а так же, чтобы не чинить query_converter, который делает тут SQL: `sid IN ()`, что невалидно
                return result
        else:
            params[SID] = sid_kw

    if STATES in kwargs:
        params[STATE] = {'$in': kwargs[STATES]}

    find_kwargs = {}
    if 'limit' in kwargs:
        find_kwargs['limit'] = kwargs['limit']
    if 'skip' in kwargs:
        find_kwargs['skip'] = kwargs['skip']
    if 'sort' in kwargs:
        find_kwargs['sort'] = [kwargs['sort']]

    for item in list(table.find(params, **find_kwargs)):
        pkey   = str(item.pop(Order.pkey))
        number = item.get(Order.uniq_key_name)

        item[Order.uniq_key_name] = number or pkey
        result.append(item)

    return result


def FindOrder(uid, number):
    '''
    Поиск заказа через его статус
    Сначала идем в Биллинг, по статусу заказа поднимаем из нужной таблицы
    Первые две строки для обратной совместимости на время перехода
    '''
    if not uid:
        return Order(number)

    client = Client(uid)
    fuckup = False
    try:
        try:
            order = Order(number)
        except errors.BillingOrderNotFound, e:
            order = ArchiveOrder(number)
        bb_status = BB.check_order(client.uid, client.ip, number, pid=order.pid)
        fuckup = bb_status.get(FUCKUP_PERIOD)
    except Exception, e:
        error_log.info('Error occurred: %s', e, exc_info=True)

    if fuckup:
        log.info('processing fuckup order %s:%s' % (uid, number))

        try:
            order = ArchiveOrder(number)
        except errors.BillingOrderNotFound, e:
            order = Order(number)

        order.set(CTIME, ctimestamp())
        order.set(FUCKUP_PERIOD, fuckup)
    else:
        order = Order(number)

    return order


def look_for_order(number):
    """
    Возвращает заказ, если он есть в базе

    Сперва ищет среди живых заказов. Если в живых нет, то ищем в архивах. Если найти не удалось - вернет None
    """
    order = None
    try:
        order = Order(number)
    except errors.BillingOrderNotFound:
        pass

    try:
        order = order or ArchiveOrder(number)
    except errors.BillingOrderNotFound:
        pass
    if not order:
        log.info("Order %s wasn't found in orders and in archived orders")
    return order
