# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Подписка

"""
import copy

from mpfs.config import settings
from mpfs.common.static.tags.billing import *
from mpfs.common.errors import billing as errors
from mpfs.core.billing.entity import StoredEntity, UniqEntity
from mpfs.core.billing.order import look_for_order
from mpfs.core.services.billing_service import BB
from mpfs.core.metastorage.control import billing_subscriptions
from mpfs.common.util import ctimestamp


class Subscription(UniqEntity):
    '''
    Подписка на автопродляемую услугу.
    Появляется при первой успешной оплате заказа с автопродлением.
    Нужна для связи Биллинговой сущности подписки и нашей услуги.
    '''

    table = billing_subscriptions
    nf_error = errors.BillingSubscriptionNotFound
    uniq_key_name = NUMBER

    def __init__(self, **kwargs):
        order = kwargs.pop(ORDER, None)
        service = kwargs.pop(SERVICE, None)

        params = copy.copy(kwargs)
        if order:
            params[DB_ID] = order
        elif service:
            params[SID] = service
        else:
            raise self.nf_error()

        StoredEntity.__init__(self, **params)
        UniqEntity.__init__(self, self._id, data=self._data)

    @classmethod
    def Create(classname, client, order, service):
        '''
        Делает подписку с заказа на услугу
        '''
        data = {
            DB_ID    : order.number,
            UID      : client.uid,
            SID      : service.sid,  # тут sid подписки пользователя, который купил себе или на группу
            CTIME    : ctimestamp(),
        }

        billing_subscriptions.insert(data)
        return classname(order=order.number, data=data)

    @classmethod
    def create_or_update(cls, client, order, service):
        try:
            subscription = Subscription(order=order.number)
        except errors.BillingSubscriptionNotFound:
            subscription = None

        if subscription:
            subscription.set(SID, service.sid)
            subscription.save()
        else:
            subscription = Subscription.Create(client, order, service)
        return subscription

    def unsubscribe(self, client):
        order = look_for_order(self.number)
        try:
            BB.unsubscribe(self.uid, client.ip, self.number, pid=order.pid)
        except errors.BigBillingBadResult:
            if settings.feature_toggles['raise_on_unsubscribe']:
                raise
