# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Процессинг админских задач

"""
import mpfs.engine.process
from mpfs.common.errors import billing as errors
from mpfs.common.static.tags.billing import *
from mpfs.common.util import time_to_str, ctimestamp
from mpfs.config import settings
from mpfs.core.billing.client import Client
from mpfs.core.billing.interface.mediabilling import queue_process_mediabilling_callback
from mpfs.core.billing.interface.orders import (
    queue_order_process_callback,
    queue_subscription_process_callback
)
from mpfs.core.billing.order import Order, OrderList
from mpfs.core.billing.processing.common import calculate_billing_time
from mpfs.core.billing.product import Product
from mpfs.core.billing.product.catalog import Catalog
from mpfs.core.billing.service import Service, ServiceList, ServiceCreate
from mpfs.core.services.billing_service import BB
from mpfs.core.user.base import User
from mpfs.core.user.constants import INVITES_REFERRAL_BONUS

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

THIRD_TARIFF_PLUS_PIDS = settings.billing['plus_pids']

def migrate_services(uid):
    ''''
    Миграция места из всего подряд в механизм услуг.

    1. Обработка сохраненных увеличений места.
        Берем из таблицы space_subscribes
        Данные в таблице есть с ноября 2012

    2. Обработка состояний.
        Берем состояния и оставляем только необработанные в шаге (1)

    3. Обработка инвайтов
        Берем состояния инвайтов и обрабатываем их отдельно.

    '''
    def migrate_service(product_name, add_date):
        item = {}
        product = Product(product_name)
        service = ServiceCreate(client, product)
        nbtime, lbtime = calculate_billing_time(service, lbtime=add_date)
        service.set(BTIME, nbtime)
        service.set(LASTBTIME, lbtime)
        service.set(CTIME, add_date)
        service.set(MTIME, add_date)
        service.set(ENABLED, True)
        item['service'] = service.sid
        item['product'] = product_name
        item['added'] = time_to_str(add_date)
        item['lbtime'] = time_to_str(service.lbtime)
        item['expires'] = time_to_str(service.btime) if service.btime else 'NEVER'
        return item

    result = []
    processed = {}
    collected = []

    user    = User(uid)
    client  = Client(uid)
    states  = user.states.list_all()
    stored  = dict(map(lambda x: (x['reason'], x), space_subscribes.get_all(uid=uid)))
    catalog = Catalog()

    '''
    1. Обработка сохраненных увеличений места
    '''
    for product, data in stored.iteritems():
        # рефералов обрабатываем отдельно, потому что информация может быть неполная
        if product == 'referral':
            continue

        pid = catalog.joined_product(product) if catalog.joined_product(product) else product
        item = {'product': pid, 'original': product, 'date': data['date']}
        processed[pid] = 1
        collected.append(item)

    '''
    2. Обработка общих состояний
    Берем из disk_info, добавляем, если не было данных в space_subscribes
    '''
    for state, value in states.get('global', {}).iteritems():
        if state in ('samsung_ultrabook', 'samsung_notebook', 'sony_notebook') and value != 'ok':
            continue

        pid = catalog.joined_product(state) if catalog.joined_product(state) else state

        if pid not in processed:
            item = {'product': pid, 'original': state, 'date': user.reg_time}
            processed[pid] = 1
            collected.append(item)

    '''
    3. а) Обработка бонуса приглашенных
    '''
    invited_by_referral = states.get('invites', {}).get('referral', False)
    if 'invited' not in processed and invited_by_referral:
        collected.append({'product': 'invited', 'original': 'invited', 'date': user.reg_time})

    '''
    Создаем услуги, игнорируя уже заведенные
    '''
    already_exist = ServiceList(client=client)
    already_exist_map = dict(map(lambda x: (x['product'].pid, x), already_exist))

    for item in collected:
        product, original, date = item['product'], item['original'], item['date']
        if product not in already_exist_map:
            try:
                log.info('try to migrate %s:%s (%s) to %s' % (uid, original, time_to_str(date), product))
                result.append(migrate_service(product, date))
            except errors.BillingProductNotFound, e:
                log.info('SKIP - no product %s' % product)

    '''
    3. б) Обработка бонусов рефералов
    '''
    referral_count = int(states.get('invites', {}).get('bonus_used', 0))/INVITES_REFERRAL_BONUS
    already_migrated_referral_count = len(filter(lambda x: x['pid'] == 'referral', already_exist))

    if referral_count > already_migrated_referral_count:
        for i in xrange (referral_count-already_migrated_referral_count):
            try:
                log.info('adding referral product %s:referral, count %s' % (uid, i))
                result.append(migrate_service('referral', user.reg_time))
            except Exception, e:
                log.error(e)

    return result


def remove_excess_referrals(uid):
    '''
    https://jira.yandex-team.ru/browse/CHEMODAN-11768

    Убираем ненужных реферралов из таблицы billing_services
    '''
    result = []

    user    = User(uid)
    client  = Client(uid)
    states  = user.states.list_all()

    services = ServiceList(client=client)
    referrals = filter(lambda x: x['pid'] == 'referral', services)
    real_referral_count = int(states.get('invites', {}).get('bonus_used', 0))/INVITES_REFERRAL_BONUS

    if len(referrals) > real_referral_count:
        for i in xrange(len(referrals) - real_referral_count):
            try:
                sid = referrals[i][SID]
                service = Service(sid, uid)
                log.info('removing excess referral product %s:%s, count %s' % (uid, sid, i))
                service.delete(disable=False)
                result.append(sid)
            except Exception, e:
                log.error(e)

    return result


def archive_lost_orders():
    """
    Берем все протухшие заказы и переводим в архив
    """
    now = ctimestamp()
    lost_interval = settings.billing['orders_lost_interval']
    border = now - lost_interval

    log.info('move orders to archive, base date is %s (%s), border is %s (%s)' % (now, time_to_str(now), border, time_to_str(border)))
    params = {CTIME_LT: border}
    orders = OrderList(**params)

    log.info('total %s orders (ctime < %s (%s)) need to be processed' % (len(orders), border, time_to_str(border)))
    error = None

    for item in orders:
        order = Order(item[NUMBER], data=item)
        client = Client(order.uid)
        try:
            bb_info = BB.check_order(order.uid, client.ip, order.number, pid=order.pid)
            if bb_info[STATUS] in (SUCCESS, OK):
                log.info('Order %s of user %s was successfully paid but for some reason we didn\'t create the service.'
                         'Intentionally putting callback handling task into the queue' % (order.number, order.uid))
                _create_service_from_bb_info(order, bb_info)
                continue
        except errors.BillingError:
            pass
        except Exception as e:
            error = e
            error_log.exception('unknown order error during archive_lost_orders %s error %s' % (item, e))
            continue
        order.close(LOST)
        order.archive()
        log.info('order %s:%s moved to archive' % (order.uid, order.number))
    if error:
        raise error


def _create_service_from_bb_info(order, bb_info_dict):
    if order.pid in THIRD_TARIFF_PLUS_PIDS:
        queue_process_mediabilling_callback(order.uid)
    elif bb_info_dict.get(PRODUCT_TYPE) == APP:
        queue_order_process_callback(
            order.uid, order.number, bb_info_dict.get(STATUS), bb_info_dict.get(STATUS_CODE), None, None)
    else:
        queue_subscription_process_callback(
            order.uid, order.number, bb_info_dict.get(STATUS), bb_info_dict.get(STATUS_CODE), None, None)
