# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Процессинг биллинга в различных видах

"""
import traceback

import mpfs.engine.process
from mpfs.common.errors import billing as errors, UserIsReadOnly
from mpfs.common.errors.billing import BillingServiceNotFound
from mpfs.common.static.tags.billing import *
from mpfs.common.util import ctimestamp, time_to_str
from mpfs.config import settings
from mpfs.core.billing.client import Client
from mpfs.core.billing.order import FindOrder, ArchiveOrder, look_for_order, Order
from mpfs.core.billing.order.subscription import Subscription
from mpfs.core.billing.processing import common
from mpfs.core.billing.processing import notify
from mpfs.core.billing.product import Product
from mpfs.core.billing.service import ArchiveService, DEFAULT_SERVICE_PID, get_service_by_pid
from mpfs.core.billing.service import Service, ServiceCreate, ServiceList
from mpfs.core.overdraft.utils import check_space_and_unblock_overdrafter
from mpfs.core.queue import mpfs_queue as queue
from mpfs.core.rostelecom_unlim.constants import ROSTELECOM_PID_TO_SERVICE_KEY
from mpfs.core.services.billing_service import BB
from mpfs.core.services.mediabilling_payments import (
    mediabilling_datetime_to_ts,
    MediabillingStatOrder,
    mediabilling_payment_service,
)


log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

MAX_SUBSCRIPTION_BTIME_LAG = settings.billing['max_subscription_btime_lag']
MAX_MEDIABILLING_SUBSCRIPTION_BTIME_LAG = settings.billing['max_mediabilling_subscription_btime_lag']
THIRD_TARIFF_PLUS_PIDS = settings.billing['plus_pids']
IS_PROLONG_BORDER = settings.billing['is_prolong_border']


def push_billing_commands():
    """
    Проставление задач на биллинг

    - взять все задачи, подлежащие биллингу в настоящий момент
    - поставить в локальную очередь
    """

    def select_and_push_services(border=None, state=None, command=None):
        params = {BTIME_LT: border, STATE: state}
        services = ServiceList(**params)
        log.info('total %s services (state = %s, btime < %s (%s)) need to be processed' %
                 (len(services), state, border, time_to_str(border))
        )

        for item in services:
            if item.get(PID) == DEFAULT_SERVICE_PID:
                continue

            uid, sid = item[UID], item[SID]

            if command == BILLING_SERVICE_SUBSCRIPTION_CANCELLED:
                queue.put({SID: sid, UID: uid}, command, deduplication_id='%s_%s_%s' % (command, uid, sid))
            elif command == BILLING_INAPP_SERVICE_SYNCRONIZE:
                queue.put({SID: sid, UID: uid}, command, deduplication_id=uid)
            else:
                queue.put({SID: sid, UID: uid}, command)
            log.info('service %s:%s pushed to %s' % (uid, sid, command))

    now = ctimestamp()
    log.info('pushing billing commands, base date is %s (%s)' % (now, time_to_str(now)))

    """
    Услуги без установленного состояния
    Выбирем и ставим на стандартный биллинг
    """
    standart_billing = now - settings.billing['billing_delay']
    select_and_push_services(
        border=standart_billing,
        state=[None, SUBSCRIPTION_CANCELLED],
        command=BILLING_SERVICE_STANDART_BILLING
    )

    """
    Неоплаченные в течении длительного времени услуги с подпиской
    Если услуга зависла в оплате более n дней (см. конфиг) ее надо отключить
    """
    regular_payment_cancel = now - settings.billing['subscription_payment_cancel']
    select_and_push_services(
        border=regular_payment_cancel,
        state=[WAITING_PAYMENT, PAYMENT_DELAYED, PAYMENT_CANCELLED],
        command=BILLING_SERVICE_SUBSCRIPTION_CANCELLED
    )

    """
    Услуги с подпиской, заждавшиеся коллбека от биллинга
    По умолчанию ждем один час от момента предыдущей оплаты
    """
    regular_payment_delay = now - settings.billing['subscription_payment_delay']
    select_and_push_services(
        border=regular_payment_delay,
        state=WAITING_PAYMENT,
        command=BILLING_SERVICE_SUBSCRIPTION_DELAYED
    )

    """
    MVP1
    In-app услуги при наступлении expire_time отправляем в грейс период. Меняется статус на INAPP_GRACE_PERIOD
    """
    select_and_push_services(
        border=now,
        state=INAPP_ACTIVE,
        command=BILLING_INAPP_SERVICE_SYNCRONIZE
    )

    """
    MVP1
    In-app услуги, которые долго находились в грейс периоде, удаляются
    """
    select_and_push_services(
        border=now - settings.billing['inapp_grace_period'],
        state=INAPP_GRACE_PERIOD,
        command=BILLING_INAPP_SERVICE_SYNCRONIZE
    )

    """In-app MVP2"""
    from mpfs.core.billing import InAppResyncManager
    InAppResyncManager.regular_resync_jobs()


def order_process_callback(uid, number, status, status_code, mode=None, trust_refund_id=None):
    """
    Обработка коллбека от ББ на обычный заказ
    Если заказ был сделан с указанием услуги - делаем продление
    """
    if mode == REFUND_RESULT:
        return process_refund_payment(uid, number, status, status_code, trust_refund_id)
    elif mode and mode not in (REFUND_RESULT, RESULT):
        raise errors.BigBillingUnknownMode('Unknown BigBilling mode: %s' % mode)

    try:
        order = FindOrder(uid, number)
        if order.sid:
            if order.get_state() in (SUCCESS, OK):
                # Защита от повторных успешных колбеков ББ, отосланных по ошибке. Мы не можем попросту проставлять
                # btime по результатам последней оплаты из ББ, поскольку услуга может быть продлена в любое время
                log.info('order %s has already been paid successfully' % number)
                return
            process_prolongate_payment(order, status, status_code)
        else:
            process_first_payment(order, status, status_code)
    except errors.BillingOrderNotFound, e:
        log.info('order %s not found' % number)
        _check_order_in_archive(number, status)


def mediabilling_subscription_process_ok_callback(uid, number, status, status_code, expires, pid,
                                                  price, currency):
    """
    Обработка коллбека подписки от mediabilling

    Коллбек может быть как по первому платежу, так и по последующим.
    Интересует только приведение услуг к корректному состоянию, поэтому:
      1. Если услуга есть - нужно проставить ей btime
      2. Если услуги нет - нужно выдать
    """
    client = Client(uid)

    def archive_order():
        try:
            order = Order(number)
            order.close(status, status_code)
            order.archive()
        except errors.BillingOrderNotFound:
            pass
        except Exception:
            error_log.error('failed to archive order %s. %s' % number, traceback.format_exc())

    product = Product(pid)
    service = get_service_by_pid(uid, pid)
    mediabilling_interval_end_ts = int(mediabilling_datetime_to_ts(expires))
    mediabilling_stat_order = MediabillingStatOrder(number=number, price=price, currency=currency)

    if service is not None:
        log.info('Processing regular payment uid=%s sid=%s status=%s pid=%s' % (
            uid,
            service.sid,
            status,
            pid
        ))

        try:
            old_btime = service.btime
            service.set(BTIME, mediabilling_interval_end_ts)
            # сбрасываем state (т.к. он может быть в состоянии "задержка платежа")
            service.set_regular_state()

            # если мы по какой-то причине не заархивировали Order на этапе первого платежа - делаем это сейчас.
            archive_order()

            # важно понимать, если у нас будет услуга длительностью меньше IS_PROLONG_BORDER, мы не сделаем запись в лог
            # чтобы не сделать лишнюю запись, когда btime изменился, но не значительно
            # может быть из-за:
            #   * пришел callback от mediabilling, но btime не изменился/продлился немного, для попытки списания
            #   * запустили ручную обработку
            if (mediabilling_interval_end_ts - old_btime) > IS_PROLONG_BORDER:
                common.stat(action=PROLONG_AUTO, client=client, order=mediabilling_stat_order, product=service.product,
                            status=status, status_code=status_code)
        except Exception:
            error_log.error(traceback.format_exc())
            raise

    else:
        log.info(
            'uid %s order %s first payment "%s (%s)" for %s (auto=%s)' %
            (uid, number, status, status_code, pid, mediabilling_stat_order.auto)
        )

        try:
            service = provide_mediabilling_service(client, product, btime=mediabilling_interval_end_ts)
            archive_order()
            common.stat(action=BUY_NEW, client=client, order=mediabilling_stat_order, product=product,
                        status=status, status_code=status_code)
        except Exception:
            error_log.error(traceback.format_exc())
            if service:
                service.delete(send_email=False)
            raise


def provide_mediabilling_service(client, product, btime):
    service = ServiceCreate(client, product, auto=False)
    service.set(BTIME, int(btime), save_to_db=False)
    service.enable(save_to_db=False)
    service.save()
    return service


def subscription_process_callback(uid, number, status, status_code, mode=None, trust_refund_id=None):
    """
    Обработка коллбека от ББ на подписку

    Коллбек может быть как по первому платежу, так и по последующим
    Для первого ищем Заказ, для последующих - Подписку

    Параметр 'mode' равен REFUND_RESULT, если это операция возврата денег. В таком случае подписка не осуществляется.
    В остальных случаях значение равно None, и результат зависит от значения 'status'.
    """

    # Обрабатываем ситуацию, когда mode отличен от None, на текущий момент, только в случае refund

    client = Client(uid)
    order = look_for_order(number)
    pid = order.pid if order else None
    bb_order_info = BB.check_order(client.uid, client.ip, number, pid=pid)
    log.info('check_order=%s' % bb_order_info)
    if not bb_order_info.has_btime:
        raise errors.BigBillingBadResult('There is no subs_until_ts in big billing order info %s' % bb_order_info)

    try:
        subscription = Subscription(order=number)
    except errors.BillingSubscriptionNotFound:
        subscription = None

    if mode == REFUND_RESULT:
        if status == SUCCESS:
            if bb_order_info.btime > ctimestamp():
                raise errors.BillingOrderUnexpectedStateError(
                    'Refund callback is received for %s:%s but btime (%s) in future' % (uid, number, bb_order_info.btime))

            if subscription:
                subscription.delete()

        return process_refund_payment(uid, number, status, status_code, trust_refund_id)
    elif mode and mode not in (REFUND_RESULT, RESULT):
        raise errors.BigBillingUnknownMode('Unknown BigBilling mode: %s' % mode)

    if subscription:
        process_regular_payment(subscription, status, status_code, bb_order_info)
        return

    try:
        order = FindOrder(uid, number)
    except errors.BillingOrderNotFound:
        try:
            ArchiveOrder(number)
        except errors.BillingOrderNotFound:
            # не нашли заказ ни среди активных, ни в архиве
            raise

        if bb_order_info.btime > ctimestamp():
            # заказ в архиве и по нему нет активной подписки, но в биллинге подписка еще действительна
            raise errors.BillingOrderUnexpectedStateError(
                'BB subscription end time %s in future, but order %s in archive without active subscription' %
                (bb_order_info.btime, number))

        log.info('Subscription doesn\'t exist but BB says subs_until_ts < current_time, do nothing')
        return
    process_first_payment(order, status, status_code, bb_btime=bb_order_info.btime)


def service_process_billing(uid, sid):
    """
    Стандартный биллинг услуги
    Команда очереди BILLING_SERVICE_STANDART_BILLING

    Если услуги уже нет, либо у нее время биллинга в будущем - выходим
    """
    client = Client(uid)

    try:
        service = Service(sid)
    except errors.BillingServiceNotFound:
        service, pid = None, 'N/A'
    else:
        pid = service.product.pid

    log.info('uid %s billing for %s:%s' % (client.uid, pid, sid))

    if not service:
        log.info('no need: service is not found')
        return

    if service.btime > ctimestamp():
        log.info('no need: billing time is in future')
        return

    if service.pid in ROSTELECOM_PID_TO_SERVICE_KEY:
        from mpfs.core.rostelecom_unlim.logic import process_billing_for_rostelecom_unlim_service
        process_billing_for_rostelecom_unlim_service(client, service)
        return

    if service.auto:
        service.remove(NOTIFIED, save_to_db=False)
        service.set(STATE, WAITING_PAYMENT, save_to_db=False)
        service.save()
    else:
        common.simple_delete_service(client, service)


def service_process_subscription_delayed(uid, sid):
    """
    Перевод услуги в состояние "заждались оплаты подписки"
    Команда очереди BILLING_SERVICE_SUBSCRIPTION_DELAYED
    """
    service = Service(sid)

    log.info(
        'uid %s subscription delayed for %s:%s' %
        (service.uid, service.product.pid, service.sid)
    )

    service.set(STATE, PAYMENT_DELAYED)


def service_process_subscription_cancelled(uid, sid):
    """
    Команда очереди BILLING_SERVICE_SUBSCRIPTION_CANCELLED
    """
    client = Client(uid)
    try:
        service = Service(sid)
    except BillingServiceNotFound:
        log.info('uid %s subscription %s was cancelled before process_subscription_cancelled' % (uid, sid))
        return
    pid = service.product.pid

    if pid in THIRD_TARIFF_PLUS_PIDS:
        #TODO(kis8ya): рассмотреть заведение отдельного таска под эту задачу
        # раньше были подписки для mediabilling-овых услуг, но сейчас уже их не заводим
        service_process_mediabilling_service_payment_delayed(client=client, service=service)
        return

    bb_order_info = BB.check_order(client.uid, client.ip, service.order, pid=pid)
    log.info('check_order=%s' % bb_order_info)

    if pid not in THIRD_TARIFF_PLUS_PIDS and not bb_order_info.has_btime:
        raise errors.BigBillingBadResult('There is no subs_until_ts in big billing order info %s' % bb_order_info)

    if bb_order_info.has_lbtime and bb_order_info.lbtime > service.lbtime:
        service.set(LASTBTIME, bb_order_info.lbtime)

    if pid not in THIRD_TARIFF_PLUS_PIDS and bb_order_info.btime > service.btime:
        log.info(
            'uid %s subscription %s:%s btime updated to %s' %
            (client.uid, service.product.pid, service.sid, bb_order_info.btime))
        service.set(BTIME, bb_order_info.btime)
        return

    if not bb_order_info.billing_is_active:
        log.info('uid %s subscription cancelled for %s:%s' % (client.uid, service.product.pid, service.sid))
        common.simple_delete_service(client, service)
        return

    # обработка случая, когда биллинг говорит, что еще будут списания, но btime сильно в прошлом
    if pid not in THIRD_TARIFF_PLUS_PIDS and ctimestamp() - bb_order_info.btime > MAX_SUBSCRIPTION_BTIME_LAG:
        log.warn('uid %s subscription %s:%s is active in billing, but btime %s is too old, subscription cancelled' %
                 (client.uid, service.product.pid, service.sid, bb_order_info.btime))
        common.simple_delete_service(client, service)
        return

    # по услуге еще будут попытки списания, ничего с ней не делаем
    log.info('uid %s subscription %s:%s wait for billing' % (client.uid, service.product.pid, service.sid))


def service_process_mediabilling_service_payment_delayed(client, service):
    if service.product.pid not in THIRD_TARIFF_PLUS_PIDS:
        return

    mediabilling_data = mediabilling_payment_service.get_last_active_interval_for_pids(uid=client.uid,
                                                                                       pids=THIRD_TARIFF_PLUS_PIDS)

    if not mediabilling_data:
        log.info('uid %s mediabilling service cancelled for %s:%s' % (client.uid, service.product.pid, service.sid))
        common.simple_delete_service(client, service, send_email=False)
        return

    if mediabilling_data:
        if (mediabilling_data[EXPIRES] and
                ctimestamp() - mediabilling_datetime_to_ts(mediabilling_data[EXPIRES]) > MAX_MEDIABILLING_SUBSCRIPTION_BTIME_LAG):
            raise errors.BillingOrderUnexpectedStateError(
                'uid %s mediabilling service %s:%s is active in mediabilling, but interval END time %s is too old' %
                (client.uid, service.product.pid, service.sid, mediabilling_data[EXPIRES]))
        else:
            log.info('service payment delayed but mediabilling is going to retry: %s' % str(mediabilling_data))


def service_process_mediabilling_subscription_cancelled(uid, sid):
    client = Client(uid)
    service = Service(sid)
    pid = service.product.pid
    bb_order_info = BB.check_order(client.uid, client.ip, service.order, pid=pid)
    log.info('check_order=%s' % bb_order_info)

    if bb_order_info['lastPaymentTime'] > service.lbtime:
        service.set(LASTBTIME, bb_order_info['lastPaymentTime'])

    if not bb_order_info['status'] == OK:
        log.info('uid %s subscription cancelled for %s:%s' % (client.uid, service.product.pid, service.sid))
        common.simple_delete_service(client, service)
        return


def inapp_service_syncronize(uid, sid):
    log.info('Syncronizing in-app service with sid=%s, uid=%s' % (sid, uid))
    try:
        service = Service(sid)
    except errors.BillingServiceNotFound:
        error_log.info('In-app service not found sid=%s, uid=%s' % (sid, uid))
        raise

    order = look_for_order(service.order)
    if not order:
        error_log.info('Order for in-app not found number=%s, uid=%s' % (service.number, uid))
        raise errors.BillingOrderNotFound()

    from mpfs.core.billing.inapp.mvp1_sync_logic import process_receipt
    process_receipt(uid, order.receipt, Product(service.pid).store_id)


def append_child_services(service, child_services):
    service.set(GROUP, True, save_to_db=False)
    service.set(CHILD_SIDS, [c.sid for c in child_services], save_to_db=False)


def create_service_for_uid(client, product, order, auto, parent_service=None, child_services=None,
                           enable_service=True, bb_btime=None, original_transaction_id=None, state=None):
    service = ServiceCreate(client, product, auto=auto)

    btime, lbtime = common.calculate_billing_time(service, lbtime=order.ctime)
    if auto and bb_btime is not None:
        btime = bb_btime
    service.set(BTIME, btime, save_to_db=False)
    service.set(LASTBTIME, lbtime, save_to_db=False)
    if state:
        service.set(STATE, state, save_to_db=False)

    if parent_service:
        service.set(GROUP, True, save_to_db=False)
        service.set(PARENT_SID, parent_service.sid, save_to_db=False)
    if child_services:
        append_child_services(service, child_services)

    if original_transaction_id:
        service.set(ORIGINAL_TRANSACTION_ID, original_transaction_id, save_to_db=False)

    if enable_service:
        service.enable(save_to_db=False)

    client.make_undeletable()
    service.set(ORDER, order.number, save_to_db=False)

    if order.group_name:
        service.set(GROUP_NAME, order.group_name, save_to_db=False)

    service.save()

    check_space_and_unblock_overdrafter(client.uid)

    return service


def process_first_payment(order, status, status_code, bb_btime=None):
    """
    Обработка первого платежа

    Общий метод, как для Подписки, так и для обычного Заказа

    Всегда считаем, что карта успешно привязалась и создаём автопродляемую услугу
    (https://st.yandex-team.ru/CHEMODAN-40840)
    """
    client = Client(order.uid)
    product = Product(order.pid)
    service = None
    subscription = None

    log.info(
        'uid %s order %s first payment "%s (%s)" for %s (auto=%s)' %
        (client.uid, order.number, status, status_code, product.pid, order.auto)
    )

    try:
        if status in (SUCCESS, OK):
            auto = order.auto
            if getattr(order, 'group_uids', None) is not None:
                # добавляем фейковую услугу пользователю, купившему групповую услугу: она не добавляет ему места
                service = create_service_for_uid(client, product, order, auto,
                                                 enable_service=False, bb_btime=bb_btime)
                services = []
                for uid in order.group_uids:
                    child_service = create_service_for_uid(Client(uid), product, order, auto,
                                                           parent_service=service, bb_btime=bb_btime)
                    services.append(child_service)

                append_child_services(service, services)
                service.enable()
            else:
                service = create_service_for_uid(client, product, order, auto, bb_btime=bb_btime)
            if auto:
                # подписка должна быть одна для главного пользователя, который сделал заказ в нее кладем sid этой услуги
                subscription = Subscription.Create(client, order, service)

            order.set(SID, service.sid)
            discount_template_id = order.get_discount_template_id()
            if discount_template_id:
                from mpfs.core.promo_codes.logic.discount_manager import DiscountManager
                DiscountManager.use_discount(client.uid, discount_template_id)

        order.close(status, status_code)
        order.archive()

        common.stat(action=BUY_NEW, client=client, order=order, product=product, status=status, status_code=status_code)
    except Exception as e:
        error_log.error(traceback.format_exc())
        if service:
            service.delete(send_email=not isinstance(e, UserIsReadOnly))
        if subscription:
            subscription.delete()
        raise


def process_regular_payment(subscription, status, status_code, bb_order_info):
    """
    Обработка последующих платежей
    """
    log.info('Processing regular payment uid=%s sid=%s status=%s' % (subscription.uid, subscription.sid, status))
    client = Client(subscription.uid)
    service = Service(subscription.sid)

    # если мы по какой-то причине не заархивировали Order на этапе первого платежа - делаем это сейчас.
    try:
        order = Order(subscription.number)
    except errors.BillingOrderNotFound:
        pass
    else:
        order.set("sid", subscription.sid)
        order.close(status, status_code)
        order.archive()

    sids = [subscription.sid]
    if service.child_sids:
        sids.extend(service.child_sids)

    if not len(sids):
        return

    group_member_services = []
    for sid in sids:
        current_service = Service(sid)
        product = current_service.product

        log.info(
            'uid %s subscription %s regular payment "%s (%s)" for %s:%s' %
            (client.uid, subscription.number, status, status_code, product.pid, current_service.sid)
        )

        try:
            old_btime = current_service.btime
            current_service.set(BTIME, bb_order_info.btime)
            if bb_order_info.lbtime:
                current_service.set(LASTBTIME, bb_order_info.lbtime)

            if status in (SUCCESS, OK):
                current_service.set_regular_state()
            current_service.remove(NOTIFIED)
            # важно понимать, если у нас будет услуга длительностью меньше IS_PROLONG_BORDER, мы не сделаем запись в лог
            if (bb_order_info.btime - old_btime) > IS_PROLONG_BORDER:
                common.stat(action=PROLONG_AUTO, client=client, order=subscription, product=product, status=status,
                            status_code=status_code)
        except Exception:
            error_log.error(traceback.format_exc())
            raise
        else:
            if sid != service.sid:
                group_member_services.append(current_service)

    try:
        service = Service(subscription.sid)
    except BillingServiceNotFound:
        if status != CANCELLED:
            raise BillingServiceNotFound
        return

    if status in (SUCCESS, OK):
        notify.regular_payment_accepted(client, service, group_member_services)
        service.remove(FIRST_PROLONGATION_FAIL)
    else:
        if bb_order_info.billing_is_active and not service.get(FIRST_PROLONGATION_FAIL):
            service.set(FIRST_PROLONGATION_FAIL, ctimestamp())
            notify.subscription_prolongate_fail(subscription.uid)
        # Закомментироно из-за https://st.yandex-team.ru/CHEMODAN-31532#1472644440000
        # Возможно, что этот код для чего-то реально нужен, но мы этого пока еще не поняли
        # subscription.unsubscribe(client)


def process_refund_payment(uid, order_id, status, status_code, trust_refund_id):
    if status not in (SUCCESS, OK):
        log.info('Unexpected refund status %s for %s:%s' % (status, uid, order_id))
        return

    order = ArchiveOrder(order_id)

    if order.get(REFUND_STATUS) in (SUCCESS, OK):
        log.info('Order %s:%s already refunded' % (uid, order_id))
        return

    if not order.sid:
        raise errors.BillingOrderUnexpectedStateError('Service for %s:%s not found during refund' % (uid, order_id))

    client = Client(uid)
    current_time = ctimestamp()
    services = _get_service_with_childs(order.sid)
    for current_service in services:
        new_btime, new_lbtime = common.calculate_billing_time_on_refund(current_service)
        current_service.set(BTIME, new_btime, save_to_db=False)
        current_service.set(LASTBTIME, new_lbtime, save_to_db=False)
        current_service.save()

        if new_btime < current_time:
            current_service.delete()

        common.stat(
            action=REFUND_ACTION, client=client, order=order, product=current_service.product, status=status,
            status_code=status_code)

    order.set_refund_status(status, trust_refund_id)
    log.info('Order %s:%s refunded' % (uid, order_id))


def process_prolongate_payment(order, status, status_code):
    """
    Обработка ручного продления услуги без Подписки
    """
    client = Client(order.uid)
    product = Product(order.pid)
    try:
        service = Service(order.sid)
    except BillingServiceNotFound:
        service = ArchiveService(order.sid)
        log.info('found service %s of user %s in archive, restoring' % (order.sid, order.uid))
        service.restore()

    sids = [order.sid]
    if service.child_sids:
        sids.extend(service.child_sids)

    if not len(sids):
        return

    for sid in sids:
        service = Service(sid)
        log.info(
            'uid %s order %s prolongate payment "%s (%s)" for %s:%s' %
            (client.uid, order.number, status, status_code, product.pid, service.sid)
        )

        try:
            service.process_prolongation(status)
            if status in (SUCCESS, OK):
                service.set(ORDER, order.number)
            order.close(status, status_code)
            order.archive()
            common.stat(action=PROLONG_MANUAL, client=client, order=order, product=product, status=status,
                        status_code=status_code)
        except Exception:
            error_log.error(traceback.format_exc())
            raise


def subscription_unsubscribe(client, service, subscription):
    """
    Обработка отключения подписки
    """
    subscription.unsubscribe(client)
    subscription.delete()

    services = [service]
    if service.child_sids:
        services.extend([Service(sid) for sid in service.child_sids])

    for srv in services:
        pid = srv.product.pid
        bb_order_info = BB.check_order(client.uid, client.ip, srv.order, pid=pid)
        log.info('check_order=%s' % bb_order_info)

        if bb_order_info.has_lbtime and bb_order_info.lbtime > srv.lbtime:
            srv.set(LASTBTIME, bb_order_info.lbtime)

        if bb_order_info.btime > srv.btime:
            log.info('uid %s subscription %s:%s btime updated to %s' %(client.uid, srv.product.pid, srv.sid, bb_order_info.btime))
            srv.set(BTIME, bb_order_info.btime)

        if srv.btime > ctimestamp():
            srv.remove(NOTIFIED, save_to_db=False)
            srv.set(AUTO, False, save_to_db=False)
            srv.set(STATE, SUBSCRIPTION_CANCELLED, save_to_db=False)
            srv.save()
        else:
            srv.delete()

    common.stat(action=UNSUBSCRIBE, client=client, order=subscription, product=service.product, service=service)


def _check_order_in_archive(number, status):
    """Проверяет наличие заказа в архиве. Если заказ найден и его статус совпадает с переданным, то считаем это повторным
    колбеком. Иначе бросаем соответствующее исключение.

    :param number: ID заказа
    :param status: статус из колбека
    """
    try:
        order = ArchiveOrder(number)
    except errors.BillingOrderNotFound:
        log.error('Neither Order nor ArchiveOrder with number %s were found.' % number)
        raise
    if order.state != status:
        log.error('Order %s is in archive but its status (%s) doesn\'t match to callback\'s status (%s)' %
                  (number, order.state, status))
        if _is_late_cancel_callback(order.state, status):
            log.info('Late CANCELLED callback, already archived order %s as LOST' % number)
            return
        raise errors.BillingOrderStatusConflict('Order status conflict: %s != %s' % (order.state, status))
    log.info('Order %s is in archive (repeated callback?)' % number)


def _is_late_cancel_callback(order_status, callback_status):
    return order_status == LOST and callback_status == CANCELLED


def _get_service_with_childs(sid):
    service = Service(sid)
    services = [service]
    if service.child_sids:
        services.extend([Service(sid) for sid in service.child_sids])
    return services
