# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Менеджер бизнес-процессов

"""
import traceback
import time
from datetime import datetime

import mpfs.engine.process
from mpfs.common.errors.billing import BillingProductIsSingletone

from mpfs.config import settings
from mpfs.common.util import logger, ctimestamp, datetime_to_unixtime
from mpfs.common.errors import billing as errors
from mpfs.common.static.tags.billing import *
from mpfs.core.billing.client import Client
from mpfs.core.billing.service import ServiceCreate, ServiceList, Service
from mpfs.core.billing.order.subscription import Subscription
from mpfs.core.billing.market import Market
from mpfs.core.filesystem.base import Filesystem
from mpfs.core.billing.events import (BillingBuyNewEvent, BillingProlongEvent, BillingOrderNewEvent,
                                      BillingUnsubscribeEvent, BillingDeleteEvent)
from mpfs.core.overdraft.utils import check_space_and_unblock_overdrafter
from mpfs.core.promo_codes.logic.discount_manager import DiscountManager, DiscountArchive
from mpfs.core.promo_codes.logic.errors import (
    AttemptToActivateDiscountRepeatedly,
    DiscountTemplateNotFound,
    AttemptToActivateArchivedDiscount,
)


default_log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

stat_billing_payments_log = mpfs.engine.process.get_stat_log("billing-payments")

BILLING_DEFAULT_MARKET = settings.billing['default_market']
FEATURE_TOGGLES_PROVIDE_DISCOUNTS_WITH_PRODUCTS = settings.feature_toggles['provide_discounts_with_products']
FEATURE_TOGGLES_PROVIDE_DISCOUNTS_WITH_PRODUCTS_FOR_YANDEXOIDS = settings.feature_toggles['provide_discounts_with_products_for_yandexoids']


def calculate_billing_time(service, lbtime=None):
    '''
    Высчитать даты биллингов

    Продукт может быть:
    - вечным
    - с периодом (delta)
    - с порогом (border)

    Для вечного продукта:
        btime = None

    Для периодического:
        если время биллинга не установлено, то current + delta
        если время биллинга установлено, то btime + delta

    Для порогового:
        btime = border

    '''
    lbtime = lbtime if lbtime else None
    btime, delta, border, current = None, None, None, ctimestamp()

    try:
        delta = service.product.get_period()
    except errors.BillingProductHasNoPeriod, e:
        try:
            border = service.product.get_border()
        except errors.BillingProductHasNoBorder, e:
            pass

    if delta:
        if lbtime:
            btime = datetime.fromtimestamp(lbtime) + delta
        elif not service.btime:
            btime = datetime.fromtimestamp(current) + delta
        elif service.btime:
            btime = datetime.fromtimestamp(service.btime) + delta
    elif border:
        btime = datetime.fromtimestamp(service.product.border)

    if btime:
        btime = datetime_to_unixtime(btime)

    return btime, current


def calculate_billing_time_on_refund(service):
    """
    Вычислить новые даты биллинга при рефанде

    Работает для сервисов с продуктами имеющими период
    :param service:
    :return:
    """
    product_period = service.product.get_period()  # для продуктов без периода бросит ошибку
    new_btime = datetime.fromtimestamp(service.btime) - product_period
    new_lbtime = datetime.fromtimestamp(service.lbtime) - product_period

    return datetime_to_unixtime(new_btime), datetime_to_unixtime(new_lbtime)


def simple_create_service(client, product, enable=True, attributes=None, send_email=True):
    service = None

    try:
        service = ServiceCreate(client, product)
        next_billing, last_billing = calculate_billing_time(service)
        service.set(BTIME, next_billing, save_to_db=False)
        service.set(LASTBTIME, last_billing, save_to_db=False)
        if enable:
            service.enable(send_email=send_email, save_to_db=False)
        if attributes:
            for key, value in attributes.iteritems():
                service.set_attribute(key, value, save_to_db=False)
        service.save()
    except BillingProductIsSingletone:
        # пробуем выдать скидку, если нужно, на случай, если это ретрай,
        # а прошлая попытка выдала услугу, но не выдала скидку
        provide_discount_for_product(client.uid, product)
        raise
    except Exception, e:
        error_log.error(traceback.format_exc())
        if service:
            service.delete()
        raise

    provide_discount_for_product(client.uid, product)
    check_space_and_unblock_overdrafter(client.uid)

    return service


def provide_paid_service_to_client(client, product, payment_method, auto=False, bb_time=None, receipt=None,
                                   original_transaction_id=None, state=None):
    """
    Выдать платную услугу пользователю.

    Отличаетя от simple_create_service тем, что умеет создавать заказ и подписку для услуги
    """
    from mpfs.core.billing import Order
    order = Order.create_and_save(client, product, payment_method, number=original_transaction_id, auto=auto)
    stat(action=ORDER_NEW, client=client, product=product, order=order)

    from mpfs.core.billing.processing.billing import create_service_for_uid
    service = create_service_for_uid(
        client, product, order, True, bb_btime=bb_time, original_transaction_id=original_transaction_id, state=state)
    if auto:
        Subscription.create_or_update(client, order, service)

    order.set(SID, service.sid)
    if receipt:
        order.set(RECEIPT, receipt)
    order.close(SUCCESS)
    order.archive()
    stat(action=BUY_NEW, client=client, order=order, product=product, status=SUCCESS)
    return service


def provide_discount_for_product(uid, product):
    """Добавляем скидку пользователю, если услуга предполагает скидку"""
    from mpfs.core.user.base import User
    if ((FEATURE_TOGGLES_PROVIDE_DISCOUNTS_WITH_PRODUCTS_FOR_YANDEXOIDS and User(uid).is_yateam()
            or FEATURE_TOGGLES_PROVIDE_DISCOUNTS_WITH_PRODUCTS)
            and hasattr(product, DISCOUNT_TEMPLATE_ID) and product.discount_template_id):
        try:
            DiscountManager.add_discount_to_user(uid, product.discount_template_id)
        except AttemptToActivateArchivedDiscount:
            DiscountManager.restore(uid, product.discount_template_id)
        except AttemptToActivateDiscountRepeatedly:
            default_log.info('Tried to provide discount (discount_template_id=%s) repeatedly for user uid=%s' % (
                product.discount_template_id,
                uid
            ))
        except DiscountTemplateNotFound:
            default_log.info('Tried to provide not found discount (discount_template_id=%s) for user uid=%s' % (
                product.discount_template_id,
                uid
            ))


def simple_delete_service(client, service, disable=True, send_email=True):
    """
    :param mpfs.core.billing.service.common.CommonService service:
    :return:
    """
    # Закомментироно из-за https://st.yandex-team.ru/CHEMODAN-31532#1472644440000
    # Возможно, что этот код для чего-то реально нужен, но мы этого пока еще не поняли
    # if service.auto:
    #     try:
    #         subscription = Subscription(service=service.sid, uid=client.uid)
    #     except errors.BillingSubscriptionNotFound, e:
    #         subscription = None
    #
    #     if subscription:
    #         try:
    #             subscription.unsubscribe(client)
    #         except Exception:
    #             error_log.error(traceback.format_exc())
    #         subscription.delete()
    service.delete(disable=disable, send_email=send_email)

    stat(action=DELETE, client=client, product=service.product, service=service)
    from mpfs.core.user.base import User
    user = User(client.uid)
    # Чистим кэш потому что при удалении услуг через ручку удаления услуг, сначала кэшируется список усулуг.
    # И проверка disk_pro_enabled может стать неверной
    user.cache_reset()
    if not user.disk_pro_enabled:
        unlimited_autouploading = user.get_unlimited_autouploading()
        if unlimited_autouploading['unlimited_video_autouploading_enabled']:
            user.set_unlimited_autouploading(video_status=False, video_reason='by_overdue')
    # Удаляем скидку пользователю, если услуга предполагает скидку
    if ((FEATURE_TOGGLES_PROVIDE_DISCOUNTS_WITH_PRODUCTS_FOR_YANDEXOIDS and user.is_yateam()
            or FEATURE_TOGGLES_PROVIDE_DISCOUNTS_WITH_PRODUCTS)
            and hasattr(service.product, DISCOUNT_TEMPLATE_ID) and service.product.discount_template_id):
        discount = DiscountManager.find_discount(client.uid, service.product.discount_template_id)
        if not discount:
            default_log.info(
                'Couldn\'t find discount among active for user. Possible fraud? uid=%s discount_template_id=%s' %
                (client.uid, service.product.discount_template_id))
            return
        DiscountManager.archive_discount(client.uid, service.product.discount_template_id)
    from mpfs.core.billing.processing.repair import recalculate_limit_by_services
    recalculate_limit_by_services(client.uid)


def stat(**params):
    try:
        client = params.get('client')
        service = params.get('service')
        product = params.get('product')
        status = params.get('status')
        status_code = params.get('status_code')
        order = params.get('order')
        action = params.get('action')
        from_ = params.get('from_')

        if status and status != SUCCESS:
            if status_code == PAYMENT_TIMEOUT:
                status = PAYMENT_TIMEOUT

        if action == BUY_NEW:
            BillingBuyNewEvent(uid=client.uid, product=product, price=order.price, currency=order.currency,
                               auto=order.auto, status=status, status_code=status_code).send()
        else:
            market = Market(client.attributes.market or BILLING_DEFAULT_MARKET)
            price = product.get_price(market)
            if action in (PROLONG_AUTO, PROLONG_MANUAL):
                BillingProlongEvent(uid=client.uid, product=product, price=price, currency=market.currency,
                                    auto=(action == PROLONG_AUTO), status=status, status_code=status_code).send()
            elif action == ORDER_NEW:
                BillingOrderNewEvent(uid=client.uid, product=product, price=price, currency=market.currency,
                                     auto=order.auto).send()
            elif action == UNSUBSCRIBE:
                BillingUnsubscribeEvent(uid=client.uid, product=product, price=price, currency=market.currency,
                                        auto=service.auto).send()
            elif action == DELETE:
                BillingDeleteEvent(uid=client.uid, product=product, price=price, currency=market.currency,
                                   auto=service.auto).send()

        uid = client.uid
        product_id = product.pid
        quota_used = None
        quota_limit = None

        number, price, currency, is_auto = None, None, None, None
        if order:
            number = order.number
            price = order.price
            currency = order.currency
            if order.auto is not None:
                is_auto = int(order.auto)

        if action in (BUY_NEW,):
            fs = Filesystem()
            quota = fs.quota.report(uid)
            quota_used = quota.get('used')
            quota_limit = quota.get('limit')
        elif action in (PROLONG_AUTO, PROLONG_MANUAL):
            market = Market(client.attributes.market or BILLING_DEFAULT_MARKET)
            price = product.get_price(market)
            currency = market.currency
            is_auto = 0
            if action == PROLONG_AUTO:
                is_auto = 1
        elif action in (ORDER_NEW,):
            market = Market(client.attributes.market or BILLING_DEFAULT_MARKET)
            currency = market.currency
            price = product.get_price(market)
            status = None
        elif action in (DELETE, UNSUBSCRIBE):
            market = Market(client.attributes.market or BILLING_DEFAULT_MARKET)
            currency = market.currency
            price = product.get_price(market)
            number = None
            status = None
            if service.auto is not None:
                is_auto = int(service.auto)

        log_data = {'uid': uid, 'action': action, 'order': number,
                    'product_id': product_id, 'auto': is_auto, 'status': status,
                    'price': price, 'currency': currency, 'used': quota_used,
                    'limit': quota_limit, 'from': from_}

        stat_billing_payments_log.info(log_data)

    except Exception as e:
        error_log.error('Failed to write `stat_log`: %s:%s', type(e), e)
