# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Процессинг маркетинга

"""
import copy

import mpfs.engine.process

from mpfs.common.static.tags.billing import (
    SUBS_RETRY_CHARGING_LIMIT,
    SUBS_RETRY_CHARGING_DELAY,
    INAPP,
    SUBS_GRACE_PERIOD
)
from mpfs.core.billing import InAppProductsHolder
from mpfs.core.billing.market import Market
from mpfs.core.billing.product.catalog import Catalog
from mpfs.core.billing.product import Product
from mpfs.core.services.billing_service import BB

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def _get_inapp_prices(mpfs_prices):
    prices = []
    for currency, value in mpfs_prices.items():
        region = 10000
        if currency == 'RUB':
            region = 225
        prices.append({
            'price': str(value),
            'currency': currency,
            'region_id': region,
            'start_ts': 0,
        })
    return prices


def get_inapp_subs_billing_representation(product):
    trust_product = {
        "product_id": product.pid,
        "package_name": product.package_name,
        "name": product.pid,
        "product_type": "inapp_subs",
        'local_names': _build_local_names(product),
        'prices': _get_inapp_prices(product.price)
    }

    if hasattr(product, SUBS_GRACE_PERIOD):
        trust_product[SUBS_GRACE_PERIOD] = getattr(product, SUBS_GRACE_PERIOD)
    if hasattr(product, 'subs_introductory_period_prices') and product.subs_introductory_period_prices:
        trust_product["subs_intro_period"] = "1D"  # not matter, just placeholder
        trust_product['subs_introductory_period_prices'] = _get_inapp_prices(product.subs_introductory_period_prices)

    return [trust_product]


def _build_local_names(product):
    local_names = []
    for locale, name in product.name.iteritems():
        local_names.append({
            'locale': locale.lower(),
            'name': name,
        })
    return local_names


def get_product_billing_representation(product, stamp):
    """
    Получить представление продукта для экспорта в Большой Биллинг

    Для каждого нашего продукта создается до трех продуктов в Биллинге:
    - обычный для разовой оплаты
    - для оплаты с продлением
    - для оплаты через in-app

    Для каждого продукта прописываются цены для всех рынков,
    даже если продукт не участвует на рынке.

    Продукт не отправляется в Биллинг, если его цены равны нулю
    """
    if product.pid in InAppProductsHolder.all_pids:
        return get_inapp_subs_billing_representation(product)

    def _construct(product, skeleton, auto=None, payment_method=None, parent=False):
        p = copy.deepcopy(skeleton)
        p['product_id'] = product.bb_pid(auto=auto, payment_method=payment_method)
        p['product_type'] = product.bb_product_type(auto=auto, payment_method=payment_method)

        if parent:
            p['parent_service_product_id'] = product.bb_pid()

        if auto:
            [(duration, amount)] = product.period.items()
            p['subs_period'] = '%s%s' % (amount, duration[0].upper())

            if hasattr(product, SUBS_RETRY_CHARGING_DELAY):
                p[SUBS_RETRY_CHARGING_DELAY] = getattr(product, SUBS_RETRY_CHARGING_DELAY)
            if hasattr(product, SUBS_RETRY_CHARGING_LIMIT):
                p[SUBS_RETRY_CHARGING_LIMIT] = getattr(product, SUBS_RETRY_CHARGING_LIMIT)
            if hasattr(product, SUBS_GRACE_PERIOD):
                p[SUBS_GRACE_PERIOD] = getattr(product, SUBS_GRACE_PERIOD)

        for price in p.get('prices'):
            if price.get('dt'):
                price['start_ts'] = float(price.pop('dt'))
            if price.get('region'):
                price['region_id'] = int(price.pop('region'))

        return p

    markets = map(lambda x: Market(x), Market.Available())
    regions = dict(map(lambda x: (x.code, x.region), markets))

    # Собираем цены и отбрасываем бесплатные
    prices = []
    for market in markets:
        if product.price.get(market.currency) != 0:
            price_struct = {
                'region': regions.get(market.code),
                'dt': stamp,
                'price': str(product.price.get(market.currency)),
                'currency': market.currency,
            }
            prices.append(price_struct)

    if not prices:
        return []

    # Собираем скелет
    skeleton = {
        'name': product.pid,
        'local_names': _build_local_names(product),
        'prices': prices,
        # В связи с изменениями в биллинге
        # https://st.yandex-team.ru/PAYSUP-332992
        'fiscal_nds': 'nds_18_118',
        'fiscal_title': u'Платное предоставление расширенных возможностей сервиса Яндекс.Диск',
    }

    # Собираем продукты
    products = [_construct(product, skeleton)]

    if product.auto:
        products.append(_construct(product, skeleton, auto=True, parent=True))

    if product.inapp:
        products.append(_construct(product, skeleton, payment_method=INAPP, parent=True))

    return products


def product_export(product, stamp, export_to_billing=True):
    """
    Экспорт продукта в Большой Биллинг
    """
    
    # Собираем продукты
    products = get_product_billing_representation(product, stamp)
    
    # Экспортируем
    if export_to_billing:
        for item in products:
            BB.create_product(item)
    
    return products


def line_export(line, stamp, export_to_billing=True):
    '''
    Экспорт линейки продуктов
    Проходим все рынки и набираем продукты из присутствующих там в указанной линейке
    Далее проводим обычный экспорт 
    '''
    products = {}
    results = []
    
    catalog = Catalog()
    for market_code in Market.Available():
        market = Market(market_code)
        for pid in catalog.get_pids(market, line, expand_group_templates=True):
            products[pid] = 1
            
    for pid in products:
        product = Product(pid)
        results.append(product_export(product, stamp, export_to_billing=export_to_billing))
        
    return results
