# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Менеджер нотификаций

"""
import time
from datetime import datetime

import mpfs.engine.process

from mpfs.config import settings
from mpfs.common.errors import billing as errors
from mpfs.common.static.tags.billing import *
from mpfs.core.billing.market import Market
from mpfs.core.billing.product.catalog import Catalog
from mpfs.core.billing.product import Product
from mpfs.core.email.logic import send_email_async_by_uid, is_campaign_enabled
from mpfs.core.pushnotifier.controller import PushController
from mpfs.core.billing.service import Service, ServiceList
from mpfs.core.pushnotifier.queue import PushQueue
from mpfs.core.services.passport_service import Passport
from mpfs.core.queue import mpfs_queue
from mpfs.common.util import time_to_str, ctimestamp

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

passport_entity = Passport()
push = PushController()
push_queue = PushQueue()


SUBSCRIPTION_PROLONGATE_FAIL_CAMPAIGN_NAME = 'subscription_prolongate_fail'


def push_notify_commands():
    """
    Проставление задач на уведомление об окончании действия услуг
    
    - взять все услуги, которые завершаются согласно правилам (2 недели для годовых, 3 дня для месячных) 
    - поставить задачи на отправку писем на локальной машине
    
    """

    def select_and_send_notify(lower=None, upper=None, product=None):
        params = {BTIME_GTE: lower, BTIME_LT: upper, PRODUCT: product}
        services = ServiceList(**params)
        log.info(
            'total %s services (product = %s, %s (%s) <= border < %s (%s)) need to be processed' %
            (len(services), product, lower, time_to_str(lower), upper, time_to_str(upper))
        )

        for item in services:
            del item[PRODUCT]
            sid = item[SID]
            uid = item[UID]

            service = Service(item[SID], data=item)
            if service.notified or service.auto:
                log.info('service %s:%s skipped, notified: %s auto: %s' % (uid, sid, service.notified, service.auto))
                continue

            user_info = passport_entity.userinfo(uid)

            args = {
                'username': user_info['username'],
                'locale': user_info['language'],
                'login': user_info['login'],
                'expired': service.btime,
                'size': service.product.attributes.amount,
                'product': service.product.pid,
                'sid': sid,
                'product_names': service.product.get_names(auto=service.auto),
            }

            template = service.product.attributes.notify_template
            data = {
                'email_to': user_info['email'],
                'template_name': template,
                'template_args': args,
            }

            mpfs_queue.put(data, 'send_email')
            log.info('service %s:%s send %s' % (uid, sid, template))

            service.set(NOTIFIED, ctimestamp())

    # Вычисляем границы выборок
    now = datetime.now()
    now_dropped_to_hour = int(time.mktime((now.year, now.month, now.day, now.hour, 0, 0, 0, 0, 0)))
    now_dropped_to_hour_plus_selection_interval = now_dropped_to_hour + settings.billing['notify_select_interval']

    # Загружаем pid продуктов для выборок
    products = {}
    for pid in Catalog().all_products():
        product = Product(pid)

        try:
            notify_before = product.attributes.notify_before
        except AttributeError:
            notify_before = None

        if notify_before:
            notify_before = product.attributes.notify_before
            if notify_before in products:
                products[notify_before][pid] = 1
            else:
                products[notify_before] = {pid: 1}

    log.info('pushing notify commands, base time is %s (%s)' % (time_to_str(now_dropped_to_hour), now_dropped_to_hour))

    for interval, pids in products.iteritems():
        if interval >= settings.billing['notify_period']:
            select_and_send_notify(
                upper=now_dropped_to_hour_plus_selection_interval + interval,
                lower=now_dropped_to_hour + interval,
                product=pids.keys(),
            )


def service_create(client, service):
    """
    Уведомление о предоставлении услуги
    """
    product = service.product
    template = product.get_template_on_create()
    if not template:
        return

    try:
        market = Market(client.attributes.market)
    except errors.BillingWrongMarket, e:
        market = Market(settings.billing['default_market'])

    user_info = passport_entity.userinfo(client.uid)

    args = {
        'username': user_info['username'],
        'locale': user_info['language'],
        'login': user_info['login'],
        'expired': service.btime,
        'size': product.attributes.amount,
        'price': product.get_price(market),
        'currency': market.currency,
        'product': product.pid,
        'product_names': product.get_names(auto=service.auto),
        'sid': service.sid,
    }

    if service.group and service.parent_sid:
        group_owner_uid = Service(service.parent_sid).uid
        owner_user_info = passport_entity.userinfo(group_owner_uid)
        args['owner_info'] = {
            'username': owner_user_info['public_name'],
            'locale': owner_user_info['language'],
            'login': owner_user_info['login'],
        }

    data = {
        'email_to': user_info['email'],
        'template_name': template,
        'template_args': args,
    }

    mpfs_queue.put(data, 'send_email')


def service_deleted(client, service):
    """
    Уведомление об удалении услуги
    """
    product = service.product
    try:
        market = Market(client.attributes.market)
    except errors.BillingWrongMarket, e:
        market = Market(settings.billing['default_market'])

    template = 'spacePack/EndingAction' if product.free(market) else 'billing/service/paidServiceDeleted'
    user_info = passport_entity.userinfo(client.uid)

    args = {
        'username': user_info['username'],
        'locale': user_info['language'],
        'login': user_info['login'],
        'expired': service.btime,
        'size': product.attributes.amount,
        'price': product.get_price(market),
        'currency': market.currency,
        'product': product.pid,
        'product_names': product.get_names(auto=service.auto),
        'sid': service.sid,
    }

    data = {
        'email_to': user_info['email'],
        'template_name': template,
        'template_args': args,
    }

    mpfs_queue.put(data, 'send_email')


def _notify_payment_accepted_personal(client, service):
    product = service.product
    market = Market(client.attributes.market)
    template = 'billing/payment/regularAccepted'
    user_info = passport_entity.userinfo(client.uid)
    email = user_info['email']

    args = {
        'username': user_info['username'],
        'locale': user_info['language'],
        'login': user_info['login'],
        'expired': service.btime,
        'size': product.attributes.amount,
        'price': product.get_price(market),
        'currency': market.currency,
        'product': product.pid,
        'product_names': product.get_names(auto=service.auto),
        'sid': service.sid,
    }

    data = {
        'email_to': email,
        'template_name': template,
        'template_args': args,
    }

    mpfs_queue.put(data, 'send_email')


def get_group_user_info(group_member_services):
    group_info = {}
    for srv in group_member_services:
        user_info = passport_entity.userinfo(srv.uid)
        group_info[srv.uid] = user_info
    return group_info


def get_group_template(group_info):
    group_template = []
    for user_info in group_info.itervalues():
        group_template.append({
            'username': user_info['public_name'],
            'locale': user_info['language'],
            'login': user_info['login'],
        })
    return group_template


def _notify_payment_accepted_group(client, service, group_member_services):
    product = service.product
    market = Market(client.attributes.market)
    template = 'billing/payment/regularAcceptedGroupOwner'
    user_info = passport_entity.userinfo(client.uid)
    email = user_info['email']

    group_info = get_group_user_info(group_member_services)

    args = {
        'username': user_info['username'],
        'locale': user_info['language'],
        'login': user_info['login'],
        'expired': service.btime,
        'size': product.attributes.amount,
        'price': product.get_price(market),
        'currency': market.currency,
        'product': product.pid,
        'product_names': product.get_names(auto=service.auto),
        'sid': service.sid,
        'group_info': get_group_template(group_info),
    }

    data = {
        'email_to': email,
        'template_name': template,
        'template_args': args,
    }

    mpfs_queue.put(data, 'send_email')

    if service.group and group_member_services:
        template = 'billing/payment/regularAcceptedGroupMember'
        for srv in group_member_services:
            # для каждого получателя места тоже отправляем письмо
            user_info = group_info[srv.uid]
            args = {
                'username': user_info['username'],
                'locale': user_info['language'],
                'login': user_info['login'],
                'expired': srv.btime,
                'size': product.attributes.amount,
                'product': product.pid,
                'product_names': product.get_names(auto=srv.auto),
                'sid': srv.sid,
            }

            data = {
                'email_to': user_info['email'],
                'template_name': template,
                'template_args': args,
            }

            mpfs_queue.put(data, 'send_email')


def regular_payment_accepted(client, service, group_member_services=None):
    """
    Уведомление о приеме платежа за подписку
    """
    if not service.group:
        _notify_payment_accepted_personal(client, service)
    elif group_member_services:
        _notify_payment_accepted_group(client, service, group_member_services)


def _notify_payment_declined_personal(client, service, status=None, status_code=None):
    product = service.product
    market = Market(client.attributes.market)
    template = 'billing/payment/regularNotAccepted'
    user_info = passport_entity.userinfo(client.uid)

    args = {
        'username': user_info['username'],
        'locale': user_info['language'],
        'login': user_info['login'],
        'expired': service.btime + settings.billing['subscription_payment_cancel'],
        'size': product.attributes.amount,
        'price': product.get_price(market),
        'currency': market.currency,
        'product': product.pid,
        'product_names': product.get_names(auto=service.auto),
        'status': status,
        'status_code': status_code,
        'sid': service.sid,
    }

    data = {
        'email_to': user_info['email'],
        'template_name': template,
        'template_args': args,
    }

    mpfs_queue.put(data, 'send_email')


def _notify_payment_declined_group(client, service, status, status_code, group_member_services):
    product = service.product
    market = Market(client.attributes.market)
    template = 'billing/payment/regularNotAcceptedGroupOwner'
    user_info = passport_entity.userinfo(client.uid)

    group_info = get_group_user_info(group_member_services)

    args = {
        'username': user_info['username'],
        'locale': user_info['language'],
        'login': user_info['login'],
        'expired': service.btime + settings.billing['subscription_payment_cancel'],
        'size': product.attributes.amount,
        'price': product.get_price(market),
        'currency': market.currency,
        'product': product.pid,
        'product_names': product.get_names(auto=service.auto),
        'status': status,
        'status_code': status_code,
        'sid': service.sid,
        'group_info': get_group_template(group_info),
    }

    data = {
        'email_to': user_info['email'],
        'template_name': template,
        'template_args': args,
    }

    mpfs_queue.put(data, 'send_email')

    if service.group and group_member_services:
        template = 'billing/payment/regularNotAcceptedGroupMember'
        for srv in group_member_services:
            # для каждого получателя места тоже отправляем письмо
            user_info = group_info[srv.uid]
            args = {
                'username': user_info['username'],
                'locale': user_info['language'],
                'login': user_info['login'],
                'expired': srv.btime + settings.billing['subscription_payment_cancel'],
                'size': product.attributes.amount,
                'product': product.pid,
                'product_names': product.get_names(auto=srv.auto),
                'sid': srv.sid,
            }

            data = {
                'email_to': user_info['email'],
                'template_name': template,
                'template_args': args,
            }

            mpfs_queue.put(data, 'send_email')


def regular_payment_declined(client, service, status=None, status_code=None, group_member_services=None):
    """
    Уведомление об отказе платежа за подписку
    """
    if not service.group:
        _notify_payment_declined_personal(client, service)
    elif group_member_services:
        _notify_payment_declined_group(client, service, status, status_code, group_member_services)


def card_binding_failed(client, service):
    """
    Уведомление о провале привязки карты при заведении подписки
    """
    product = service.product
    market = Market(client.attributes.market)
    template = 'billing/payment/bindingFailed'
    user_info = passport_entity.userinfo(client.uid)

    args = {
        'username': user_info['username'],
        'locale': user_info['language'],
        'login': user_info['login'],
        'expired': service.btime,
        'size': product.attributes.amount,
        'price': product.get_price(market),
        'currency': market.currency,
        'product': product.pid,
        'product_names': product.get_names(auto=service.auto),
        'sid': service.sid,
    }

    data = {
        'email_to': user_info['email'],
        'template_name': template,
        'template_args': args,
    }

    mpfs_queue.put(data, 'send_email')


def subscription_prolongate_fail(uid):
    if not is_campaign_enabled(SUBSCRIPTION_PROLONGATE_FAIL_CAMPAIGN_NAME, uid):
        return
    user_info = passport_entity.userinfo(uid)
    template_args = {
        'user_name': user_info.get('display_name') or user_info['login'],
    }
    send_email_async_by_uid(uid, SUBSCRIPTION_PROLONGATE_FAIL_CAMPAIGN_NAME, template_args)
