# -*- coding: utf-8 -*-
import mpfs.engine.process

from mpfs.common import errors
from mpfs.common.static import tags
from mpfs.core.billing.client import Client
from mpfs.core.billing.constants import (
    PRODUCT_INITIAL_10GB_ID,
    PRODUCT_INITIAL_3GB_ID,
    PRODUCT_APP_INSTALL_ID,
    PRODUCT_FILE_UPLOADED_ID,
    PRODUCT_PROMO_SHARED_ID,
    DEFAULT_SERVICE_ID,
)
from mpfs.core.billing.processing.repair import calculate_limit_by_services, recalculate_limit_by_services
from mpfs.core.billing.product.default_products_service import DefaultProductsService
from mpfs.core.billing.service import ServiceCreate, ServiceList, Service
from mpfs.core.billing.product import Product
from mpfs.core import billing, metastorage
from mpfs.common.static.tags.billing import SID
from mpfs.core.filesystem.quota import Quota
from mpfs.core.user import base as core_user_module
# Module mpfs.core.billing.product has module level variable 'catalog',
# so we can't get JOINED_PRODUCTS through already imported 'billing' module and have to import it separately.
from mpfs.core.billing.product.catalog import JOINED_PRODUCTS
from mpfs.core.billing.processing import common

USERSTATE_FILE_UPLOADED_KEY = PRODUCT_FILE_UPLOADED_ID
USERSTATE_PROMO_SHARED_KEY = PRODUCT_PROMO_SHARED_ID

PRODUCT_USERSTATE_MAP = {
    PRODUCT_APP_INSTALL_ID: tuple(JOINED_PRODUCTS.keys()),
    PRODUCT_FILE_UPLOADED_ID: (USERSTATE_FILE_UPLOADED_KEY,),
    PRODUCT_PROMO_SHARED_ID: (USERSTATE_PROMO_SHARED_KEY,),
}

PRODUCTS_TO_PUSH_THE_LIMITS_TO_10GB = (PRODUCT_APP_INSTALL_ID, PRODUCT_FILE_UPLOADED_ID, PRODUCT_PROMO_SHARED_ID,)

usrctl = mpfs.engine.process.usrctl()
dbctl = mpfs.engine.process.dbctl()


default_log = mpfs.engine.process.get_default_log()


def create_service(uid, pid, send_email=True):
    client = Client(uid)
    product = billing.Product(pid)
    try:
        common.simple_create_service(client, product, send_email=send_email)
    except errors.billing.BillingProductIsSingletone:
        pass


def create_services(uid, pids, send_email=True):
    """
    Create services for pid and set user states according to PRODUCT_USERSTATE_MAP.
    :param uid:
    :param pids:
    :return: dict {<product_id>: <service_instance>, ...}
    """
    services = {}
    client = Client(uid)
    user = client.attributes.user

    for pid in pids:
        product = billing.Product(pid)
        set_service_activated_state(user, pid)
        try:
            service = common.simple_create_service(client, product, send_email=send_email)
            services[pid] = service
        except errors.billing.BillingProductIsSingletone:
            pass

    return services


def delete_old_10gb_services(uid, services=None):
    client = Client(uid)
    if services is None:
        services = ServiceList(client=client)

    for service in services:
        if service[tags.billing.PID] in PRODUCTS_TO_PUSH_THE_LIMITS_TO_10GB:
            common.simple_delete_service(client, Service(sid=service[SID]), send_email=False)


def calculate_quota_without_old_10gb_services(services):
    filtered_services = []
    for service in services:
        if service[tags.billing.PID] in PRODUCTS_TO_PUSH_THE_LIMITS_TO_10GB:
            continue
        filtered_services.append(service)
    return calculate_limit_by_services(filtered_services)


def set_service_activated_state(user, product_id):
    """
    Utility properly setting user state for activated service.
    :param user: User instance.
    :param product_id: Service name from SERVICE_USER_STATE_MAP.
    :return:
    """
    # Some products may be represented by number of different states,
    # so we need to check all this states before set up new one.
    is_state_set = reduce(lambda ret, st: ret or bool(int(user.states.get(st) or 0)),
                          PRODUCT_USERSTATE_MAP[product_id],
                          False)
    if not is_state_set:
        user.states.update_and_save(PRODUCT_USERSTATE_MAP[product_id][0], 1,
                                    core_user_module.constants.DEFAULT_NAMESPACE)


def services_list(uid, pid=None):
    client = Client(uid)
    services = ServiceList(client=client)
    return filter(lambda x: x[tags.billing.PID] == pid, services) if pid else services


def is_proper_for_10gb_user(uid):
    is_share = uid == mpfs.engine.process.share_user()
    # stolen from https://github.yandex-team.ru/MPFS/MPFS/blob/5c39a0a0de7fdf488a5e4ef2ababc59fe6ac0306/common/lib/mpfs/core/user/standart.py#L61
    db_info = usrctl.check(uid)
    is_standart = db_info.get('type') == core_user_module.StandartUser.type
    return is_standart and not is_share


def can_pushto_10gb(uid, pid):
    """
    Check if product is available to enlarge user space or not.
    :param uid: User ID
    :param pid: Product ID
    :return: bool
    """
    has_initial_service = False
    if pid in PRODUCTS_TO_PUSH_THE_LIMITS_TO_10GB:
        has_initial_service = DefaultProductsService.has_initial_service(uid)
    return not has_initial_service and is_proper_for_10gb_user(uid)


def pushto_10gb(uid):
    user_services = services_list(uid)
    user_products = set(map(lambda s: s[tags.billing.PRODUCT].id, user_services))
    created_services = {}

    if PRODUCT_INITIAL_10GB_ID not in user_products and is_proper_for_10gb_user(uid):
        created_services = create_services(uid, set(PRODUCTS_TO_PUSH_THE_LIMITS_TO_10GB) - user_products)
        if PRODUCT_INITIAL_3GB_ID not in user_products:
            client = Client(uid)
            product = Product(PRODUCT_INITIAL_3GB_ID)
            service = ServiceCreate(client, product)
            nbtime, lbtime = billing.processing.common.calculate_billing_time(service)
            service.set(tags.billing.BTIME, nbtime)
            service.set(tags.billing.LASTBTIME, lbtime)
            service.set(tags.billing.ENABLED, True)
            created_services[PRODUCT_INITIAL_3GB_ID] = service

    return created_services


def ensure_initial_services(uid):
    user_services = services_list(uid)
    user_product_ids = set(map(lambda s: s[tags.billing.PRODUCT].id, user_services))
    user_service_ids = set(map(lambda s: s[tags.billing.SID], user_services))
    quota = get_current_user_quota_size(uid)

    if (PRODUCT_INITIAL_10GB_ID in user_product_ids and
            DEFAULT_SERVICE_ID not in user_service_ids):
        has_old_10gb_products = set(PRODUCTS_TO_PUSH_THE_LIMITS_TO_10GB) & user_product_ids
        if has_old_10gb_products:
            new_limit_by_services = calculate_quota_without_old_10gb_services(user_services)

            if new_limit_by_services >= quota:
                delete_old_10gb_services(uid, user_services)
                recalculate_limit_by_services(uid)
            else:
                default_log.info('User %s has incorrect quota %d, skipping' % (uid, quota))

        return

    if PRODUCT_INITIAL_3GB_ID in user_product_ids:
        not_provided_products = set(PRODUCTS_TO_PUSH_THE_LIMITS_TO_10GB) - user_product_ids
        new_limit_by_services = calculate_limit_by_services(user_services)
        for pid in not_provided_products:
            new_limit_by_services += billing.Product(pid).attributes.amount

        if new_limit_by_services >= quota:
            create_services(uid, not_provided_products, send_email=False)
            recalculate_limit_by_services(uid)
        else:
            default_log.info('User %s has incorrect quota %d, skipping' % (uid, quota))

        return

    new_limit_by_services = calculate_quota_without_old_10gb_services(user_services)
    new_limit_by_services += billing.Product(PRODUCT_INITIAL_10GB_ID).attributes.amount

    if new_limit_by_services >= quota:
        delete_old_10gb_services(uid, user_services)
        create_service(uid, PRODUCT_INITIAL_10GB_ID, send_email=False)
        recalculate_limit_by_services(uid)
    else:
        default_log.info('User %s has incorrect quota %d, skipping' % (uid, quota))


def get_current_user_quota_size(uid):
    try:
        return Quota().limit(uid=uid)
    except errors.StorageEmptyDiskInfo:
        return 0
