# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Продукт

"""
import re
import time
from collections import OrderedDict

from dateutil.relativedelta import *

import mpfs.engine.process
from mpfs.common.static.tags import experiment_names

from mpfs.common.static.tags.billing import *
from mpfs.common.errors import billing as errors, StorageInitUser
from mpfs.common.static.tags.experiment_names import DISABLE_ADS_FOR_MAIL
from mpfs.common.util.experiments.logic import experiment_manager
from mpfs.core.billing.client import PaymentMethods
from mpfs.core.billing.product.catalog import Catalog
from mpfs.core.services.mediabilling_payments import mediabilling_payment_service
from mpfs.core.services.passport_service import passport
from mpfs.core.user.constants import DEFAULT_LOCALE
from mpfs.config import settings

BEST_OFFER_PIDS = settings.billing_products['is_best_offer']['ids']
BEST_OFFER_PATTERN = settings.billing_products['is_best_offer']['regexp_pattern']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()

# FIXME: this variable has the same name as module '.catalog' and lead to misunderstanding on imports in foreign modules
catalog = Catalog()


def get_default_products_line():
    return PRIMARY_2019_V4


class ProductAttributes(object):
    pass


class Product(object):

    def __init__(self, pid):
        for key, value in catalog.product(pid).iteritems():
            if key == ATTRIBUTES:
                values = ProductAttributes()
                for a, b in value.iteritems():
                    setattr(values, a, b)
                setattr(self, key, values)
            else:
                setattr(self, key, value)

        self.pid = pid

    @staticmethod
    def parse(product_str):
        if ':' in product_str:
            product_str_chunks = product_str.split(':')
            return Product(product_str_chunks[0]), {'product.amount': int(product_str_chunks[1])}
        else:
            return Product(product_str), {}

    def get_template_on_create(self):
        attributes = getattr(self, ATTRIBUTES, None)
        if not attributes:
            return None
        return getattr(attributes, NOTIFY_TEMPLATE_ON_CREATE, None)

    def get_price(self, market):
        return self.price.get(market.currency)

    def free(self, market=None):
        if market:
            return self.price.get(market.currency) == 0
        else:
            for currency, value in self.price.iteritems():
                if value != 0:
                    return False
            return True

    def get_space(self):
        return self.attributes.amount

    def is_month(self):
        if not self.period:
            return False
        return self.get_period() == relativedelta(months=1)

    def is_year(self):
        if not self.period:
            return False
        return self.get_period() == relativedelta(years=1)

    def get_period(self):
        if not self.period:
            raise errors.BillingProductHasNoPeriod(self.pid)

        [(duration, amount)] = self.period.items()
        return relativedelta(**{duration: +amount})

    def get_border(self):
        if hasattr(self, BORDER):
            return self.border
        else:
            raise errors.BillingProductHasNoBorder(self.pid)

    def _bb_suffix(self, payment_method=None, auto=False):
        '''
        Выдает суффикс для ID продукта в ББ

        Для inapp продуктов = inapp
        Для автопродления = subs
        '''
        suffix = APP

        if auto:
            suffix = SUBS
        elif payment_method == INAPP:
            suffix = INAPP

        return suffix

    def bb_pid(self, **kwargs):
        suffix = self._bb_suffix(**kwargs)
        return '%s_%s' % (self.pid, suffix)

    def bb_product_type(self, **kwargs):
        return self._bb_suffix(**kwargs)

    def get_names(self, auto=False):
        if not auto:
            return self.name
        try:
            return self.subscription_name
        except AttributeError:
            return self.name

    def get_is_best_offer(self):
        if re.match(BEST_OFFER_PATTERN, self.pid) or self.pid in BEST_OFFER_PIDS:
            return True
        return False

    def get_is_yandex_plus(self):
        attributes = getattr(self, ATTRIBUTES, None)
        if not attributes:
            return False
        return getattr(attributes, IS_YANDEX_PLUS, False)

    def get_disables_mail_ads(self):
        attributes = getattr(self, ATTRIBUTES, {})
        return getattr(attributes, DISABLES_MAIL_ADS, False)


class ProductCard(object):
    LINE_TO_DISCOUNT_PERCENTAGE = {
        PRIMARY_2018_DISCOUNT_10: 10,
        PRIMARY_2018_DISCOUNT_20: 20,
        PRIMARY_2018_DISCOUNT_30: 30,
        PRIMARY_2019_DISCOUNT_10: 10,
        PRIMARY_2019_DISCOUNT_20: 20,
        PRIMARY_2019_DISCOUNT_30: 30,
        PRIMARY_2019_V2_DISCOUNT_10: 10,
        PRIMARY_2019_V2_DISCOUNT_20: 20,
        PRIMARY_2019_V2_DISCOUNT_30: 30,
        PRIMARY_2019_V3_DISCOUNT_10: 10,
        PRIMARY_2019_V3_DISCOUNT_20: 20,
        PRIMARY_2019_V3_DISCOUNT_30: 30,
        PRIMARY_2019_V4_DISCOUNT_10: 10,
        PRIMARY_2019_V4_DISCOUNT_20: 20,
        PRIMARY_2019_V4_DISCOUNT_30: 30,
        THIRD_TARIFF_2019: 0,
        THIRD_TARIFF_2019_DISCOUNT_10: 10,
        THIRD_TARIFF_2019_DISCOUNT_20: 20,
        THIRD_TARIFF_2019_DISCOUNT_30: 30,
        YANDEX_PLUS_UPSALE: 0,
    }

    PRODUCTS_GROUPED_BY_SPACE = OrderedDict([
        ('100gb', {
            SPACE_IN_UNITS: 100,
            UNITS: {
                RU_RU: u'ГБ',
                UK_UA: u'ГБ',
                EN_EN: u'GB',
                TR_TR: u'GB',
            },
            LINES: {
                PRIMARY_2018: {
                    'month_pid': '100gb_1m_2018',
                    'year_pid': '100gb_1y_2018',
                },
                PRIMARY_2018_DISCOUNT_10: {
                    'month_pid': '100gb_1m_2018_discount_10',
                    'year_pid': '100gb_1y_2018_discount_10',
                },
                PRIMARY_2018_DISCOUNT_20: {
                    'month_pid': '100gb_1m_2018_discount_20',
                    'year_pid': '100gb_1y_2018_discount_20',
                },
                PRIMARY_2018_DISCOUNT_30: {
                    'month_pid': '100gb_1m_2018_discount_30',
                    'year_pid': '100gb_1y_2018_discount_30',
                },
                PRIMARY_2019: {
                    'month_pid': '100gb_1m_2019',
                    'year_pid': '100gb_1y_2019',
                },
                PRIMARY_2019_DISCOUNT_10: {
                    'month_pid': '100gb_1m_2019_discount_10',
                    'year_pid': '100gb_1y_2019_discount_10',
                },
                PRIMARY_2019_DISCOUNT_20: {
                    'month_pid': '100gb_1m_2019_discount_20',
                    'year_pid': '100gb_1y_2019_discount_20',
                },
                PRIMARY_2019_DISCOUNT_30: {
                    'month_pid': '100gb_1m_2019_discount_30',
                    'year_pid': '100gb_1y_2019_discount_30',
                },
                PRIMARY_2019_V2: {
                    'month_pid': '100gb_1m_2019_v2',
                    'year_pid': '100gb_1y_2019_v2',
                },
                PRIMARY_2019_V2_DISCOUNT_10: {
                    'month_pid': '100gb_1m_2019_v2_discount_10',
                    'year_pid': '100gb_1y_2019_v2_discount_10',
                },
                PRIMARY_2019_V2_DISCOUNT_20: {
                    'month_pid': '100gb_1m_2019_v2_discount_20',
                    'year_pid': '100gb_1y_2019_v2_discount_20',
                },
                PRIMARY_2019_V2_DISCOUNT_30: {
                    'month_pid': '100gb_1m_2019_v2_discount_30',
                    'year_pid': '100gb_1y_2019_v2_discount_30',
                },
                PRIMARY_2019_V3: {
                    'month_pid': '100gb_1m_2019_v3',
                    'year_pid': '100gb_1y_2019_v3',
                },
                PRIMARY_2019_V3_DISCOUNT_10: {
                    'month_pid': '100gb_1m_2019_v3_discount_10',
                    'year_pid': '100gb_1y_2019_v3_discount_10',
                },
                PRIMARY_2019_V3_DISCOUNT_20: {
                    'month_pid': '100gb_1m_2019_v3_discount_20',
                    'year_pid': '100gb_1y_2019_v3_discount_20',
                },
                PRIMARY_2019_V3_DISCOUNT_30: {
                    'month_pid': '100gb_1m_2019_v3_discount_30',
                    'year_pid': '100gb_1y_2019_v3_discount_30',
                },
                PRIMARY_2019_V4: {
                    'month_pid': '100gb_1m_2019_v4',
                    'year_pid': '100gb_1y_2019_v4',
                },
                PRIMARY_2019_V4_DISCOUNT_10: {
                    'month_pid': '100gb_1m_2019_v4_discount_10',
                    'year_pid': '100gb_1y_2019_v4_discount_10',
                },
                PRIMARY_2019_V4_DISCOUNT_20: {
                    'month_pid': '100gb_1m_2019_v4_discount_20',
                    'year_pid': '100gb_1y_2019_v4_discount_20',
                },
                PRIMARY_2019_V4_DISCOUNT_30: {
                    'month_pid': '100gb_1m_2019_v4_discount_30',
                    'year_pid': '100gb_1y_2019_v4_discount_30',
                },
            },
        }),
        ('1tb', {
            SPACE_IN_UNITS: 1,
            UNITS: {
                RU_RU: u'ТБ',
                UK_UA: u'ТБ',
                EN_EN: u'TB',
                TR_TR: u'TB',
            },
            LINES: {
                PRIMARY_2018: {
                    'month_pid': '1tb_1m_2018',
                    'year_pid': '1tb_1y_2018',
                },
                PRIMARY_2018_DISCOUNT_10: {
                    'month_pid': '1tb_1m_2018_discount_10',
                    'year_pid': '1tb_1y_2018_discount_10',
                },
                PRIMARY_2018_DISCOUNT_20: {
                    'month_pid': '1tb_1m_2018_discount_20',
                    'year_pid': '1tb_1y_2018_discount_20',
                },
                PRIMARY_2018_DISCOUNT_30: {
                    'month_pid': '1tb_1m_2018_discount_30',
                    'year_pid': '1tb_1y_2018_discount_30',
                },
                PRIMARY_2019: {
                    'month_pid': '1tb_1m_2019',
                    'year_pid': '1tb_1y_2019',
                },
                PRIMARY_2019_DISCOUNT_10: {
                    'month_pid': '1tb_1m_2019_discount_10',
                    'year_pid': '1tb_1y_2019_discount_10',
                },
                PRIMARY_2019_DISCOUNT_20: {
                    'month_pid': '1tb_1m_2019_discount_20',
                    'year_pid': '1tb_1y_2019_discount_20',
                },
                PRIMARY_2019_DISCOUNT_30: {
                    'month_pid': '1tb_1m_2019_discount_30',
                    'year_pid': '1tb_1y_2019_discount_30',
                },
                PRIMARY_2019_V2: {
                    'month_pid': '1tb_1m_2019_v2',
                    'year_pid': '1tb_1y_2019_v2',
                },
                PRIMARY_2019_V2_DISCOUNT_10: {
                    'month_pid': '1tb_1m_2019_v2_discount_10',
                    'year_pid': '1tb_1y_2019_v2_discount_10',
                },
                PRIMARY_2019_V2_DISCOUNT_20: {
                    'month_pid': '1tb_1m_2019_v2_discount_20',
                    'year_pid': '1tb_1y_2019_v2_discount_20',
                },
                PRIMARY_2019_V2_DISCOUNT_30: {
                    'month_pid': '1tb_1m_2019_v2_discount_30',
                    'year_pid': '1tb_1y_2019_v2_discount_30',
                },
                PRIMARY_2019_V3: {
                    'month_pid': '1tb_1m_2019_v3',
                    'year_pid': '1tb_1y_2019_v3',
                },
                PRIMARY_2019_V3_DISCOUNT_10: {
                    'month_pid': '1tb_1m_2019_v3_discount_10',
                    'year_pid': '1tb_1y_2019_v3_discount_10',
                },
                PRIMARY_2019_V3_DISCOUNT_20: {
                    'month_pid': '1tb_1m_2019_v3_discount_20',
                    'year_pid': '1tb_1y_2019_v3_discount_20',
                },
                PRIMARY_2019_V3_DISCOUNT_30: {
                    'month_pid': '1tb_1m_2019_v3_discount_30',
                    'year_pid': '1tb_1y_2019_v3_discount_30',
                },
                PRIMARY_2019_V4: {
                    'month_pid': '1tb_1m_2019_v4',
                    'year_pid': '1tb_1y_2019_v4',
                },
                PRIMARY_2019_V4_DISCOUNT_10: {
                    'month_pid': '1tb_1m_2019_v4_discount_10',
                    'year_pid': '1tb_1y_2019_v4_discount_10',
                },
                PRIMARY_2019_V4_DISCOUNT_20: {
                    'month_pid': '1tb_1m_2019_v4_discount_20',
                    'year_pid': '1tb_1y_2019_v4_discount_20',
                },
                PRIMARY_2019_V4_DISCOUNT_30: {
                    'month_pid': '1tb_1m_2019_v4_discount_30',
                    'year_pid': '1tb_1y_2019_v4_discount_30',
                },
            },
        }),
        ('3tb', {
            SPACE_IN_UNITS: 3,
            UNITS: {
                RU_RU: u'ТБ',
                UK_UA: u'ТБ',
                EN_EN: u'TB',
                TR_TR: u'TB',
            },
            LINES: {
                YANDEX_PLUS_UPSALE: {
                    'month_pid': 'ru.yandex.web.disk.native.1month.autorenewable.notrial.disk_basic.750',
                    'year_pid': 'ru.yandex.web.disk.native.1year.autorenewable.notrial.disk_basic.6500',
                },
                THIRD_TARIFF_2019: {
                    'month_pid': '3tb_1m_2019',
                    'year_pid': '3tb_1y_2019',
                },
                THIRD_TARIFF_2019_DISCOUNT_10: {
                    'month_pid': '3tb_1m_2019_discount_10',
                    'year_pid': '3tb_1y_2019_discount_10',
                },
                THIRD_TARIFF_2019_DISCOUNT_20: {
                    'month_pid': '3tb_1m_2019_discount_20',
                    'year_pid': '3tb_1y_2019_discount_20',
                },
                THIRD_TARIFF_2019_DISCOUNT_30: {
                    'month_pid': '3tb_1m_2019_discount_30',
                    'year_pid': '3tb_1y_2019_discount_30',
                },
            }
        }),
    ])

    @classmethod
    def get_discount_percentage_by_line(cls, line):
        return cls.LINE_TO_DISCOUNT_PERCENTAGE[line]

    @classmethod
    def get_products(cls, line, market=RU, locale=RU_RU, discount=None, percentage=None):
        products_info = []
        for products_info_by_space in cls.PRODUCTS_GROUPED_BY_SPACE.itervalues():
            if products_info_by_space is None:
                continue
            card_info = cls._get_card(line, products_info_by_space, market, locale, discount, percentage)
            if card_info is None:
                continue
            products_info.append(card_info)
        return products_info

    @classmethod
    def _get_card(cls, line, products_info_by_space, market=RU, locale=RU_RU, discount=None, percentage=None):
        product_line_info = products_info_by_space[LINES].get(line)
        if not product_line_info:
            return None

        products_info_by_period = {}
        for display_period, period_key in (('month', 'month_pid'), ('year', 'year_pid')):
            product = Product(product_line_info[period_key])
            products_info_by_period[display_period] = cls._get_product_info(product, market)
        # используем последний продукт для получения места в байтах

        card_info = {
            'currency': market.currency,
            'space': product.attributes.amount,
            'display_space': products_info_by_space[SPACE_IN_UNITS],
            'display_space_units': products_info_by_space[UNITS][locale],
            'periods': products_info_by_period,
            IS_BEST_OFFER: product.get_is_best_offer(),
            IS_YANDEX_PLUS: product.get_is_yandex_plus(),
        }
        if experiment_manager.is_feature_active(DISABLE_ADS_FOR_MAIL):
            card_info[DISABLES_MAIL_ADS] = product.get_disables_mail_ads()

        if discount or percentage:
            card_info['discount'] = {
                'percentage': percentage or cls.get_discount_percentage_by_line(line),
            }
            if discount.end_datetime is not None:
                card_info['discount']['active_until_ts'] = int(time.mktime(discount.end_datetime.timetuple()))

        return card_info

    @classmethod
    def get_third_tariff_products(cls, user_has_plus, country='ru', market=RU, locale=RU_RU,
                                  discount=None, percentage=None, is_country_in_passport=False,
                                  uid=None, ip_address=None):
        try:
            mediabilling_products = mediabilling_payment_service.native_products(ip_address, uid)
        except Exception:
            mediabilling_products = []
        if not is_country_in_passport or country != 'ru' or market.code != RU or user_has_plus or \
                not mediabilling_products or not settings.feature_toggles['yandex_plus_upsale_2019_enabled']:
            if percentage == 10:
                line = THIRD_TARIFF_2019_DISCOUNT_10
            elif percentage == 20:
                line = THIRD_TARIFF_2019_DISCOUNT_20
            elif percentage == 30:
                line = THIRD_TARIFF_2019_DISCOUNT_30
            else:
                line = THIRD_TARIFF_2019
        else:
            line = YANDEX_PLUS_UPSALE
        return cls.get_products(line, market, locale, discount)

    @classmethod
    def _get_product_info(cls, product, market):
        product_info = {
            'price': product.get_price(market),
            'product_id': product.pid,
        }
        return product_info

    @classmethod
    def get_products_for_not_initialized_user(cls, uid, locale, ip_address, skip_third_tariff=False):
        from mpfs.core.billing import Market

        line = get_default_products_line()
        userinfo = passport.userinfo(uid)
        is_country_in_passport = 'country' in userinfo
        country = userinfo.get('country', 'ru')
        has_plus = userinfo.get('has_plus', False)
        market = Market.from_country(country)
        ip = mpfs.engine.process.hostip()
        payment_methods = PaymentMethods(uid, ip, market, check_with_billing=False).all_applicable()
        items = cls.get_products(line, market, locale)
        if not skip_third_tariff:
            items.extend(cls.get_third_tariff_products(has_plus, country, market, locale,
                                                       is_country_in_passport=is_country_in_passport,
                                                       uid=uid, ip_address=ip_address))

        result = {
            'items': items,
            'payment_methods': payment_methods,
        }
        return result

    @classmethod
    def get_current_products_for(cls, uid, locale=DEFAULT_LOCALE, ip=None, skip_third_tariff=False):
        from mpfs.core.billing import Client, Market
        if not locale:
            locale = DEFAULT_LOCALE

        try:
            client = Client(uid)
        except StorageInitUser:
            return cls.get_products_for_not_initialized_user(uid, locale, ip)

        userinfo = passport.userinfo(uid)
        is_country_in_passport = 'country' in userinfo
        country = userinfo.get('country', 'ru')
        has_plus = userinfo.get('has_plus', False)

        if not client.attributes.market:
            from mpfs.core.user.base import User
            user = User(uid)
            if not user.is_standart:
                raise errors.BillingClientNotBindedToMarket(uid)
            market = Market.from_country(country)
            Client(uid).bind_market(market)
            client = Client(uid)
        else:
            market = Market(client.attributes.market)

        from mpfs.core.promo_codes.logic.discount_manager import DiscountManager

        discount = DiscountManager.get_cheapest_available_discount(uid)
        line = get_default_products_line()
        percentage = None
        if discount:
            line = discount.provided_line
            percentage = cls.LINE_TO_DISCOUNT_PERCENTAGE[line]
        items = cls.get_products(line, market, locale, discount)
        if not skip_third_tariff:
            items.extend(cls.get_third_tariff_products(has_plus, country, market, locale, discount,
                                                       percentage, is_country_in_passport, uid=uid, ip_address=ip))
        result = {
            'items': items,
            'payment_methods': client.list_payment_methods(),
        }
        return result
