# -*- coding: utf-8 -*-
"""

MPFS
BILLING

Услуга

"""
import copy

import mpfs.engine.process

from mpfs.common.static.tags.billing import *
from mpfs.common.errors import billing as errors
from mpfs.core.billing.entity import UniqEntity
from mpfs.core.billing.product import Product
from mpfs.core.billing.service.attributes import (
    ArchivedServiceAttributes,
    ServiceAttributes,
)
from mpfs.core.metastorage.control import billing_services, billing_services_history
from mpfs.common.util import ctimestamp

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


class CommonService(UniqEntity):

    table = billing_services
    nf_error = errors.BillingServiceNotFound
    uniq_key_name = SID
    mutable_attrs = [BTIME]

    def __init__(self, sid, **kwargs):
        super(CommonService, self).__init__(sid, **kwargs)
        self.product = Product(self.pid)

        if self.product.altattrs:
            self.original_product = Product(self.pid)
            self.load_alternative_attributes()

    def load_alternative_attributes(self):
        '''
        Загрузка альтернативных атрибутов услуги по продутку
        '''
        self.altattrs = ServiceAttributes(self.sid, self.uid)
        self._set_fetched_alt_attrs()

    def set_attribute(self, key, value, save_to_db=True):
        '''
        Установка атрибута, как услуги, так и продукта
        '''
        area, attr = key.split('.')

        if not isinstance(value, int) and value.isdigit():
            value = int(value)

        if area == SERVICE and attr in self.mutable_attrs:
            self.set(attr, value, save_to_db=save_to_db)
            self._postprocess_attr(attr, new=value, old=getattr(self, attr))
        elif area == PRODUCT:
            self._set_product_attribute(attr, value)
        else:
            raise errors.BillingServiceAttributeViolated(key)

    def remove_attribute(self, key):
        '''
        Удаление атрибута, как услуги, так и продукта
        '''
        area, attr = key.split('.')

        if area != PRODUCT:
            raise errors.BillingServiceAttributeViolated(key)

        self._remove_product_attribute(attr)

    def _set_product_attribute(self, key, value):
        '''
        Установить альтернативный атрибут продукта
        '''
        if not self.product.altattrs:
            raise errors.BillingProductAltAttrsNotAllowed(self.product.pid)

        try:
            old_value = getattr(self.product.attributes, key)
        except AttributeError, e:
            raise errors.BillingProductHasNoAttribute(key)

        if isinstance(old_value, int):
            value = int(value)

        self.altattrs.set(key, value)
        setattr(self.product.attributes, key, value)

        self._postprocess_altattr(key, new=value, old=old_value)

    def _remove_product_attribute(self, key):
        '''
        Удалить альтернативный атрибут продукта
        '''
        if not self.product.altattrs:
            raise errors.BillingProductAltAttrsNotAllowed(self.product.pid)

        old_value = getattr(self.product.attributes, key, None)
        if not old_value:
            raise errors.BillingProductHasNoAttribute(key)

        orig_value = getattr(self.original_product.attributes, key)

        self.altattrs.remove(key)
        setattr(self.product.attributes, key, orig_value)

        self._postprocess_altattr(key, new=orig_value, old=old_value)

    def _set_fetched_alt_attrs(self):
        for key, value in self.altattrs.iteritems():
            setattr(self.product.attributes, key, value)


    @classmethod
    def Create(classname, client, product, auto=False):
        current_time = ctimestamp()

        sid = classname.GenerateID(client.uid, current_time, product.pid)
        data = {
            DB_ID: sid,
            UID: client.uid,
            PID: product.pid,
            CTIME: current_time,
            MTIME: current_time,
            BTIME: None,
            LASTBTIME: None,
            ENABLED: False,
            STATE: None,
            AUTO: auto,

            GROUP: False,  # групповая услуга
            PARENT_SID: None,  # sid услуги пользователя, купившего место для группы
            CHILD_SIDS: None,  # список sid'ов услуг пользователей, которым купили место
            GROUP_NAME: None,  # имя группы
        }

        try:
            classname.table.insert(data)
        except errors.BillingEntityAlreadyExists, e:
            error_log.warn('failed to insert, clean %s where %s=%s' % (classname.table, DB_ID, sid))
            classname.table.remove(**{DB_ID: sid})
            error_log.warn('cleaned %s where %s=%s' % (classname.table, DB_ID, sid))
            raise

        log.info('created service %s %s:%s (auto=%s)' % (product.pid, client.uid, sid, auto))
        return classname(sid, data=data)

    def delete(self, disable=True, send_email=True):
        self.touch()

        if disable:
            self.disable(send_email=send_email)

        if self.product.altattrs:
            self.altattrs.delete()

        history_record = copy.deepcopy(self.dict())
        history_record[CommonService.uniq_key_name] = self.sid
        del history_record[CommonService.pkey]
        billing_services_history.insert(history_record)

        super(CommonService, self).delete()
        log.info('deleted service %s %s:%s' % (self.product.pid, self.uid, self.sid))

    def enable(self, save_to_db=True, **kwargs):
        self.process_enable(**kwargs)
        self.set(ENABLED, True, save_to_db)

    def disable(self, save_to_db=True, send_email=True):
        self.process_disable(send_email=send_email)
        self.set(ENABLED, False, save_to_db)

    def set_regular_state(self, save_to_db=True):
        self.set(STATE, None, save_to_db)

    def process_enable(self, **kwargs):
        pass

    def process_disable(self, **kwargs):
        pass

    def _postprocess_altattr(self, attr, new=None, old=None):
        pass

    def _postprocess_attr(self, attr, new=None, old=None):
        pass


class ArchiveService(CommonService):

    table = billing_services_history
    pkey = SID

    def load_alternative_attributes(self):
        self.altattrs = ArchivedServiceAttributes(self.sid, self.uid)
        self._set_fetched_alt_attrs()

    def archive(self):
        pass

    def disable(self):
        pass

    def restore(self, enable=True):
        self.touch()

        if enable:
            self.enable(save_to_db=False)

        if self.product.altattrs:
            self.altattrs.restore()

        restored_record = self.dict()
        restored_record[CommonService.pkey] = self.sid
        restored_record.pop(CommonService.uniq_key_name)
        billing_services.insert(restored_record)

        UniqEntity.delete(self)
        log.info('restored service %s %s:%s' % (self.product.pid, self.uid, self.sid))
