# -*- coding: utf-8 -*-

from collections import defaultdict

import mpfs.engine.process

from mpfs.core import factory
from mpfs.core.address import Address, ResourceId
from mpfs.core.factory import get_resources_by_resource_ids
from mpfs.core.filesystem.resources.photounlim import PhotounlimFile
from mpfs.core.operations import manager
from mpfs.core.social.publicator import Publicator
from mpfs.core.user.constants import PUBLIC_UID
from mpfs.common import errors
from mpfs.common.util import from_json, chunks2
from mpfs.config import settings
from mpfs.core.filesystem.resources.attach import AttachFile
from mpfs.core.filesystem.resources.share import SharedFile
from mpfs.core.filesystem.resources.disk import DiskFile
from mpfs.core.user.constants import PHOTOUNLIM_AREA_PATH


SYSTEM_BULK_DOWNLOAD_LIST_CHUNK_SIZE = settings.system['bulk_download_list_chunk_size']

log = mpfs.engine.process.get_default_log()
error_log = mpfs.engine.process.get_error_log()


def _construct_items_to_archive_for_bulk_download_prepare(resources):
    """На основе переданного списка ресурсов сформировать структуру,
    которая будет записана в операцию загрузки."""
    items_to_archive = defaultdict(list)
    for resource in resources:
        if isinstance(resource, SharedFile):
            owner_uid = resource.link.group.owner
        elif isinstance(resource, (DiskFile, AttachFile, PhotounlimFile)):
            owner_uid = resource.uid
        else:
            # skip other resources (folders and etc)
            continue

        items_to_archive[owner_uid].append(resource.owner_file_id)

    return items_to_archive


def _create_operation_and_response_for_bulk_download_prepare(uid, items_to_archive):
    """Создать операцию и вернуть ответ, который будет возвращён клиенту."""
    if not len(items_to_archive):
        raise errors.BulkDownloadNoFilesToDownload()

    operation = manager.create_operation(
        uid,
        'download',
        'bulk-download-prepare',
        odata=dict(
            items=items_to_archive,
        ),
    )
    return {
        'download_url': operation['download_url'],
        'oid': operation.id,
        'type': operation.type,
        'at_version': operation.at_version
    }


def bulk_download_prepare(req):
    """
    Подготовка операции для скачивания группы файлов в архиве

    Query string аргументы:
      * uid [обязательный]

    Тело запроса:
    {
        "items": ["path1", "path2", "path3"]
    }
    """
    if not req.http_req.data:
        raise errors.BulkDownloadJsonBodyExpectedError()

    raw_data = from_json(req.http_req.data)
    raw_items = raw_data.get('items', [])

    addresses = map(lambda item: Address.Make(req.uid, item), raw_items)
    resources = factory.get_resources(req.uid, addresses,
                                      available_service_ids=['/disk', '/attach', PHOTOUNLIM_AREA_PATH])

    items_to_archive = _construct_items_to_archive_for_bulk_download_prepare(resources)
    return _create_operation_and_response_for_bulk_download_prepare(req.uid, items_to_archive)


def public_bulk_download_prepare(req):
    """Подготовить операцию для скачивания группы публичных файлов/папок в архиве.

    Query string аргументы:
      * uid [обязательный]

    Тело запроса:
    {
        "items": ["public_hash_1:/relative_1", "public_hash_2", "public_hash_1:/relative_2", ...]
    }

    Известные проблемы:
      * не поддерживаем работу с папками
      * нет проверки на заблокированность ресурсов
    """
    from mpfs.core.address import PublicAddress
    from mpfs.core.social.publicator import crypt_agent
    from mpfs.core.yateam.logic import check_yateam_subtree_access
    from mpfs.core.filesystem.resources.disk import append_meta_to_office_files
    from mpfs.core.user.base import User

    if not req.http_req.data:
        raise errors.BulkDownloadJsonBodyExpectedError()

    uid = req.uid
    pb = Publicator(request=req)

    raw_data = from_json(req.http_req.data)
    raw_items = raw_data.get('items', [])
    public_addresses = [PublicAddress(public_hash) for public_hash in raw_items]

    public_hash_to_paths = defaultdict(set)
    for public_address in public_addresses:
        public_hash_to_paths[public_address.hash].add(public_address.relative)

    resources = []
    for public_hash, relative_addresses in public_hash_to_paths.iteritems():
        symlink_hash = crypt_agent.decrypt(public_hash)

        try:
            resource = pb.fs.get_symlink_resource(PUBLIC_UID, symlink_hash)
            check_yateam_subtree_access(uid, resource.path, resource.uid)
            append_meta_to_office_files(resource, req)
        except errors.ResourceNotFound:
            # просто пропускаем этот ресурс
            log.info('Appropriate resource for public_hash "%s" not found.' % public_hash)
            continue

        try:
            resource.assert_blocked()
        except errors.ResourceBlocked:
            # ресурс заблокирован => заблокированы и все папки/файлы внутри него
            log.info('Resource "%s" blocked.' % resource.id)
            continue

        user = User(resource.uid)
        try:
            user.check_blocked()
        except errors.UserBlocked:
            # юзер заблокирован => заблокированы и все его публичные папки/файлы
            log.info('User "%s" blocked.' % user.uid)
            continue

        try:
            user.public_info(raise_if_does_not_exist=True)
        except errors.PassportUserDoesNotExistError:
            log.info('User "%s" does not exist.' % user.uid)
            continue

        if not resource.is_fully_public():
            log.info('Resource "%s" is not fully public.' % resource.id)
            continue

        full_addresses = []
        for relative_address in relative_addresses:
            full_addresses.append(
                Address.Make(resource.uid, resource.path + relative_address.path)
            )

        resources.extend(
            factory.get_resources(
                resource.uid, full_addresses,
                available_service_ids=('/disk', '/attach', PHOTOUNLIM_AREA_PATH)
            )
        )

    items_to_archive = _construct_items_to_archive_for_bulk_download_prepare(resources)
    return _create_operation_and_response_for_bulk_download_prepare(req.uid, items_to_archive)


def bulk_download_list(req):
    """
    Получение списка файлов из операции, созданной ручкой bulk_download_prepare

    Query string аргументы:
      * uid [обязательный]
      * oid [обязательный] - id операции, полученной из bulk_download_prepare
    """
    operation = manager.get_operation(req.uid, req.oid)
    items_to_archive = operation['items']

    resources = []
    for uid, file_ids in items_to_archive.iteritems():
        resource_ids = (ResourceId(uid, file_id) for file_id in file_ids)
        chunks = chunks2(resource_ids, SYSTEM_BULK_DOWNLOAD_LIST_CHUNK_SIZE)
        for chunk in chunks:
            resources_to_append = [
                x for x in get_resources_by_resource_ids(uid, chunk, enable_service_ids=['/disk', '/attach', PHOTOUNLIM_AREA_PATH])
                if x is not None
            ]
            resources.extend(resources_to_append)
    return {
        'operation': operation,
        'resources': resources,
    }
