# -*- coding: utf-8 -*-

"""
Модуль содержит интерфейс для кеширования с помощью uWSGI Caching Framework.
Для поддержки общего кеша в uwsgi, необходимо менять его конфигурацию.
(см. http://uwsgi-docs.readthedocs.org/en/latest/Caching.html)

Если модуль uwsgi не доступен, например, в тестах, то происходит фолбек на внутренний кеш.
"""

__all__ = ['cache_set', 'cache_get', 'cache_exists', 'cache_keys']

try:
    # автоматически добавляется мастер процессом
    import uwsgi
    UWSGI = True
except ImportError:
    UWSGI = False

_caches = {}


def cache_set(key, value, expire, cache):
    """ Добавить значение `value` элемента `key` в кеш `cache`.

    Если `expire` == 0, то значение хранится в кеше бессрочно.
    Иначе оно хранится `expire` секунд.

    ..warning::
        Во внутреннем кеше `expire` игнорируется.

    :type key: str | unicode
    :type value: str
    :type expire: int
    :type cache: str
    :rtype: bool | None
    """
    global _caches
    if not isinstance(value, (str, unicode)):
        raise TypeError('Cache item must be `str` or `unicode`.')

    if UWSGI:
        if uwsgi.cache_exists(key, cache):
            return uwsgi.cache_update(key, value, expire, cache)
        else:
            return uwsgi.cache_set(key, value, expire, cache)

    if cache not in _caches:
        _caches[cache] = {}
    _caches[cache][key] = value
    return True


def cache_get(key, cache):
    """ Извлечь значение элемента `key` из кеша `cache`.

    Вернется ``None``, если нет такого кеша или элемента в кеше.

    :type key: str
    :type cache: str
    :rtype: str | None
    """
    if UWSGI:
        return uwsgi.cache_get(key, cache)
    return _caches.get(cache, {}).get(key, None)


def cache_exists(key, cache):
    """ Проверить есть ли элемент с ключом `key` в кеше `cache`.

    Вернется ``True`` только если есть такой кеш и в нем элемент `key`,
    иначе ``None``.

    :type key: str
    :type cache: str
    :rtype: bool | None
    """
    if UWSGI:
        return uwsgi.cache_exists(key, cache)

    if cache not in _caches:
        return None

    if key in _caches[cache]:
        return True
    return None


def cache_keys(cache):
    """Вернуть все ключи в кеше `cache`.

    Если такой кеш отсутствует. то вернется ``None``.
    """

    if UWSGI:
        return uwsgi.cache_keys(cache)

    if cache not in _caches:
        return None

    return _caches[cache].keys()
