# -*- coding: utf-8 -*-
from mpfs.common.errors import ResourceNotFound
from mpfs.common.errors.share import (
    ShareNotFound,
)
from mpfs.core import factory
from mpfs.core.address import ResourceId
from mpfs.core.comments import errors
from mpfs.core.comments.permissions import CommentPermissions
from mpfs.core.filesystem.resources.group import GroupRootFolder
from mpfs.core.filesystem.resources.share import SharedResource
from mpfs.core.comments.logic import common


def get_comment_permissions_for_uid(entity_type, entity_id, uid):
    """Получить разрешенные действия комментирования для конкретного ресурса и пользователя."""
    resource_id = ResourceId.parse(entity_id)
    if entity_type == 'public_resource':
        permissions = _get_permissions_for_public_resource(resource_id, uid)
    elif entity_type == 'private_resource':
        permissions = _get_permissions_for_private_resource(resource_id, uid)
    else:
        raise errors.BadEntityType()
    return permissions.as_dict()


def _get_permissions_for_public_resource(resource_id, uid):
    owner_uid = common.get_public_resource_owner(resource_id)
    if uid == owner_uid:
        return CommentPermissions.create_all_permissions()
    else:
        return CommentPermissions.create_read_only_permissions()


def _get_permissions_for_private_resource(resource_id, uid):
    resource = factory.get_resources_by_resource_ids(uid, [resource_id])[0]
    if resource is not None:
        if isinstance(resource, SharedResource) and resource.link.group.owner != uid:
            return CommentPermissions.create_read_only_permissions()
        return CommentPermissions.create_owner_permissions()

    return _get_permissions_for_private_resource_fallback(resource_id, uid)


def _get_permissions_for_private_resource_fallback(resource_id, uid):
    resource = factory.get_resources_by_resource_ids(resource_id.uid, [resource_id])[0]
    if resource is None or not isinstance(resource, GroupRootFolder):
        raise ResourceNotFound()

    try:
        group_link = resource.group.get_link_by_uid(uid)
        if group_link.group.owner == uid:
            return CommentPermissions.create_owner_permissions()
        else:
            return CommentPermissions.create_read_only_permissions()
    except ShareNotFound:
        raise ResourceNotFound()

