# -*- coding: utf-8 -*-
from copy import deepcopy
import pymongo
from mpfs.core.models import ObjectIdField
from mpfs.common.errors import ZeroUpdateControllerMPFSError


class BaseController(object):
    """Базовый класс контроллера."""
    pass


class BaseModelControllerMetaClass(type):
    """Связывает модель с контроллером."""
    def __new__(mcs, name, parents, attrs):
        model = attrs.get('model', None)
        cls = super(BaseModelControllerMetaClass, mcs).__new__(mcs, name, parents, attrs)
        if model is not None:
            model.controller = cls()
        return cls


class BaseModelController(BaseController):
    """
    Базовый класс для контроллеров обеспечивающих управление конкретной моделью.
    Связывает модель с коллекцией.

    Класс инкапсилирующий всю рутинную логику работы с коллекцией моделей:
      * преобразование сырых данных полученных из БД в объекты.
      * преобразование объектов в сырые данны пригодные для сохранения в БД.
      * преобразование запросов по полям объектов в запросы по полям записев в БД.

    Пример использования:
    >>> from mpfs.metastorage.mongo.collections.base import BaseCollection
    >>> from mpfs.core.models import Model, ModelField, ObjectIdField
    >>>
    >>> class PetsCollection(BaseCollection):
    ...     name = 'pets'
    ...
    >>> class Pet(Model):
    ...     primary_key_field = 'id'
    ...     id = ObjectIdField(source='_id')
    ...     name = ModelField(required=True)
    ...     age = ModelField()
    ...
    >>> class PetsController(BaseModelController):
    ...     model = Pet
    ...     collection = PetsCollection()
    ...
    >>> # создаём живтное
    >>> p = Pet(name=u'Барсик')
    >>> p.save()
    >>> # создаём животное вторым способом
    >>> p = PetsController().create(name=u'Шашлык')
    >>> # создаём много животных
    >>> pets = PetsController().bulk_create([Pet(name=u'Шарик1'), Pet(name=u'Шарик2')])
    >>> for p in pets:
    ...     print p.id, p.name
    ...
    >>> # получаем барсика из БД
    >>> p = PetsController().get(name=u'Барсик')
    >>> print p.id, p.name
    >>> # проставляем возраст барсику и обновляем барсика в БД
    >>> p.age = 5
    >>> p.save()
    >>> # пройдёмся по всем животным и расставми им возраст
    >>> for p in PetsController().all():
    ...     p.age = 10
    ...     p.save()
    ...
    >>> # обновим шарикам возраст одним запросом
    >>> PetsController().filter(name={'$in': [u'Шарик1', u'Шарик2']}).update(age=3)
    >>> # поудаляем всех шашлыков
    >>> PetsController.filter(name=u'Шашлык').delete()
    >>>
    >>> # при этом контроллер доступен из модели
    >>> Pet.controller.get(name=u'Шарик1')
    """
    __metaclass__ = BaseModelControllerMetaClass
    model = None
    collection = None

    REPR_OUTPUT_SIZE = 3

    def __init__(self, spec=None, limit=0, offset=0, order_by=None):
        super(BaseModelController, self).__init__()
        self.spec = spec
        self.limit = limit
        self.offset = offset
        self.order = order_by
        self._result_cache = None

    def order_by(self, *args):
        """
        Добавляет к запросу параметры сортировки.

        :param args: Список имён атрибутов объекта по которым следует сортировать выборку.
                     Для сортировки по убыванию к имени атрибута следует добавить "-".
        """
        return type(self)(spec=self.spec, limit=self.limit, offset=self.offset, order_by=args)

    def count(self):
        """
        Возвращает количество элементов попадающих в выборку.

        :rtype: int
        """
        return self.collection.get_count(**self._as_pymongo_kwargs()['spec'])

    def create(self, **kwargs):
        """
        Создаёт объект и сохраняет его в БД.

        :param kwargs: Значения атрибутов создаваемого объекта.
        """
        obj = self.model(**kwargs)
        obj.save(force_insert=True)
        return obj

    def filter(self, **kwargs):
        """
        Создаёт запрос на выборку. При этом сам по себе не делает запрос в БД.
        Запрос будет выполнен только при непосредственном обращении к элементам выборки.

        Называется не find, т.к. сам по себе ничего не ищет, а лишь задаёт фильтр по которому будут выбраны данные.

        :param kwargs: Критерии выборки.
        """
        new_spec = {}
        if self.spec is not None:
            new_spec.update(self.spec)
        new_spec.update(kwargs)
        return type(self)(spec=new_spec, limit=self.limit, offset=self.offset, order_by=self.order)

    def get(self, **kwargs):
        """
        Возвращает объект, соответствующий критериям или None.

        :param kwargs: Критерии выборки.
        """
        spec = {}
        if self.spec:
            spec.update(self.spec)
        spec.update(kwargs)

        db_object = self._fetch_one(self._format_spec(spec))

        if db_object:
            return self.model.from_db(db_object)
        return None

    def setup_result_cache(self, objects):
        """
        Устанавливаем ранее полученные объекты контроллеру, эмулируя поход в базу
        """
        self._result_cache = []
        for obj in objects:
            if not isinstance(obj, self.model):
                raise TypeError("All objects should be %r" % self.model.__class__)
            self._result_cache.append(obj)

    def reset_result_cache(self):
        self._result_cache = None

    def update(self, upsert=False, multi=False, **kwargs):
        """
        Обновляет атрибуты объектов попавших в выборку.

        :param kwargs: Значения атрибутов.
        :param upsert: update c upsert-ом
        """
        data = self.model.map_to_db_fields(kwargs)
        mongo_result = self.collection.update(self._as_pymongo_kwargs()['spec'], {'$set': data}, upsert=upsert, multi=multi)
        if mongo_result['n'] == 0:
            raise ZeroUpdateControllerMPFSError()
        self._result_cache = None

    def delete(self):
        """Удаляет объекты попавшие в выборку."""
        self.collection.remove(**self._format_spec(self.spec))
        self._result_cache = None

    def bulk_create(self, objs, continue_on_error=False):
        """
        Создаёт объекты в базе данных в один запрос.

        :param objs: Список объектов для создания.
        :param continue_on_error: http://api.mongodb.org/python/2.7rc0/api/pymongo/collection.html?highlight=insert#pymongo.collection.Collection.insert
        :return: Список ключей созданных объектов.
        :return: Возвращает список объектов с установленным первичнм ключём.
        :rtype: list
        """
        if not objs:
            return []
        db_objs = [o.as_db_object() for o in objs]
        ids = self._insert(db_objs, continue_on_error=continue_on_error)
        for i, o in enumerate(objs):
            setattr(o, o.primary_key_field, ids[i])
        return objs

    def all(self):
        return self

    def iter_all(self):
        for db_object in self.collection.find(**self._as_pymongo_kwargs()):
            yield self.model.from_db(db_object=db_object)

    def _insert(self, db_objects, continue_on_error=False):
        raw_ids = self.collection.insert(db_objects, continue_on_error=continue_on_error)
        field = ObjectIdField()
        if isinstance(raw_ids, list):
            return [field.to_native(_id) for _id in raw_ids]
        else:
            return field.to_native(raw_ids)

    def __iter__(self):
        if self._result_cache is None:
            self._fetch_all()
        for db_object in self._result_cache:
            yield db_object

    def __getitem__(self, item):
        if self._result_cache is not None:
            return self._result_cache[item]

        if isinstance(item, slice):
            if item.start is not None:
                offset = int(item.start)
            else:
                offset = 0
            if item.stop is not None:
                limit = int(item.stop) - offset
            else:
                limit = 0
            query = type(self)(spec=self.spec, limit=limit, offset=offset,
                               order_by=self.order)
            if item.step:
                return list(query)[::item.step]
            else:
                return query
        else:
            query = type(self)(spec=self.spec, limit=item, offset=item+1,
                               order_by=self.order)
            return list(query)[0]

    def __len__(self):
        if self._result_cache is not None:
            return len(self._result_cache)
        return self.count()

    def __repr__(self):
        data = list(self[:self.REPR_OUTPUT_SIZE + 1])
        if len(data) > self.REPR_OUTPUT_SIZE:
            data[-1] = '...(remaining elements truncated)...'
        return "%r. %r" % (self.__class__, data)

    def __bool__(self):
        return bool(self.count())

    def _as_pymongo_kwargs(self):
        resp = {'spec': self._format_spec(self.spec)}
        if self.limit:
            resp['limit'] = self.limit
        if self.offset:
            resp['skip'] = self.offset
        if self.order:
            sort = []
            for k in self.order:
                key = k.lstrip('-')
                if key in self.model.fields:
                    key = self.model.fields.get(key).source
                direction = pymongo.ASCENDING if not k.startswith('-') else pymongo.DESCENDING
                sort.append((key, direction))
            resp['sort'] = sort
        return resp

    def _format_spec(self, spec, parent=None):
        """
        Преобразует имена полей модели в запросах в имена полей в БД.
        """
        ret = {}
        if spec:
            spec = deepcopy(spec)
            for k, v in spec.iteritems():
                field = None
                if k in self.model.fields:
                    field = self.model.fields.get(k)

                if isinstance(v, dict):
                    v = self._format_spec(v, parent=k)
                elif isinstance(v, list):
                    for i in xrange(len(v)):
                        if isinstance(v[i], dict):
                            v[i] = self._format_spec(v[i])
                        elif parent and parent in self.model.fields:
                            v[i] = self.model.fields.get(parent).from_native(v[i])
                else:
                    # если в атрибуте лежит простое значение
                    if field:
                        v = field.from_native(v)

                if field:
                    k = field.source
                ret[k] = v
        return ret

    def _fetch_all(self):
        self._result_cache = [self.model.from_db(db_object=i) for i in self.collection.find(**self._as_pymongo_kwargs())]
        return self._result_cache

    def _fetch_one(self, spec):
        return self.collection.find_one(spec)
