# -*- coding: utf-8 -*-
"""
Объекты InDBRPSLimiter заводим в этом модуле в глобальной переменно в суффиксом *_RL (см. пример внизу)
"""
import time
import datetime

from mpfs.common import errors
from mpfs.config import settings
from mpfs.core.metastorage.control import disk_info
from mpfs.metastorage.mongo.util import generate_version_number


class InDBRPSLimiter(object):
    """Лимитирование RPS в базе"""

    def __init__(self, counter_name):
        """Создание нового limiter'а

        :param string counter_name: название limiter'а
        """
        if counter_name not in settings.db_rate_limiter:
            raise ValueError("Cannot find settings for %s counter" % counter_name)

        self._counter_name = counter_name
        counter = settings.db_rate_limiter[counter_name]
        self._burst = counter['burst']
        self._rph = counter['rph']

    def check_limit_and_increment_counter(self, uid, requests_num=1):
        """Исчерпан лимит или нет. Если нет, то увеличиваем счетчик

        True - лимит не исчерпан
        False - лимит исчерпан. Надо лимитировать

        :param string uid:
        :param int requests_num:
        :rtype: bool
        """
        # Если счетчик в базе будет обновлен другим процессом, пробуем обновлять еще раз (и так до трех раз)
        for i in range(3):
            key = '/db_ratelimiter/%s' % self._counter_name
            now = datetime.datetime.now()
            counter = disk_info.find_one_by_field(uid, {'key': key})
            if not counter:
                counter = self._create_counter(uid, now)
            old_version = counter['version']

            delta = now - datetime.datetime.fromtimestamp(counter['data']['last_ts'])
            restored_burst = self._rph * delta.total_seconds() / 3600.0
            new_burst = max(0, counter['data']['current_burst'] - restored_burst) + requests_num

            if new_burst > self._burst:
                return False
            now_ts = time.mktime(now.timetuple()) + now.microsecond / 1000000.0
            result = self._update_counter(uid, now_ts, new_burst, old_version)
            if result['updatedExisting']:
                return True
        return False

    def _create_counter(self, uid, now=None):
        """Инициализация счетчика в базе

        :param string uid:
        :param now:
        :return:
        """
        key = '/db_ratelimiter/%s' % self._counter_name
        if not now:
            now = datetime.datetime.now()
        last_ts = time.mktime(now.timetuple()) + now.microsecond / 1000000.0
        try:
            disk_info.make_folder(uid, '/db_ratelimiter', '')
        except errors.StorageFolderAlreadyExist:
            pass
        disk_info.put(uid, key, {'last_ts': last_ts, 'current_burst': 0})
        return disk_info.find_one_by_field(uid, {'key': key})

    def _update_counter(self, uid, last_ts, current_burst, old_version=None):
        """Обновление значения счетчика в базе

        :param string uid:
        :param int last_ts:
        :param int current_burst:
        :param int old_version:
        :return:
        """
        key = '/db_ratelimiter/%s' % self._counter_name
        new_value = {'last_ts': last_ts, 'current_burst': current_burst}
        data, query = disk_info.data_for_mkfile(uid, key, new_value, old_version, generate_version_number())
        query_update = {'$set': data}
        result = disk_info.update(query, query_update)
        return result


HANDLE_SHARE_INVITE_USER_RL = InDBRPSLimiter('handle_share_invite_user')
