import re

from mpfs.engine.process import get_error_log

from mpfs.common.errors import InternalServerError
from mpfs.config import settings
from mpfs.core.user.anonymous import AnonymousUser, AnonymousUID
from mpfs.core.user.base import User
from mpfs.core.user.common import CommonUser
from mpfs.metastorage.postgres.query_executer import PGQueryExecuter

error_log = get_error_log()


def wrap_degradation(fn):
    def wrap(*args, **kwargs):
        try:
            if not settings.degradations.get('enabled', False):
                return True

            return fn(*args, **kwargs)
        except InternalServerError:
            raise
        except Exception as e:
            error_log.exception(e)

    return wrap


@wrap_degradation
def is_allow_only_office(user_or_uid):
    if not isinstance(user_or_uid, CommonUser):
        user_or_uid = User(user_or_uid)

    if isinstance(user_or_uid, AnonymousUser):
        return settings.degradations['only_office']['allow_anonymous']

    if user_or_uid.is_free():
        return settings.degradations['only_office']['allow_free']
    else:
        return settings.degradations['only_office']['allow_paid']


@wrap_degradation
def check_degradation_by_path(req, params):
    uid = params.get('uid')
    if not uid or AnonymousUID.is_anonymous_uid(uid):
        return

    path = req.http_req.environ['REQUEST_URI']

    for path_regex, shards in settings.degradations.get('by_path', {}).iteritems():
        if not shards:
            continue

        if re.search(path_regex, path) is not None:
            user_shard = PGQueryExecuter().get_shard_id(uid)

            if str(user_shard) in [str(x) for x in shards]:
                raise InternalServerError()


@wrap_degradation
def check_degradation_by_client(req, params):
    uid = params.get('uid')
    if not uid or AnonymousUID.is_anonymous_uid(uid):
        return True

    if not hasattr(req, 'tvm_2_0_service_client'):
        return True

    shards = settings.degradations.get('by_client', {}).get(req.tvm_2_0_service_client.name)
    if not shards:
        return True

    user_shard = PGQueryExecuter().get_shard_id(uid)

    if str(user_shard) in [str(x) for x in shards]:
        raise InternalServerError()
