# -*- coding: utf-8 -*-

from mpfs.common import errors
from mpfs.common.util import from_json
from mpfs.core.deletion_from_local_device.logic import SafeDeleteChecker
from mpfs.core.filesystem.hardlinks.common import construct_hid


def can_delete(req):
    """По хэшам и размеру понять, можно ли удалить файл с локального устройства или нет. Проверка подразумевает
    следующее:
      1. Файл c хидом, построенным по хэшам, должен находиться в /disk или /photounlim
      2. Если файл в фотоанлиме, то он должен быть проиндексирован

    Query string аргументы:
        * uid [обязательный] - uid пользователя, чьи файлы хотим проверить.

    Body аргументы:
        {
            "items": [
                {
                    "sha256": <sha256>,
                    "md5": <md5>,
                    "size": <size>
                },
                ...
            ]
        }

    Формат ответа (возвращается в том же порядке, что и запросы)
        {
            "items": [
                {
                    # когда файл можно удалить
                    "resource_id": <resource_id>,
                    "can_delete": True
                },
                {
                    # когда файл нельзя удалить
                    "can_delete": False
                },
            ]
        }

    """
    if not req.http_req.data:
        raise errors.JsonBodyExpectedError()
    raw_data = from_json(req.http_req.data)
    hids = [construct_hid(x['md5'], int(x['size']), x['sha256']) for x in raw_data.get('items', [])]
    results_ordered = SafeDeleteChecker().is_able_to_delete_files_from_local_by_uid_and_hids(req.uid, hids)
    return {'items': results_ordered}
