# -*- coding: utf-8 -*-
from mpfs.core.factory import get_resource_from_doc
from mpfs.core.filesystem.dao.resource import (
    PhotounlimDAO,
    ResourceDAO,
)
from mpfs.core.filesystem.resources.group import GroupFile
from mpfs.core.filesystem.resources.share import SharedFile
from mpfs.core.services.search_service import SearchDB
from mpfs.engine.process import get_default_log

default_log = get_default_log()
DB_CHUNK_SIZE = 100


class SafeDeleteChecker(object):

    def __init__(self):
        self.search_db_service = SearchDB()

    def is_able_to_delete_files_from_local_by_uid_and_hids(self, uid, ordered_hids):
        hids_to_find = set(ordered_hids)
        found_hids_to_resource_id = {}
        for dao in [ResourceDAO(), PhotounlimDAO()]:
            cur_offset = 0
            while hids_to_find:
                # преобразовываем в список, посокльку дальше понадобится и итерация и получение длины
                file_dao_items = list(dao.find_by_uid_and_hids(uid, list(hids_to_find), limit=DB_CHUNK_SIZE,
                                                               offset=cur_offset))
                found_resources = [get_resource_from_doc(uid, i.get_mongo_representation()) for i in file_dao_items]

                found_resources = self.filter_files_from_shared_directories(found_resources)
                if isinstance(dao, PhotounlimDAO):
                    found_resources = self.filter_only_indexed_resources(uid, found_resources)

                for i in found_resources:
                    default_log.info('Can delete file with hid %s of uid %s from local device' % (i.hid, uid))

                    if str(i.hid) in found_hids_to_resource_id:
                        continue
                    found_hids_to_resource_id[str(i.hid)] = i.resource_id.serialize()

                if not len(found_resources):
                    # за итерацию не нашли ни одного нового файла, который позволил бы удалить локально
                    number_of_found_items = len(file_dao_items)
                    if number_of_found_items < DB_CHUNK_SIZE:
                        # кончилась выборка
                        break
                    cur_offset += number_of_found_items
                    continue

                hids_to_find -= found_hids_to_resource_id.viewkeys()
                # выкинули несколько хидов, сбрасываем оффсет и ищем по новой по остаткам
                cur_offset = 0

        result = []
        for i in ordered_hids:
            if i in found_hids_to_resource_id:
                result.append({'can_delete': True, 'resource_id': found_hids_to_resource_id[i]})
            else:
                result.append({'can_delete': False})
        return result

    @staticmethod
    def filter_files_from_shared_directories(resources_to_filter):
        return [x for x in resources_to_filter if not isinstance(x, (SharedFile, GroupFile))]

    def filter_only_indexed_resources(self, uid, resources_to_filter):
        """Фильтрует ресурсы, оставляя, только попавшие в фотосрез"""
        if not resources_to_filter:
            return []

        file_id_to_resource = {x.meta['file_id']: x for x in resources_to_filter}
        result = self.search_db_service.get_indexed_files(uid, file_id_to_resource.keys())
        indexed_resources_file_ids = [i['file_id'] for i in result['items']]

        indexed_resources = []
        for file_id in indexed_resources_file_ids:
            indexed_resources.append(file_id_to_resource[file_id])
        return indexed_resources
