# -*- coding: utf-8 -*-
from mpfs.config import settings
from mpfs.core.email import errors
from mpfs.core.services.passport_service import passport
from mpfs.core.user.constants import DEFAULT_LOCALE
from mpfs.core.queue import mpfs_queue


def get_campaign_id(campaign_name, locale):
    templates = settings.email_sender_campaigns[campaign_name]['templates']
    if locale not in templates:
        raise errors.UnsupportedEmailLocaleError
    return templates[locale] if locale in templates else templates[DEFAULT_LOCALE]


def is_campaign_enabled(campaign_name, uid=None):
    return bool(
        settings.email_sender_campaigns[campaign_name]['enabled']
        or (uid and uid in settings.email_sender_campaigns[campaign_name].get('test_uids', []))
    )


def _is_gdpr_safe(campaign_name, user_info):
    # FIXME: временное решение для https://wiki.yandex-team.ru/sender/policy/restriction
    return settings.email_sender_campaigns[campaign_name].get('gdpr_safe', False) or user_info.get('country') == 'ru'


def _extract_send_to_emails(bb_address_list):
    # отправляем на: на дефолтный, на нативный (если он не дефолтный) и на внешние
    # https://docs.yandex-team.ru/blackbox/concepts/emails#external

    emails = []
    defaults = [i for i in bb_address_list if i.get('default') is True]
    natives = [i for i in bb_address_list if i.get('native') is True]
    if len(defaults) > 0:
        default = defaults[0]
        emails.append(default['address'])
        if default.get('native') is False and len(natives) > 0:
            emails.append(natives[0]['address'])
    elif len(natives) > 0:
        emails.append(natives[0]['address'])

    for item in bb_address_list:
        if (
            not item
            or not item.get('address')
            or item.get('native') is True
            or item.get('default') is True
        ):
            continue
        emails.append(item['address'])
    return emails


def send_email_async_by_uid(uid, campaign_name, template_args=None, locale=None, allowed_locales=None, default_locale=None, send_on_all_emails=False):
    if send_on_all_emails:
        # может быть закеширован ответ userinfo без всех email
        passport.reset()
    user_info = passport.userinfo(uid, all_emails=send_on_all_emails)

    if not _is_gdpr_safe(campaign_name, user_info):
        return

    if not locale:
        lang = user_info.get('language')
        if lang:
            locale = lang.lower()
        else:
            locale = default_locale

    if allowed_locales and default_locale and locale not in allowed_locales:
        locale = default_locale

    if send_on_all_emails and user_info.get('address-list'):
        emails = _extract_send_to_emails(user_info.get('address-list'))
    else:
        emails = [user_info['email']]

    use_mail_account = settings.email_sender_campaigns[campaign_name].get('use_mail_account', False)
    for email in emails:
        _send_email_async(
            email,
            campaign_name,
            locale=locale or user_info['language'].lower(),
            template_args=template_args,
            use_mail_account=use_mail_account
        )


def _send_email_async(email, campaign_name, locale=None, template_args=None, use_mail_account=False):
    """
    Функция отправки письма через рассылятор

    Achtung: письмо отправляется без проверок и смс

    :param email: адрес получателя
    :param campaign_name: название рассылки из конфига
    :param locale: локаль отправляемого письма для выбора шаблона. Если не передано - будет выбран шаблон
    с локалью по-умолчанию
    :param template_args: аргументы для передачи в шаблон
    :return: None
    """
    if not email:
        return
    try:
        campaign_id = get_campaign_id(campaign_name, locale)
    except errors.UnsupportedEmailLocaleError:
        return
    mpfs_queue.put({
        'email_address': email,
        'campaign_id': campaign_id,
        'template_args': template_args,
        'use_mail_account': use_mail_account,
    }, 'email_sender_send')
